/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__DATAFILE_TYPE__
#define EME__DATAFILE_TYPE__
/*----------------------------------------------------------------------------
  Datafile Object Property

  - Types
  - Casts
  - Draw
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

struct DATAFILE;
class Entry;
struct DIALOG;

#include "translat.h"


struct DatafileObject {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;
  typedef int Base;
  typedef StaticString* Param;


  typedef void (*DrawingCallback) (
    const Property *tile, BITMAP *bmp,
    int x, int y, int w, int h, float scale, int l, int i, int j
  );


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Draw
     ---------------------------------*/
  static void Draw(
    const DATAFILE *data, int idx, const char *fname,
    BITMAP *bmp, int x, int y, int w, int h, float scale
  );


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(StaticString *datafile, int idx);
    Property(StaticString *datafile, const char *name);
    ~Property(void);
    Property *Clone(void) const;

    /* Accessors
       --------------------*/
    void Set(int idx);
    void Set(const char *name);
    void SetFname(StaticString *filename)
      {if(fname_) fname_->Release(); fname_=filename; fname_->Hold();}
    int Get(void) const
      {return index_;}
    const DATAFILE *GetData(void) const
      {return data_;}
    StaticString *GetFname(void) const
      {return fname_;}

    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);
    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    /* Be carefull when using this one, there are no tests! */
    Property(StaticString *datafile, const DATAFILE *data, int index);

    const DATAFILE *data_;
    StaticString *fname_;
    int index_;
  };


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(
      StaticString *name,
      StaticString *datafile,              /* Datafile (path and) name */
      int idx
    );
    Creator(
      StaticString *name,
      StaticString *datafile,              /* Datafile (path and) name */
      const char *objname
    );
    Creator *Clone(void) const;
    ~Creator(void);

    /* Drawing callback
       --------------------*/
    void SetDrawingCallback (DrawingCallback cb) const
      {drawing_cb_ = cb;}
    DrawingCallback GetDrawingCallback (void) const
      {return drawing_cb_;}

    /* Properties
       --------------------*/
    Property *Create(int idx) const
      { return new Property(GetFname(), idx); }
    Property *Create(const char *objname) const
      { return new Property(GetFname(), objname); }

    int IsSameType (const BaseProperty *p) const;

    virtual void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h, float s,
      int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;


    /* Entry handlers
       --------------------*/
    void UpdateEntry (Entry *e, const BaseProperty *p) const;
    void UpdateProperty (BaseProperty *p, const Entry *e) const;

    StaticString *GetFname(void) const
      {return fname_;}

  protected:
    Creator(StaticString *name, StaticString *datafile, int idx, Property *ref);
    Creator(
      StaticString *name, StaticString *datafile, const char *objname,
      Property *ref
    );
    Creator(const Creator *other);

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<DatafileObject>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<DatafileObject>(GetVariableReference()); }

    StaticString *fname_;
    mutable DrawingCallback drawing_cb_;
  }; /* class DatafileObject::Creator */


  /* Private
     ---------------------------------*/
private:
  static const Translation i18nDatafile_not_found;
  static int display_callback(
    BITMAP *dest, int index, int x, int y, int w, DIALOG *d
  );

}; /* struct DatafileObject */

#ifdef DEBUG
typedef DatafileObject::Creator DatafileObjectCreator;
typedef DatafileObject::Property DatafileObjectProperty;
#endif

#endif /* EME__DATAFILE_TYPE__ */

