/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__COMPOSITE_TYPE__
#define EME__COMPOSITE_TYPE__
/*----------------------------------------------------------------------------
  Composite Property

  - Types
  - Casts
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

class Entry;
struct DIALOG;


struct Composite {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;


  typedef void (*DrawingCallback)(
    const Property *tile, BITMAP *bmp,
    int x, int y, int w, int h, float scale, int l, int i, int j
  );


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(int max_props);
    ~Property(void);
    Property *Clone(void) const;


    /* Accessing the contained properties
       --------------------*/
    /* Add the property 'p' at 'index' */
    void Add(int index, BaseProperty *p);

    /* Returns the 'n'th property */
    BaseProperty *Get(int n) const
      { return props_[n]; }
    int Count(void) const
      { return num_props_; }
    int Max(void) const
      { return max_props_; }


    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);
    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    BaseProperty **props_;
    int num_props_;
    int max_props_;
  }; /* class Composite::Property */


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(StaticString *name, int max_props);
    Creator *Clone(void) const;
    ~Creator(void);

    /* Accessors
       --------------------*/
    int Add(int idx, BaseCreator *c);
    BaseCreator *Get(int idx) const
      {return creators_[idx];}
    int Count(void) const
      {return max_props_;}

    /* Drawing callback
       --------------------*/
    void SetDrawingCallback(DrawingCallback cb) const
      {drawing_cb_ = cb;}
    DrawingCallback GetDrawingCallback(void) const
      {return drawing_cb_;}


    /* Properties
       --------------------*/
    Property *Create(int max_props) const
      { return new Property(max_props); }

    int IsSameType(const BaseProperty *p) const;
    void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h, float s,
      int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;


    /* Entry handlers
       --------------------*/
    void UpdateEntry(Entry *e, const BaseProperty *p) const;
    void UpdateProperty(BaseProperty *p, const Entry *e) const;

  protected:
    Creator(const Creator *other);
    void Init(void);

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<Composite>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<Composite>(GetVariableReference()); }

    BaseCreator **creators_;
    DIALOG *popup_;
    Entry **entries_;
    int max_props_;
    mutable DrawingCallback drawing_cb_;
  }; /* class Composite::Creator */


  /* Private
     ---------------------------------*/
private:
  static int popup_proc(int msg, DIALOG *d, int c);

}; /* struct Composite */

#ifdef DEBUG
typedef Composite::Property CompositeProperty;
typedef Composite::Creator CompositeCreator;
#endif

#endif /* EME__COMPOSITE_TYPE__ */

