/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "brush.h"

#include "debug.h"
#include "utils.h"

#include "tstack.h"
#include "creator.h"
#include "selected.h"
#include "lutils.h"
#include "lviewed.h"
#include "stiles.h"


Brush::Brush(
  const TilesStack *map, const ViewedLayers *layers, const SelectedTiles *sel
)
{
  layers_ = LayerUtils::Copy(map, layers, sel);
  int mini = layers_->begini();
  int minj = layers_->beginj();
  const ViewedLayers l(layers_->depth());
  LayerUtils::Move(layers_, &l, -mini, -minj);
}


Brush::Brush(int num_layers): layers_(new TilesStack())
{
  for (int l=0; l<num_layers; ++l) {
    layers_->set(l, 0);
  }
}


Brush::Brush(): layers_(new TilesStack())
{
}


Brush::Brush(const Brush &other): layers_(other.layers_->clone())
{
}


Brush::~Brush(void)
{
  delete layers_;
}

void Brush::SetLayer(int l, SparseTiles *tiles)
{
  DBG_ASSERT(!layers_->get(l));
  layers_->set(l, tiles);
}


void Brush::ClearLayer(int l)
{
  Tiles *layer = layers_->get(l);
  layers_->clear(l);
  delete layer;
}


int Brush::GetNumLayers(void) const
{
  return layers_->depth();
}


const SparseTiles *Brush::GetLayer(int l) const
{
  return layers_->get(l) ? layers_->get<SparseTiles>(l) : 0;
}

int Brush::MinX(void) const
{
  return layers_->begini();
}


int Brush::MinY(void) const
{
  return layers_->beginj();
}


int Brush::MaxX(void) const
{
  return layers_->endi()-1;
}


int Brush::MaxY(void) const
{
  return layers_->endj()-1;
}


void Brush::MoveLayers(int di, int dj)
{
  for (int l=0; l<layers_->depth(); ++l) {
    SparseTiles *layer = DYNAMIC_CAST(SparseTiles*, layers_->get(l));
    layer->move(di, dj);
  }
  /* Tells the TilesStack that layer size and/or position has changed */
  layers_->update();
}

