/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "semetile.h"

#include "semedlg.h"

#include "ustring.h"
#include "translat.h"
#include "alclean.h"

#include <stdlib.h>
#include <allegro.h>

#include "semenew.h"
#include "sememap.h"


/*------------------------------------------------------------------------------
  Tile Dialog
  To enter the tile shape, size, map shape, etc.
------------------------------------------------------------------------------*/

/* Indices to the dialog array 'get_tile_dialog()' */
typedef enum {
  TILE_BOX,
  TILE_TITLE,

  /* The map shape */
  TILE_TEXT1,
  TILE_MRECT,     /* d->flags&D_SELECTED: MAP_SHAPE_RECT */
  TILE_MDIAMOND,  /* d->flags&D_SELECTED: MAP_SHAPE_DIAMOND */

  /* The tiles shape */
  TILE_TEXT2,
  TILE_TRECT,     /* d->flags&D_SELECTED: TILE_SHAPE_RECT */
  TILE_TDIAMOND,  /* d->flags&D_SELECTED: TILE_SHAPE_DIAMOND */

  /* Tile size */
  TILE_TEXT3,
  TILE_WIDTH,     /* uatoi(d->dp): tile width */
  TILE_TEXT4,
  TILE_HEIGHT,    /* uatoi(d->dp): tile height */

  /* The oddness */
  TILE_TEXT7,
  TILE_ODD_NONE,  /* d->flags&D_SELECTED: ODD_NONE */
  TILE_ODD_LEFT,  /* d->flags&D_SELECTED: ODD_LEFT */
  TILE_ODD_RIGHT, /* d->flags&D_SELECTED: ODD_RIGHT */

  /* The tile increment */
  TILE_TEXT5,
  TILE_DX,        /* uatoi(d->dp): tile dx */
  TILE_TEXT6,
  TILE_DY,        /* uatoi(d->dp): tile dy */

  TILE_OK,
  TILE_CANCEL,
  NUM_TILE_DIALOGS
} TILE_DIALOG;


/* Strings to enter the tile size and increment */
#define TSIZE_LEN 10
char twidth_str[(TSIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;
char theight_str[(TSIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;
char tdx_str[(TSIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;
char tdy_str[(TSIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;


static const Translation i18nTile("Tiles");
static const Translation i18nMap_Shape("Map shape:");
static const Translation i18nTile_Shape("Tile shape:");
static const Translation i18nRectangle("rectangle");
static const Translation i18nDiamond("diamond");
static const Translation i18nTile_Width("Tile width:");
static const Translation i18nTile_Height("height:");
static const Translation i18nOddness("Oddness:");
static const Translation i18nnone("none");
static const Translation i18nleft("left");
static const Translation i18nright("right");
static const Translation i18nTile_DX("Tile DX:");
static const Translation i18nTile_DY("DY:");
static const Translation i18nOK("OK");
static const Translation i18nCancel("Cancel");


static DIALOG *get_tile_dialog()
{
  static DIALOG dlg[NUM_TILE_DIALOGS+1];
  static bool initialized = false;
  if (!initialized) {
    int i = 0;
    box(dlg[i++],     0, 0, 320, 200);
    title(dlg[i++], 160, 8, 160, 20, i18nTile.string());

    int group = 1;
    text(dlg[i++],    8, 38, 100, 20, MAP_SHAPE_HELP, i18nMap_Shape.string());
    radio(dlg[i++], 108, 38, 100, 16, MAP_SHAPE_HELP, group, D_SELECTED, i18nRectangle.string());
    radio(dlg[i++], 208, 38, 100, 16, MAP_SHAPE_HELP, group,          0, i18nDiamond.string());

    group ++;
    text(dlg[i++],    8, 68, 100, 20, TILE_SHAPE_HELP, i18nTile_Shape.string());
    radio(dlg[i++], 108, 68, 100, 16, TILE_SHAPE_HELP, group, D_SELECTED, i18nRectangle.string());
    radio(dlg[i++], 208, 68, 100, 16, TILE_SHAPE_HELP, group,          0, i18nDiamond.string());

    text(dlg[i++],    8, 90, 90, 20, TILE_SIZE_HELP, i18nTile_Width.string());
    edit(dlg[i++],   98, 90, 60, 20, TILE_SIZE_HELP, twidth_str, TSIZE_LEN);
    text(dlg[i++],  158, 90, 90, 20, TILE_SIZE_HELP, i18nTile_Height.string());
    edit(dlg[i++],  248, 90, 60, 20, TILE_SIZE_HELP, theight_str, TSIZE_LEN);

    group++;
    text(dlg[i++],    8, 118, 100, 20, TILE_ODDNESS_HELP, i18nOddness.string());
    radio(dlg[i++], 108, 118,  70, 16, TILE_ODDNESS_HELP, group, D_SELECTED, i18nnone.string());
    radio(dlg[i++], 178, 118,  70, 16, TILE_ODDNESS_HELP, group,          0, i18nleft.string());
    radio(dlg[i++], 248, 118,  70, 16, TILE_ODDNESS_HELP, group,          0, i18nright.string());

    text(dlg[i++],    8, 140, 90, 20, TILE_OFFSET_HELP, i18nTile_DX.string());
    edit(dlg[i++],   98, 140, 60, 20, TILE_OFFSET_HELP, tdx_str, TSIZE_LEN);
    text(dlg[i++],  158, 140, 90, 20, TILE_OFFSET_HELP, i18nTile_DY.string());
    edit(dlg[i++],  248, 140, 60, 20, TILE_OFFSET_HELP, tdy_str, TSIZE_LEN);

    button(dlg[i++],   8, 172, 80, 20, '\r', D_EXIT, i18nOK.string());
    button(dlg[i++], 232, 172, 80, 20,    0, D_EXIT, i18nCancel.string());

    end(dlg[i]);

    initialized = true;
  }
  return dlg;
}



static void set_map_shape(void)
{
  MAP_SHAPE ms = seme_map->GetMapShape();
  DBG_ASSERT(ms==MAP_SHAPE_RECT || ms==MAP_SHAPE_DIAMOND);
  if(ms == MAP_SHAPE_RECT) {
    get_tile_dialog()[TILE_MRECT].flags |= D_SELECTED;
    get_tile_dialog()[TILE_MDIAMOND].flags &= ~D_SELECTED;
  }
  else {
    get_tile_dialog()[TILE_MRECT].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_MDIAMOND].flags |= D_SELECTED;
  }
}


static void set_tile_shape(void)
{
  TILE_SHAPE ts = seme_map->GetTileShape();
  DBG_ASSERT(ts==TILE_SHAPE_RECT || ts==TILE_SHAPE_DIAMOND);
  if(ts == TILE_SHAPE_RECT) {
    get_tile_dialog()[TILE_TRECT].flags |= D_SELECTED;
    get_tile_dialog()[TILE_TDIAMOND].flags &= ~D_SELECTED;
  }
  else {
    get_tile_dialog()[TILE_TRECT].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_TDIAMOND].flags |= D_SELECTED;
  }
  uszprintf(
    twidth_str, TSIZE_LEN*SIZEOF_CHARACTER, UString("%d"), seme_map->GetTileW()
  );
  uszprintf(
    theight_str,TSIZE_LEN*SIZEOF_CHARACTER, UString("%d"), seme_map->GetTileH()
  );
}


static void set_tile_oddness(void)
{
  ODD odd = seme_map->GetTileOdd();
  DBG_ASSERT(odd==ODD_NONE || odd==ODD_LEFT || odd==ODD_RIGHT);
  if(odd == ODD_NONE) {
    get_tile_dialog()[TILE_ODD_NONE].flags |= D_SELECTED;
    get_tile_dialog()[TILE_ODD_RIGHT].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_ODD_LEFT].flags &= ~D_SELECTED;
  }
  else if(odd == ODD_RIGHT) {
    get_tile_dialog()[TILE_ODD_NONE].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_ODD_RIGHT].flags |= D_SELECTED;
    get_tile_dialog()[TILE_ODD_LEFT].flags &= ~D_SELECTED;
  }
  else {
    get_tile_dialog()[TILE_ODD_NONE].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_ODD_RIGHT].flags &= ~D_SELECTED;
    get_tile_dialog()[TILE_ODD_LEFT].flags |= D_SELECTED;
  }
  uszprintf(
    tdx_str, TSIZE_LEN*SIZEOF_CHARACTER, UString("%d"), seme_map->GetTileDX()
  );
  uszprintf(
    tdy_str, TSIZE_LEN*SIZEOF_CHARACTER, UString("%d"), seme_map->GetTileDY()
  );
}


static void get_map_shape(void)
{
  if(get_tile_dialog()[TILE_MRECT].flags & D_SELECTED) {
    seme_map->SetMapShape(MAP_SHAPE_RECT);
  }
  else {
    DBG_ASSERT(get_tile_dialog()[TILE_MDIAMOND].flags & D_SELECTED);
    seme_map->SetMapShape(MAP_SHAPE_DIAMOND);
  }
}


static void get_tile_shape(void)
{
  if(get_tile_dialog()[TILE_TRECT].flags & D_SELECTED) {
    seme_map->SetTileShape(
      TILE_SHAPE_RECT, uatoi(twidth_str), uatoi(theight_str));
  }
  else {
    DBG_ASSERT(get_tile_dialog()[TILE_TDIAMOND].flags & D_SELECTED);
    seme_map->SetTileShape(
      TILE_SHAPE_DIAMOND, uatoi(twidth_str), uatoi(theight_str));
  }
}


static void get_tile_oddness(void)
{
  if(get_tile_dialog()[TILE_ODD_NONE].flags & D_SELECTED) {
    seme_map->SetTileOffset(uatoi(tdx_str), uatoi(tdy_str), ODD_NONE);
  }
  else if(get_tile_dialog()[TILE_ODD_RIGHT].flags & D_SELECTED) {
    seme_map->SetTileOffset(uatoi(tdx_str), uatoi(tdy_str), ODD_RIGHT);
  }
  else {
    DBG_ASSERT(get_tile_dialog()[TILE_ODD_LEFT].flags & D_SELECTED);
    seme_map->SetTileOffset(uatoi(tdx_str), uatoi(tdy_str), ODD_LEFT);
  }
}


int seme_tile_proc(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_LPRESS || msg == MSG_KEY) {
    seme_button_proc(msg, d, c);
    scare_mouse_area(d->x, d->y, d->w, d->h);
    seme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while(mouse_b) {
    }
    if(IS_IN(d, mouse_x, mouse_y) || msg == MSG_KEY) {
      set_dialog_color(get_tile_dialog(), gui_fg_color, gui_bg_color);
      centre_dialog(get_tile_dialog());
      set_map_shape();
      set_tile_shape();
      set_tile_oddness();
      int ret = popup_dialog(get_tile_dialog(), -1);
      if(ret == TILE_OK) {
        get_map_shape();
        get_tile_shape();
        get_tile_oddness();
      }
    }
    d->flags &= ~D_SELECTED;
    return D_REDRAW;
  }
  return seme_button_proc(msg, d, c);
}

