/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "sememngr.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include <stdlib.h>

#include "semeprop.h"


int SemePropertyManager::count = 0;
const char **SemePropertyManager::names = 0;
const char **SemePropertyManager::ascii_names = 0;
SemeProperty *(**SemePropertyManager::creators)(void) = 0;
SemeProperty **SemePropertyManager::props = 0;


int SemePropertyManager::IsValid(int index)
{
  return index>=0 && index<SemePropertyManager::count;
}


int SemePropertyManager::Count(void)
{
  return SemePropertyManager::count;
}


const char *SemePropertyManager::GetName(int index)
{
  if(!IsValid(index)) return 0;
  return SemePropertyManager::names[index];
}


const char *SemePropertyManager::GetAsciiName(int index)
{
  if(!IsValid(index)) return 0;
  return SemePropertyManager::ascii_names[index];
}


int SemePropertyManager::GetIndex(const char *name)
{
  for (int i=0; i<SemePropertyManager::Count(); i++) {
    if (ustrcmp(GetName(i), name) == 0) {
      return i;
    }
  }
  return SemePropertyManager::Count();
}


SemeProperty *SemePropertyManager::Create(int index)
{
  if(!IsValid(index)) return 0;
  return SemePropertyManager::creators[index]();
}


SemeProperty *SemePropertyManager::Create(const char *name)
{
  return SemePropertyManager::Create(SemePropertyManager::GetIndex(name));
}


int SemePropertyManager::AskAdditionalData(int index, const SemeProperty *p)
{
  if(!IsValid(index)) return 0;
  if(p) props[index]->CopyFrom(p);
  return props[index]->AskAdditionalData();
}


int SemePropertyManager::AskAdditionalData(const char *name, const SemeProperty *p)
{
  return SemePropertyManager::AskAdditionalData(
    SemePropertyManager::GetIndex(name), p
  );
}


SemeProperty *SemePropertyManager::Get(int index)
{
  if(!IsValid(index)) return 0;
  return props[index];
}


SemeProperty *SemePropertyManager::Get(const char *name)
{
  return SemePropertyManager::Get(SemePropertyManager::GetIndex(name));
}


int SemePropertyManager::Register(
  const char *name, SemeProperty *(*creator)(void)
)
{
  names = (const char**) realloc(names, sizeof(char*)*(Count()+1));
  CHECK_POINTER(names);
  names[Count()] = name;

  ascii_names = (const char**) realloc(ascii_names, sizeof(char*)*(Count()+1));
  CHECK_POINTER(ascii_names);
  ascii_names[Count()] = uconvert_dup(name, U_CURRENT, U_ASCII);

  creators = (SemeProperty *(**)(void)) realloc(
    creators, sizeof(SemeProperty*(**)(void))*(Count()+1)
  );
  CHECK_POINTER(creators);
  creators[Count()] = creator;

  props = (SemeProperty **) realloc(props, sizeof(SemeProperty*)*(Count()+1));
  CHECK_POINTER(props);
  props[Count()] = creator();

  count++;
  return Count()-1;
}

