/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "plugin.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include "globals.h"
#include "gui.h"
#include "brushlst.h"
#include "brush.h"
#include "wrapper.h"

#include "semevrsn.h"
#include "semenew.h"
#include "sememap.h"

#include "semefile.h"

#include <allegro.h>
#include <string.h>


#define ALERT_NO_FORMAT() \
  alert( \
    Translation("Please create or load a map format first"), \
    Translation("Menu: Map Format->Load map format"), \
    Translation("or Map Format->New map format"), \
    Translation("OK"), 0, 0, 0 \
  )
  
#define ALERT_NO_BRUSH() \
  alert( \
    Translation("Please create a brush by selecting"), \
    Translation("an area and Edit->Copy Brush"), \
    0, Translation("OK"), 0, 0, 0 \
  )
  
#define UNABLE_TO_SAVE(file) \
  alert(Translation("Unable to save file"), file, 0, Translation("OK"), 0, 0, 0)

#define UNABLE_TO_LOAD(file) \
  alert(Translation("Unable to load file"), file, 0, Translation("OK"), 0, 0, 0)

#define SEME_SECTION UString("[seme]")
#define SEME_CONFIG_FORMAT UString("format")


SemeMap *current_map_format = 0;


/* Encoding
   ---------------------------------*/
int plugin_encoding(void)
{
  return U_UTF8;
}


/* Help
   ---------------------------------*/
#define FORMAT_FILE_STR "--format="

const char *plugin_help(void)
{
  return "seme, version " SEME_VERSION_TEXT "\n"
         "You can also specify a map format file\n"
         "--format=<fname>  Loads the map format file <fname>\n";
}

/* About
   ---------------------------------*/
const char *plugin_about(void)
{
  static char about[1024];
  uszprintf(
    about, sizeof(about), UString("seme, version %d.%d"),
    SEME_VERSION_MAJOR, SEME_VERSION_MINOR
  );
  return about;
}



/* User menus
   ---------------------------------*/
static int _load_map_format(const char *fname)
{
  SemeMap *map_format = new SemeMap();
  CHECK_POINTER(map_format);
  int err = map_format->LoadFormat(fname);
  if(err == LS_NO_ERROR) {
    if(current_map_format) delete current_map_format;
    current_map_format = map_format;
  }
  return err;
}


static void save_map_format(Map * /*map*/)
{
  static char path[STRING_PATH_BASE_LENGTH*SIZEOF_CHARACTER] = EMPTY_STRING;
  if(current_map_format) {
    if(file_select(Translation("Save map format"), path, 0) == TRUE) {
      if(exists(path)) {
        if(!OVERWRITE_FILE(path)) {
          return;
        }
      }
      GUI.SetMouseBusy();
      if(current_map_format->SaveFormat(path) != LS_NO_ERROR) {
        GUI.UnsetMouseBusy();
        ERROR_WRITING(path, EmptyString());
      }
      else {
        char *rel_path = urelativize(path);
        set_config_string(SEME_SECTION, SEME_CONFIG_FORMAT, rel_path);
        free(rel_path);
        GUI.UnsetMouseBusy();
      }
    }
  }
  else {
    ALERT_NO_FORMAT();
  }
}


static void load_map_format(Map * /*map*/)
{
  static char path[STRING_PATH_BASE_LENGTH*SIZEOF_CHARACTER] = EMPTY_STRING;

  if(GUI.GetMap() && !LOSE_MAP()) {
    return;
  }

  if(file_select(Translation("Load map format"), path, 0) == TRUE) {
    if(!exists(path)) {
      DOESNT_EXIST(path);
      return;
    }
    GUI.SetMouseBusy();
    if(_load_map_format(path) != LS_NO_ERROR) {
      GUI.UnsetMouseBusy();
      ERROR_READING(path, EmptyString());
    }
    else {
      char *rel_path = urelativize(path);
      set_config_string(SEME_SECTION, SEME_CONFIG_FORMAT, rel_path);
      free(rel_path);
      brush_list.Clear();
      GUI.ClearMap();
      GUI.UnsetMouseBusy();
    }
  }
}


static void new_map_format(Map * /*map*/)
{
  if(GUI.GetMap() && !LOSE_MAP()) {
    return;
  }

  SemeMap *map_format = seme_popup_edit(0);
  if(map_format) {
    delete current_map_format;
    current_map_format = map_format;
    if(map_format->IsDirty()) {
      brush_list.Clear();
      GUI.ClearMap();
    }
  }
}


static void edit_map_format(Map * /*map*/)
{
  SemeMap *map_format = seme_popup_edit(current_map_format);
  if(map_format) {
    delete current_map_format;
    current_map_format = map_format;
    if(map_format->IsDirty()) {
      brush_list.Clear();
      GUI.ClearMap();
    }
  }
}


static void save_c_code(Map * /*map*/)
{
  static char path[STRING_PATH_BASE_LENGTH*SIZEOF_CHARACTER] = EMPTY_STRING;
  if(current_map_format) {
    if(file_select(Translation("Save C code"), path, 0) == TRUE) {
      GUI.SetMouseBusy();
      if(!current_map_format->SaveCCode(path)) {
        GUI.UnsetMouseBusy();
        UNABLE_TO_SAVE(path);
      }
      else {
        GUI.UnsetMouseBusy();
      }
    }
  }
  else {
    ALERT_NO_FORMAT();
  }
}


static Brush *load_brush(const char *fname)
{
  Brush *brush = 0;
  if(current_map_format) {
    GUI.SetMouseBusy();
    brush = current_map_format->LoadBrush(fname);
    if (!brush) {
      UNABLE_TO_LOAD(fname);
    }
    GUI.UnsetMouseBusy();
  }
  else {
    ALERT_NO_FORMAT();
  }
  return brush;
}


static bool save_brush(const char *fname, const Brush *brush)
{
  bool ret = false;
  if(current_map_format) {
    if(GUI.GetBrush()) {
      int err;
      GUI.SetMouseBusy();
      err = current_map_format->SaveBrush(fname, brush);
      GUI.UnsetMouseBusy();
      if(err) {
        UNABLE_TO_SAVE(fname);
      }
      else {
        //Brush *new_brush = brush->Clone();
        //CHECK_POINTER(new_brush);
        //brush_list.AddBrush(new_brush, fname);
        ret = true;
      }
    }
    else {
      ALERT_NO_BRUSH();
    }
  }
  else {
    ALERT_NO_FORMAT();
  }
  return ret;
}


/* Initialization
   ---------------------------------*/
int plugin_init(int argc, const char **argv)
{
  if(get_uformat() != U_ASCII) {
    //abort();
  }
  char *format_file = 0;
  for(int i=1; i<argc; i++) {
    if(strncmp(argv[i], FORMAT_FILE_STR, strlen(FORMAT_FILE_STR)) == 0) {
      if(format_file) free(format_file);
      format_file = uconvert_dup(
        argv[i]+strlen(FORMAT_FILE_STR), U_ASCII, U_CURRENT
      );
      DBG_ASSERT(format_file);
    }
  }
  if(!format_file) {
    const char *ff =
      get_config_string(SEME_SECTION, SEME_CONFIG_FORMAT, EmptyString());
    format_file = uderelativize(ff);
    DBG_ASSERT(format_file);
  }
  if(format_file) {
    _load_map_format(format_file);
    free(format_file);
  }

  GUI.SetAppName("Simple eme");

  GUI.SetUserMenuTitle(Translation("Map Forma&t"));
  GUI.SetUserMenu(0, Translation("&Load map format"), load_map_format);
  GUI.SetUserMenu(1, Translation("&Save map format"), save_map_format);
  GUI.SetUserMenu(2, Translation("&New map format"), new_map_format);
  GUI.SetUserMenu(3, Translation("&Edit map format"), edit_map_format);
  GUI.SetUserMenu(4, EmptyString(), 0);
  GUI.SetUserMenu(5, Translation("Save &C code"), save_c_code);

  GUI.AllowFeature(true, Features::Group::Everything);
  /* Disallow having layers of various size */
  GUI.AllowFeature(false, Features::Group::VariableLayerSize);
  /* Disallow changing the number and place of layers */
  GUI.AllowFeature(false, Features::Group::VariableLayerCount);

  GUI.SetLoadBrush(load_brush);
  GUI.SetSaveBrush(save_brush);

  return LS_NO_ERROR;
}


/* Exit
   ---------------------------------*/
void plugin_exit(void)
{
  if(current_map_format) delete current_map_format;
}


/* New map
   ---------------------------------*/
int plugin_new(void)
{
  if(current_map_format) {
    int err;
    err = current_map_format->NewMap();
    return err;
  }
  else {
    ALERT_NO_FORMAT();
  }
  return LS_NO_ERROR;
}


/* Load map
   ---------------------------------*/
int plugin_load(const char *fname)
{
  if(current_map_format) {
    int err;
    GUI.SetMouseBusy();
    err = current_map_format->LoadMap(fname);
    GUI.UnsetMouseBusy();
    return err;
  }
  else {
    ALERT_NO_FORMAT();
  }
  return LS_NO_ERROR;
}


/* Save map
   ---------------------------------*/
int plugin_save(const char *fname, const Map *map)
{
  if(current_map_format) {
    int err;
    GUI.SetMouseBusy();
    const Wrapper wrapper(map);
    err = current_map_format->SaveMap(fname, wrapper);
    GUI.UnsetMouseBusy();
    return err;
  }
  else {
    ALERT_NO_FORMAT();
  }
  return LS_NO_ERROR;
}

