/* ieme - a simple bitmap editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "plugin.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include "wrapper.h"

#include "globals.h"
#include "gui.h"
#include "brushlst.h"
#include "selected.h"
#include "brush.h"
#include "tiles.h"
#include "stiles.h"
#include "ftiles.h"
#include "commands.h"
#include "cmdgroup.h"
#include "cmdmap.h"
#include "cmdpaste.h"
#include "cmdselec.h"
#include "cmdlayer.h"
#include "lutils.h"
#include "lviewed.h"
#include "map.h"

#include "ccolor.h"
#include "cpercent.h"

#include "alxpm.h"

#include <altheme.h>



/* Have to specify the layer count at construction because of the define just
 * above */
#define MIN_LAYER_COUNT 2

#define PROPERTY_COUNT 0


#define COLOR_LAYER_NAME "Color"
#define MASK_LAYER_NAME  "Mask"
#define NEW_LAYER_NAME   "New layer"


static void alert(const char *s1, const char *s2, const char *s3)
{
  alert(s1, s2, s3, Translation("OK"), 0, 0, 0);
}



/* Mask creator and property
   Inherit from Percent, just need to re-define draw, the property type is 
   still Percent
   ---------------------------------*/
const int TransparentMask = 0;
const int OpaqueMask = 255;

struct Mask {
  class Creator;
  typedef Percent::Property Property;
  typedef int Base;
  typedef void Param;


  class Creator: public Percent::Creator {

  public:
    Creator *Clone(void) const
      { return new Creator(GetName(), Cast<Mask>(GetReference())->Get()); }

    /*
    void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
      float scale, int l, int i, int j
    ) const
      {
        DBG_ASSERT(IsSameType(p));
        const Property *pm = Cast<Mask>(p);
        const int v = pm->Get();
        rectfill(bmp, x, y, x+w-1, y+h-1, makecol(v, v, v));
      }
      */

    Creator(StaticString *name, int value): Percent::Creator(name, 255, value)
      { DBG_SET_CREATOR_TYPE("Mask"); }
    virtual ~Creator(void)
      {}
  }; /* class Mask::Creator */
}; /* struct Mask */



/* Palette for 8 bits images
   ---------------------------------*/
static RGB static_palette[256];



/* Blur
   ---------------------------------*/
static void _add_blur(
  const Tiles *layer, int i, int j, int factor,
  int *r, int *g, int *b, int *n
)
{
  if (layer->is_in(i, j)) {
    const Color::Property *c = Cast<Color>(layer->get(i, j));
    if (c) {
      (*r) += c->GetRed() * factor;
      (*g) += c->GetGreen() * factor;
      (*b) += c->GetBlue() * factor;
      (*n) += factor;
    }
  }
}

static void _tileable_add_blur(
  const Tiles *layer, int i, int j, int factor,
  int *r, int *g, int *b, int *n
)
{
  i = ((i+layer->endi())%layer->endi());
  j = ((j+layer->endj())%layer->endj());
  const Color::Property *c = Cast<Color>(layer->get(i, j));
  if (c) {
    (*r) += c->GetRed() * factor;
    (*g) += c->GetGreen() * factor;
    (*b) += c->GetBlue() * factor;
    (*n) += factor;
  }
}

static void _blur_tile(
  const Tiles *layer, const Color::Creator *creator, SparseTiles *tiles,
  int i, int j,
  void (*add_blur)(const Tiles*, int, int, int, int*, int*, int*, int*)
)
{
  typedef Color::Property ColorProperty;
  const Color::Property *c = Cast<Color>(layer->get(i, j));
  if (c) {
    int num = 1;
    int red = c->GetRed();
    int green = c->GetGreen();
    int blue = c->GetBlue();
    int factor_1 = 10;
    int factor_sqrt2 = 7;
    add_blur(layer, i-0, j-0, 4*factor_1,     &red, &green, &blue, &num);
    add_blur(layer, i-1, j-1, factor_sqrt2, &red, &green, &blue, &num);
    add_blur(layer, i-1, j+0, factor_1,     &red, &green, &blue, &num);
    add_blur(layer, i-1, j+1, factor_sqrt2, &red, &green, &blue, &num);
    add_blur(layer, i+0, j+1, factor_1,     &red, &green, &blue, &num);
    add_blur(layer, i+1, j+1, factor_sqrt2, &red, &green, &blue, &num);
    add_blur(layer, i+1, j+0, factor_1,     &red, &green, &blue, &num);
    add_blur(layer, i+1, j-1, factor_sqrt2, &red, &green, &blue, &num);
    add_blur(layer, i+0, j-1, factor_1,     &red, &green, &blue, &num);
    red/=num; green/=num; blue/=num;
    ColorProperty *new_c = creator->Create(red, green, blue);
    tiles->set(i, j, new_c);
  }
}


static void blur(Map *map)
{
  if(!map) return;

  Wrapper wrapper(map);
  int layer_index = wrapper.GetActiveLayerIndex();

  if (layer_index==wrapper.FindLayerIndex(MASK_LAYER_NAME)) {
    popup_printf(Translation("Error"), Translation("Can't blur mask layer"));
    return;
  }

  void (*add_blur)(const Tiles*, int, int, int, int*, int*, int*, int*);
  int tileable = alert(
    EmptyString(), Translation("Tileable ?"), EmptyString(),
    Translation("Yes"), Translation("No"), 0, 0
  )==1;

  if(tileable) {
    add_blur = _tileable_add_blur;
  }
  else {
    add_blur = _add_blur;
  }

  GUI.SetMouseBusy();

  const Color::Creator *creator =
    wrapper.GetLayerCreator<Color>(layer_index);
  SparseTiles *tiles = new SparseTiles(creator->Clone());

  Tiles *layer = wrapper.GetLayers()->get(layer_index);
  if (wrapper.GetSelectedTiles()->Num() > 0) {
    const SelectedTiles *sel = wrapper.GetSelectedTiles();
    for(SelectedTiles::const_iterator id=sel->Begin(); id!=sel->End(); ++id) {
      _blur_tile(
        layer, creator, tiles, sel->GetX(id), sel->GetY(id), add_blur
      );
    }
  }
  else {
    int width = layer->endi();
    int height = layer->endj();
    for(int i=0; i<width; i++) {
      for(int j=0; j<height; j++) {
        _blur_tile(layer, creator, tiles, i, j, add_blur);
      }
    }
  }

  wrapper.Execute(
    new CommandPaste(map, layer_index, tiles, tiles->begini(), tiles->beginj())
  );
  delete tiles;

  GUI.UnsetMouseBusy();
}


/* Outline
   ---------------------------------*/
static void outline(Map *map)
{
  typedef Color::Property ColorProperty;
  if(!map) return;

  Wrapper wrapper(map);
  int layer = wrapper.GetActiveLayerIndex();

  if (layer==wrapper.FindLayerIndex(MASK_LAYER_NAME)) {
    popup_printf(Translation("Error"), Translation("Can't outline mask layer"));
    return;
  }

  /* Change the mouse pointer to a clock */
  GUI.SetMouseBusy();

  const BaseCreator *creator = wrapper.RawGetCreator(layer);
  SparseTiles *tiles = new SparseTiles(creator->Clone());

  /* Put all the commands in a group so they can be undone in one shot */
  CommandGroup *group = new CommandGroup();
  /* Should begin with bottom right, because it'll change the map size */
  group->Add(new CommandStack<CommandInsertColumn>(map, wrapper.GetMapWidth(), wrapper.GetMapHeight(), 1));
  group->Add(new CommandStack<CommandInsertRow>(map, wrapper.GetMapWidth(), wrapper.GetMapHeight(), 1));
  group->Add(new CommandStack<CommandInsertColumn>(map, 0, 0, 1));
  group->Add(new CommandStack<CommandInsertRow>(map, 0, 0, 1));

  int width = wrapper.GetLayer(layer)->endi();
  int height = wrapper.GetLayer(layer)->endj();
  for(int i=0; i<width; i++) {
    for(int j=0; j<height; j++) {
      const Color::Property *c = wrapper.GetTileProperty<Color>(layer, i, j);
      if (!c || c->Get()==Color::majik) {
        const Color::Property *right =
          i<width-1 ? wrapper.GetTileProperty<Color>(layer, i+1, j+0) : 0;
        const Color::Property *left =
          i>0 ? wrapper.GetTileProperty<Color>(layer, i-1, j+0) : 0;
        const Color::Property *down =
          j<height-1 ? wrapper.GetTileProperty<Color>(layer, i+0, j+1) : 0;
        const Color::Property *up =
          j>0 ? wrapper.GetTileProperty<Color>(layer, i+0, j-1) : 0;
        if (
          right && right->Get()!=Color::majik ||
          left && left->Get()!=Color::majik ||
          down && down->Get()!=Color::majik ||
          up && up->Get()!=Color::majik
        ) {
          tiles->copy(i, j, creator->GetReference());
        }
      }
    }
  }

  group->Add(
    new CommandPaste(map, layer, tiles, tiles->begini(), tiles->beginj())
  );
  wrapper.Execute(group);

  delete tiles;

  /* Put back the arrow mouse pointer */
  GUI.UnsetMouseBusy();
}



/* Saving mask
   ---------------------------------*/
static void save_mask(Map *map)
{
  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;

  if(!map) return;

  Wrapper wrapper(map);
  int mask_layer_index = wrapper.FindLayerIndex(MASK_LAYER_NAME);

  if (mask_layer_index==wrapper.GetNumLayers()) {
    popup_printf(Translation("Error"), Translation("No mask layer"));
    return;
  }
  
  if(file_select(Translation("Save mask file"), path, NULL) == TRUE) {
    if(exists(path)) {
      if(!OVERWRITE_FILE(path)) {
        return;
      }
    }
    GUI.SetMouseBusy();
    BITMAP *bmp;

    bmp = create_bitmap(wrapper.GetMapWidth(), wrapper.GetMapHeight());
    if(!bmp) {
      GUI.UnsetMouseBusy();
      alert(
        Translation("Error"),
        Translation("Unable to create temporarly bitmap"), 0
      );
      return;
    }

    for(int i=0; i<bmp->w; i++) {
      for(int j=0; j<bmp->h; j++) {
        if (wrapper.TileExists(mask_layer_index, i, j)) {
          const Mask::Property *c =
            wrapper.GetTileProperty<Mask>(mask_layer_index, i, j);
          putpixel(bmp, i, j, makecol(c->Get(), c->Get(), c->Get()));
        }
        else {
          putpixel(bmp, i, j, makecol(255, 255, 255));
        }
      }
    }

    if(save_bitmap(path, bmp, 0)!=0) {
      destroy_bitmap(bmp);
      GUI.UnsetMouseBusy();
      alert(Translation("Error"), Translation("Unable to save file"), path);
      return;
    }

    destroy_bitmap(bmp);
    GUI.UnsetMouseBusy();
  }
}


/* Loading mask
   ---------------------------------*/
static void load_mask(Map *map)
{
  if(!map) return;

  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;
  if(file_select(Translation("Load mask file"), path, NULL) == TRUE) {
    GUI.SetMouseBusy();
    BITMAP *bmp;
    RGB pal[256];

    bmp = load_bitmap(path, pal);
    if(!bmp) {
      GUI.UnsetMouseBusy();
      alert(Translation("Error"), Translation("Unable to load file"), path);
      return;
    }

    // TODO

    destroy_bitmap(bmp);
    GUI.UnsetMouseBusy();
  }
}


#if 0
/* Palette editing
   ---------------------------------*/
typedef enum {
  PALETTE_BOX,
  PALETTE_TITLE,
  PALETTE_LIST,
  PALETTE_CANCEL,
  NUM_PALETTE_DIALOGS
} PALETTE_DIALOG;

static char *color_cb(int index, int *listsize, DIALOG * /*d*/)
{
  if(index<0) {
    *listsize = 256;
    return 0;
  }
  else {
    return (char*)EmptyString().string();
  }
}

static int color_drawer(BITMAP *bmp, int index, int x, int y, int w, DIALOG *d)
{
  static const int height = 10;
  if(bmp) {
    int r = static_palette[index].r;
    int g = static_palette[index].g;
    int b = static_palette[index].b;
    rectfill(bmp, x, y, x+height*2-1, y+height-1, makecol(r<<2, g<<2, b<<2));
    text_mode(gui_bg_color);
    textprintf(
      bmp, font, x+height*3, y, gui_fg_color,
      UString("%3d %3d %3d"), r<<2, g<<2, b<<2
    );
  }
  return height;
}

static DIALOG *palette_dialogs(void)
{
  static const Translation i18nPalette("Palette");
  static const Translation i18nDone("Done");

  static DIALOG palette_dialogs[] = {
  /* proc:
       x,   y,   w,   h,fg,bg,key,flags,d1,d2,             dp, dp2, dp3 */
    {altheme_shadow_box_proc,
       0,   0, 320, 200, 0, 0,  0,     0, 0, 0,              0,       0,0},
    {altheme_ctext_proc,
     160,   8, 160,  20, 0, 0,  0,     0, 0, 0,(void*)i18nPalette.string(),0,0},
  
    {altheme_elist_proc,
       8,  36, 304, 128, 0, 0,  0,D_EXIT, 0, 0, color_cb, color_drawer, 0},
  
    {altheme_button_proc,
     120, 172,  80,  20, 0, 0,  0,D_EXIT, 0, 0,(void*)i18nDone.string(),   0,0},
    {}
  };

  return palette_dialogs;
}

static void init_palette_dialogs(void)
{
  static int done = 0;
  if(!done) {
    //TODO
    done = 1;
  }
}

static int color_distance(int r, int g, int b, int n)
{
  return
    ABS(static_palette[n].r-r) + ABS(static_palette[n].g-g) +
    ABS(static_palette[n].b-b);
}

static int nearest_color(int r, int g, int b)
{
  int dist = color_distance(r, g, b, 0);
  int index = 0;
  for(int n=0; n<256; n++) {
    int d = color_distance(r, g, b, n);
    if(d<dist) {
      dist = d;
      index = n;
    }
  }
  return index;
}

static void edit_palette(Map *map)
{
  init_palette_dialogs();

  if(!map) return;

  Wrapper wrapper(map);
  int color_layer_index = wrapper.FindLayerIndex(COLOR_LAYER_NAME);

  /* Must not take the default from the layer, because it's a real default,
     while the creator's default prop is variable */
  const BaseCreator *entry_creator = wrapper.RawGetCreator(color_layer_index);
  const Color::Property *prop = Cast<Color>(entry_creator->GetReference());
  palette_dialogs()[PALETTE_LIST].d1 = palette_dialogs()[PALETTE_LIST].d2 =
    nearest_color(prop->GetRed()>>2, prop->GetGreen()>>2, prop->GetBlue()>>2);

  set_dialog_color(palette_dialogs(), gui_fg_color, gui_bg_color);
  centre_dialog(palette_dialogs());

  popup_dialog(palette_dialogs(), -1);

  int r = static_palette[palette_dialogs()[PALETTE_LIST].d1].r;
  int g = static_palette[palette_dialogs()[PALETTE_LIST].d1].g;
  int b = static_palette[palette_dialogs()[PALETTE_LIST].d1].b;

  entry_creator->UpdateReferenceProperty(Color::Property(r<<2, g<<2, b<<2));
}
#else
static void edit_palette(Map *map)
{
  NOT_YET_DONE();
}
#endif


/* Layers
   ---------------------------------*/
static void new_layer(Map *map, int active_layer_index)
{
  Wrapper wrapper(map);
  int x = 0;
  int y = 0;
  int w = wrapper.GetMapWidth();
  int h = wrapper.GetMapHeight();
  if (popup_size_and_pos(Translation("New Layer"), &x, &y, &w, &h, 0, 0) == TRUE) {
    GUI.SetMouseBusy();
    /* Create a sparse color layer */
    BaseCreator *creator = new Color::Creator(new StaticString(NEW_LAYER_NAME), Color::majik);
    Tiles *layer = new SparseTiles(creator, x, y, w, h);
    /* Insert the layer in the map */
    Command *cmd = new CommandInsertLayer(map, layer, active_layer_index);
    wrapper.Execute(cmd);
    GUI.UnsetMouseBusy();
  }
}


static void flatten_image(Map *map)
{
  Wrapper wrapper(map);
  int mask_layer_index = wrapper.FindLayerIndex(MASK_LAYER_NAME);

  int w = wrapper.GetMapWidth();
  int h = wrapper.GetMapHeight();
  BaseCreator *creator =
    new Color::Creator(new StaticString(COLOR_LAYER_NAME), Color::majik);
  Tiles *color_layer = new FullTiles(creator, 0, 0, w, h);

  // FIXME i should not have to do that here
  for (int j=0; j<h; ++j) for (int i=0; i<w; ++i) {
    BaseProperty *p = creator->Create();
    color_layer->set(i, j, p);
  }

  SelectedTiles *all = LayerUtils::SelectAll(color_layer);
  for (int l=0; l<wrapper.GetNumLayers(); ++l) {
    if (l!=mask_layer_index) {
      Tiles *layer = wrapper.GetLayer(l);
      SparseTiles *tiles = LayerUtils::Copy(layer, all);
      int x = tiles->begini();
      int y = tiles->beginj();
      LayerUtils::Paste(color_layer, x, y, tiles);
      delete tiles;
    }
  }
  delete all;

  CommandGroup *group = new CommandGroup();
  for (int l=wrapper.GetNumLayers()-1; l>=0; --l) {
    if (l!=mask_layer_index) {
      group->Add(new CommandRemoveLayer(map, l));
    }
  }
  group->Add(new CommandInsertLayer(map, color_layer, 0));
  wrapper.Execute(group);
}



/* Brushes
   ---------------------------------*/
static Brush *load_brush(const char *fname)
{
  typedef Color::Creator ColorCreator;
  typedef Color::Property ColorProperty;

  Brush *brush = 0;

  GUI.SetMouseBusy();
  RGB pal[256];
  BITMAP *bmp = load_bitmap(fname, pal);
  if(bmp) {
    /* Create an empty brush */
    brush = new Brush();
    BaseCreator *creator =
      new ColorCreator(new StaticString(Translation(COLOR_LAYER_NAME)), Color::majik);
    /* Create a layer for the brush (SparseTiles gets ownership of creator) */
    SparseTiles *colors = new SparseTiles(creator);
    int mask = bitmap_mask_color(bmp);
    for(int i=0; i<bmp->w; i++) {
      for(int j=0; j<bmp->h; j++) {
        int pix = getpixel(bmp, i, j);
        if(pix!=mask) {
          /* Fill the layer */
          ColorProperty *p =
            Cast<Color>(creator)->Create(getr(pix), getg(pix), getb(pix));
          colors->set(i, j, p);
        }
      }
    }
    /* Put the layer in the brush */
    brush->SetLayer(0, colors);
    destroy_bitmap(bmp);
    GUI.UnsetMouseBusy();
  }
  else {
    GUI.UnsetMouseBusy();
    alert(Translation("Error"), Translation("Unable to load file"), fname);
  }
  return brush;
}


static bool save_brush(const char *fname, const Brush *brush)
{
  popup_printf(Translation("Error"), Translation("Please don't do that"));
  return false;
  // FIXME: what happens when the brush contains only a mask layer !!
  // Or when there are several layers

  GUI.SetMouseBusy();
  /* Get the current brush */
  if(brush) {
    int dx = brush->MinX();
    int dy = brush->MinY();
    int w = brush->MaxX() - dx + 1;
    int h = brush->MaxY() - dy + 1;
    BITMAP *bmp = create_bitmap(w, h);
    if(bmp) {
      clear_to_color(bmp, bitmap_mask_color(bmp));
      /* Get a layer from the brush */
      const SparseTiles *layer = brush->GetLayer(0);
      if (layer) {
        SparseTiles::const_iterator id;
        for (id=layer->begin(); id!=layer->end(); ++id) {
          int i = layer->get_i(id) - dx;
          int j = layer->get_j(id) - dy;
          /* Get a tile from the brush */
          const Color::Property *p = Cast<Color>(layer->get_tile(id));
          int pix = makecol(p->GetRed(), p->GetGreen(), p->GetBlue());
          putpixel(bmp, i, j, pix);
        }
        save_bitmap(fname, bmp, 0);
      }
      destroy_bitmap(bmp);
    }
    /* Add the brush to the brush list */
    brush_list.AddBrush(brush->Clone(), fname);
  }
  GUI.UnsetMouseBusy();
  return true;
}


/* Encoding
   ---------------------------------*/
int plugin_encoding(void)
{
  return U_UTF8;
}


/* Help
   ---------------------------------*/
const char *plugin_help(void)
{
  return "";
}

const char *plugin_about(void)
{
  return EmptyString();
}


/* Init
   ---------------------------------*/
int plugin_init(int argc, const char ** argv)
{
  if(argc>1) {
    return LS_ERROR_OPTION;
  }

  register_bitmap_file_type("xpm", load_xpm, save_xpm);

  GUI.SetAppName("Image eme");

  /* Allow all commands */
  GUI.AllowFeature(true, Features::Group::Everything);

  /* Create my own menu commands */
  GUI.SetUserMenuTitle(Translation("&Misc"));
  int i=0;
  GUI.SetUserMenu(i++, Translation("&Blur"), blur);
  GUI.SetUserMenu(i++, Translation("&Outline"), outline, 'o');
  GUI.SetUserMenu(i++, EmptyString(), 0);
  GUI.SetUserMenu(i++, Translation("&Load Mask"), load_mask);
  GUI.SetUserMenu(i++, Translation("&Save Mask"), save_mask);
  GUI.SetUserMenu(i++, EmptyString(), 0);
  GUI.SetUserMenu(i++, Translation("&Pick Color"), edit_palette);
  GUI.SetUserMenu(i++, EmptyString(), 0);
  GUI.SetUserMenu(i++, Translation("&Flatten Image"), flatten_image);

  /* Specify the load and save brush functions */
  GUI.SetLoadBrush(load_brush);
  GUI.SetSaveBrush(save_brush);

  GUI.SetCreateLayer(new_layer);

  get_palette(static_palette);

  return LS_NO_ERROR;
}


/* Exit
   ---------------------------------*/
void plugin_exit(void)
{
}


/* New
   ---------------------------------*/
int plugin_new(void)
{
  int width = 0;
  int height = 0;

  if (popup_size_and_pos(
    Translation("New image"), 0, 0, &width, &height, 0, 0
  ) == TRUE) {

    if (width>0 && height>0) {
      GUI.SetMouseBusy();

      int color_layer_index = 0;
      int mask_layer_index = 1;

      /* The wrapper creates the map and allows you to set or get its values */
      Wrapper wrapper(width, height, MIN_LAYER_COUNT, PROPERTY_COUNT);

      /* Some misc settings */
      wrapper.SetTileShape(TILE_SHAPE_RECT, 1, 1);
      wrapper.SetTileOffset(1, 1, ODD_NONE);
      wrapper.SetMapShape(MAP_SHAPE_RECT);

      /* Create the color layer */
      wrapper.SetLayer<Color>(
        color_layer_index, Translation(COLOR_LAYER_NAME), FULL_LAYER, Color::majik
      );
      /* Fill it with the default value */
      wrapper.FillLayer(color_layer_index);

      /* Create the mask layer and let it empty. When no pixel on the mask,
         visible */
      wrapper.SetLayer<Mask>(
        mask_layer_index, Translation(MASK_LAYER_NAME), SPARSE_LAYER, TransparentMask
      );

      /* Give the map to the GUI */
      wrapper.SetMap();
      /* After calling SetMap, wrapper will not be able to do other things */

      generate_332_palette(static_palette);

      GUI.UnsetMouseBusy();
    }
    else {
      /* width_val<=0 || height_val<=0 */
      alert(
        Translation("Map width and height"),
        Translation("must be strictly positive"), 0
      );
    }
  }
  return LS_NO_ERROR;
}


/* Load
   ---------------------------------*/
int plugin_load(const char *fname)
{
  GUI.SetMouseBusy();

  set_color_conversion(COLORCONV_NONE);

  BITMAP *bmp = load_bitmap(fname, static_palette);
  if(!bmp) {
    GUI.UnsetMouseBusy();
    return LS_ERROR_READ;
  }

  select_palette(static_palette);

  /* Create the wrapper */
  Wrapper wrapper(bmp->w, bmp->h, MIN_LAYER_COUNT, PROPERTY_COUNT);
  int color_layer_index = 0;
  int mask_layer_index = 1;

  /* Some misc settings */
  wrapper.SetTileShape(TILE_SHAPE_RECT, 1, 1);
  wrapper.SetTileOffset(1, 1, ODD_NONE);
  wrapper.SetMapShape(MAP_SHAPE_RECT);

  /* Create the color layer */
  wrapper.SetLayer<Color>(
    color_layer_index, Translation(COLOR_LAYER_NAME), FULL_LAYER, Color::majik
  );
  /* Create the mask layer. When no pixel on the mask, visible */
  wrapper.SetLayer<Mask>(
    mask_layer_index, Translation(MASK_LAYER_NAME), SPARSE_LAYER, TransparentMask
  );

  int mask_color = bitmap_mask_color(bmp);
  int bpp = bitmap_color_depth(bmp);
  for(int i=0; i<bmp->w; i++) {
    for(int j=0; j<bmp->h; j++) {
      int pix = getpixel(bmp, i, j);
      /* Put the correct color */
      wrapper.SetTileValue<Color>(
        color_layer_index, i, j, Color::pack(
          getr_depth(bpp, pix), getg_depth(bpp, pix), getb_depth(bpp, pix)
        )
      );
      /* Put a mask pixel, only if the color is the mask color or if there is
         a mask specified in the file */
      if(pix==mask_color) {
        wrapper.SetTileValue<Mask>(
          mask_layer_index, i, j, TransparentMask
        );
      }
      else if(bpp==32 && geta_depth(bpp, pix)!=OpaqueMask) {
        wrapper.SetTileValue<Mask>(
          mask_layer_index, i, j, geta_depth(bpp, pix)
        );
      }
    }
  }

  if (GUI.IsInteractive()) {
    GUI.UnsetMouseBusy();
    popup_printf(
      Translation("Info"), Translation("%dx%d pixels, %d bpp"),
      bmp->w, bmp->h, bitmap_color_depth(bmp)
    );
    GUI.SetMouseBusy();
  }

  /* Give the map to the GUI */
  wrapper.SetMap();
  /* After calling SetMap, wrapper will not be able to do other things */

  destroy_bitmap(bmp);

  set_color_conversion(COLORCONV_TOTAL);
  GUI.UnsetMouseBusy();
  return LS_NO_ERROR;
}


/* Save
   ---------------------------------*/
static int save_checks(const Wrapper &wrapper, int &color_layer_index, int &mask_layer_index)
{
  int new_layer_index = wrapper.FindLayerIndex(NEW_LAYER_NAME);
  if (new_layer_index<wrapper.GetNumLayers()) {
    popup_printf(
      Translation("Error"),
      Translation("Please flatten your image\n Menu: Misc->Flatten Image")
    );
    return LS_ERROR_UNKNOWN;
  }
  color_layer_index = wrapper.FindLayerIndex(COLOR_LAYER_NAME);
  if (color_layer_index==wrapper.GetNumLayers()) {
    popup_printf(
      Translation("Error"),
      Translation("Please flatten your image\n Menu: Misc->Flatten Image")
    );
    return LS_ERROR_UNKNOWN;
  }
  mask_layer_index = wrapper.FindLayerIndex(MASK_LAYER_NAME);

  return LS_NO_ERROR;
}

static bool save_options(
  const Wrapper &wrapper, int mask_layer_index, int &do_save_mask, int &convert_to_majik, int &bpp
)
{
  static const Translation i18nIgnore_mask("Ignore mask");
  static const Translation i18nSave_transparent_mask_as_majik(
    "Save transparent mask as majik"
  );
  static const Translation i18nSave_mask_as_alpha("Save mask as alpha");
  static const Translation i18nSave_transparent_mask_as_majik_and_mask_as_alpha(
    "Save transparent mask as majik and mask as alpha"
  );
  struct {
    int do_save_mask; int convert_to_majik; const char *text;
  } options[] = {
    {0, 0, i18nIgnore_mask},
    {0, 1, i18nSave_transparent_mask_as_majik},
    {1, 0, i18nSave_mask_as_alpha},
    {1, 1, i18nSave_transparent_mask_as_majik_and_mask_as_alpha},
  };
  static const int default_todo = 2;

  static const Translation us8_bpp("8 bpp");
  static const Translation us15_bpp("15 bpp");
  static const Translation us16_bpp("16 bpp");
  static const Translation us24_bpp("24 bpp");
  static const Translation us32_bpp("32 bpp");
  struct {
    int bpp; const char *text;
  } cdepths[] = {
    {8,  us8_bpp},
    {15, us15_bpp},
    {16, us16_bpp},
    {24, us24_bpp},
    {32, us32_bpp}
  };
  static const int default_cdepth = 4;

  int todo = default_todo;
  if (GUI.IsInteractive()) {
    if (mask_layer_index!=wrapper.GetNumLayers()) {
      /* A mask, so ask how it should be handled */
      todo = popup_radio(
        Translation("Mask Handling"), 0, 4,
        options[0].text, options[1].text, options[2].text, options[3].text
      );
      if (todo<0) {
        return false;
      }
    }
    else {
      /* No mask, so ignore it */
      todo = 0;
    }
  }

  int cdepth = default_cdepth;
  if (GUI.IsInteractive()) {
    cdepth = popup_radio(
      Translation("Color depth"), 2, 5,
      cdepths[0].text, cdepths[1].text, cdepths[2].text,
      cdepths[3].text, cdepths[4].text
    );
    if (cdepths<0) {
      return false;
    }
  }

  do_save_mask = options[todo].do_save_mask;
  convert_to_majik = options[todo].convert_to_majik;
  bpp = cdepths[cdepth].bpp;

  return true;
}


static int save_get_color(
  const Color::Property *c, const Mask::Property *m,
  int convert_to_majik, int do_save_mask, int bpp
)
{
  int color;
  if(convert_to_majik && m && m->Get()==TransparentMask) {
    color = makecol_depth(bpp, 255, 0, 255);
  }
  else if(do_save_mask) {
    if(m) {
      color = makeacol_depth(
        bpp, c->GetRed(), c->GetGreen(), c->GetBlue(), m->Get()
      );
    }
    else {
      color = makeacol_depth(
        bpp, c->GetRed(), c->GetGreen(), c->GetBlue(), OpaqueMask
      );
    }
  }
  else {
    color = makecol_depth(bpp, c->GetRed(), c->GetGreen(), c->GetBlue());
  }
  return color;
}


int plugin_save(const char *fname, const Map *map)
{
  const Wrapper wrapper(map);
  BITMAP *bmp;
  int color_layer_index;
  int mask_layer_index;
  int err;

  /* Checks */
  err = save_checks(wrapper, color_layer_index, mask_layer_index);
  if (err!=LS_NO_ERROR) {
    return err;
  }

  int do_save_mask;
  int convert_to_majik;
  int bpp;
  if (!save_options(wrapper, mask_layer_index, do_save_mask, convert_to_majik, bpp)) {
    return LS_NO_ERROR;
  }

  GUI.SetMouseBusy();

  bmp = create_bitmap_ex(bpp, wrapper.GetMapWidth(), wrapper.GetMapHeight());
  if(!bmp) {
    GUI.UnsetMouseBusy();
    return LS_ERROR_MEM;
  }

  if(bpp==8) {
    select_palette(static_palette);
  }

  /* Create the bitmap to save */
  for(int i=0; i<bmp->w; i++) {
    for(int j=0; j<bmp->h; j++) {
      /* Get the color and mask of the current pixel */
      const Color::Property *c =
        wrapper.GetTileProperty<Color>(color_layer_index, i, j);
      const Mask::Property *m =
        mask_layer_index!=wrapper.GetNumLayers() && wrapper.TileExists(mask_layer_index, i, j) ?
        wrapper.GetTileProperty<Mask>(mask_layer_index, i, j) : 0;

      /* Compute the pixel value depending on the save options */
      int color = save_get_color(c, m, convert_to_majik, do_save_mask, bpp);

      /* Put the pixel on the bitmap */
      putpixel(bmp, i, j, color);
    }
  }

  int ret = LS_NO_ERROR;
  if(bpp==8) {
    err = save_bitmap(fname, bmp, static_palette);
    unselect_palette();
  }
  else {
    err = save_bitmap(fname, bmp, 0);
  }
  if(err!=0) {
    ret = LS_ERROR_WRITE;
  }

  GUI.UnsetMouseBusy();
  destroy_bitmap(bmp);
  return LS_NO_ERROR;
}

