/*
 * An example of how to use seme generated C file
 *
 * The base structure of the code should be (<FILE> is the uppercase name of
 * the C file, <file> is the lowercase name of the C file):
 *  - declare the map:
 *        <FILE>_MAP *map;
 *  - load the map:
 *        map = <file>_load_map(filename);
 *    Remember to test the return value
 *  - access the map fields:
 *        map-><map_prop_name>
 *    or the tiles:
 *        map->tiles[i][j]-><layer_name>
 *  - destroy the map:
 *        <file>_destroy_map(map)
 *
 * The code specific to seme is tagged with the word SEME
 */
#include <allegro.h>


/* SEME: include the generated header,
 * Compile the generated .c file and link with it */
#include "loader.h"


#define ESC 27

/* SEME: some data, such as the tile size and the additional data (i.e. the
 * ones asked before creating a map) are not saved in the map file
 */
#define TILE_WIDTH 32
#define TILE_HEIGHT 32

#define GROUND_DATAFILE "grounds.dat"
#define TL_DATAFILE "tl.dat"
#define TR_DATAFILE "tr.dat"
#define BL_DATAFILE "bl.dat"
#define BR_DATAFILE "br.dat"


/* Ugly global vars...
 */
static LOADER_MAP *map = NULL; /* SEME: the map structure */
static DATAFILE *grounds = NULL;
static DATAFILE *tl = NULL;
static DATAFILE *tr = NULL;
static DATAFILE *bl = NULL;
static DATAFILE *br = NULL;
/* Top-left corner of the viewed part of the map */
static int map_dx = 0;
static int map_dy = 0;


/* init_gfx:
 * Tries various screen color depth an size before giving up, returns 0 on
 * failure, 1 on sucess
 */
int init_gfx(void)
{
   static int cdepth[] = {32, 24, 16, 15, 8, 0};
   static int width[] = {/*800, 640,*/ 320, 0};
   static int height[] = {/*600, 480,*/ 240, 0};

   int c, w, h;
   for(c=0; cdepth[c]; c++) {
      for(w=0; width[w]; w++) {
         for(h=0; height[h]; h++) {
            set_color_depth(cdepth[c]);
            if(set_gfx_mode(GFX_AUTODETECT, width[w], height[h], 0, 0) == 0) {
               return 1;
            }
         }
      }
   }
   return 0;
}


/* load_map:
 *  Initialize the map, returns 0 if it failed, else 1
 */
int load_map(char *fname)
{
   /* SEME: First load the map, this should be done after 'allegro_init' */
   map = loader_load_map(fname);
   if(!map) {
      return 0;
   }
   return 1;
}


/* load_datafiles:
 *  Load the datafiles, return 0 if it failed, else 1
 */
int load_datafiles(void)
{
   static char dfname[1024];
   static char path[1024];

   get_executable_name(path, sizeof(path));

   replace_filename(dfname, path, GROUND_DATAFILE, sizeof(dfname));
   grounds = load_datafile(dfname);
   if(!grounds) {
      return 0;
   }
   replace_filename(dfname, path, TL_DATAFILE, sizeof(dfname));
   tl = load_datafile(dfname);
   if(!tl) {
      return 0;
   }
   replace_filename(dfname, path, TR_DATAFILE, sizeof(dfname));
   tr = load_datafile(dfname);
   if(!tr) {
      return 0;
   }
   replace_filename(dfname, path, BL_DATAFILE, sizeof(dfname));
   bl = load_datafile(dfname);
   if(!bl) {
      return 0;
   }
   replace_filename(dfname, path, BR_DATAFILE, sizeof(dfname));
   br = load_datafile(dfname);
   if(!br) {
      return 0;
   }
   return 1;
}


/* uninit:
 *  Cleans the data
 */
void uninit(void)
{
   /* SEME: When the map is no more used, destroy it */
   loader_destroy_map(map);
   unload_datafile(br);
   unload_datafile(bl);
   unload_datafile(tr);
   unload_datafile(tl);
   unload_datafile(grounds);
}


/* print_help:
 *  Prints an help text an wait for a keypress
 */
void print_help(void)
{
   int black = makecol(0, 0, 0);
   int white = makecol(255, 255, 255);
   int line = 0;
   int dh = text_height(font)*2;

   clear_to_color(screen, black);
   textout(screen, font, "Test for seme", 0, (line++)*dh, white);
   textout(screen, font, "", 0, (line++)*dh, white);
   textout(screen, font, "Arrows, PageUp, PageDown, Home, End to move the map",
     0, (line++)*dh, white
   );
   textout(screen, font, "Esc to quit", 0, (line++)*dh, white);
   textout(screen, font, "", 0, (line++)*dh, white);
   textout(screen, font, "... Press a key", 0, (line++)*dh, white);
   readkey();
}


/* draw_map:
 *  Draw the map on the screen
 */
void draw_map(BITMAP *dest)
{
   int i, j;
   int base_y = 0;

   clear(dest);
   text_mode(-1);
   /* SEME: the map properties can be accessed by: pointer_to_map->name_of_prop
    * You can look at the header file (in this case loader.h) to see the
    * actual C type of each property */
   textout(dest, font, map->Name, 0, base_y, makecol(255, 255, 255));
   base_y += 16;

   set_clip(dest, 0, base_y, dest->w-1, dest->h-1);
   for(i=0; i<map->width; i++) {
      for(j=0; j<map->height; j++) {
         int index;
         RLE_SPRITE *rle;

         int xposition = i*TILE_WIDTH-map_dx;
         int yposition = j*TILE_HEIGHT-map_dy+base_y;

         /* SEME: the tile values can be accessed by:
          * pointer_to_map->tiles[horz_index][vert_index].name_of_layer
          * Again, you can look at the header file (in this case loader.h) to
          * see the actual C type of each property */
         index = map->tiles[i][j].Ground;
         rle = (RLE_SPRITE*)(grounds[index].dat);
         draw_rle_sprite(dest, rle, xposition, yposition);

         index = map->tiles[i][j].Bottom_left;
         rle = (RLE_SPRITE*)(bl[index].dat);
         draw_rle_sprite(dest, rle, xposition, yposition);

         index = map->tiles[i][j].Bottom_right;
         rle = (RLE_SPRITE*)(br[index].dat);
         draw_rle_sprite(dest, rle, xposition, yposition);

         index = map->tiles[i][j].Top_left;
         rle = (RLE_SPRITE*)(tl[index].dat);
         draw_rle_sprite(dest, rle, xposition, yposition);

         index = map->tiles[i][j].Top_right;
         rle = (RLE_SPRITE*)(tr[index].dat);
         draw_rle_sprite(dest, rle, xposition, yposition);
      }
   }
   set_clip(dest, 0, 0, dest->w-1, dest->h-1);
}


/* read_inputs:
 *  Handles the "player" inputs
 */
int read_inputs(void)
{
   int key;
   int speed=4;

   if(keypressed()) {
      key = readkey();
      if((key&0xff) == ESC) {
         return 1;
      }
      switch(key>>8) {
         case KEY_UP:
         case KEY_8_PAD:
            map_dy-=speed;
         break;

         case KEY_DOWN:
         case KEY_2_PAD:
            map_dy+=speed;
         break;

         case KEY_LEFT:
         case KEY_4_PAD:
            map_dx-=speed;
         break;

         case KEY_RIGHT:
         case KEY_6_PAD:
            map_dx+=speed;
         break;

         case KEY_PGUP:
            map_dy=0;
         break;

         case KEY_PGDN:
            map_dy = map->height*TILE_HEIGHT;
         break;

         case KEY_HOME:
            map_dx=0;
         break;

         case KEY_END:
            map_dx = map->width*TILE_WIDTH;
         break;
     }
   }

   if(map_dx < 0)
      map_dx = 0;
   if(map_dx > map->width*TILE_WIDTH - SCREEN_W)
      map_dx = map->width*TILE_WIDTH - SCREEN_W;

   if(map_dy < 0)
      map_dy = 0;
   if(map_dy > map->height*TILE_HEIGHT - SCREEN_H)
      map_dy = map->height*TILE_HEIGHT - SCREEN_H;

   return 0;
}


/* main:
 *  Main function
 */
int main(int argc, char **argv)
{
   int end;
   BITMAP *buffer;

   allegro_init();

   if(argc!=2) {
      allegro_message("Usage: game <map file>\n");
      return 1;
   }

   if(install_keyboard()<0) {
      allegro_message("Can't initialize keyboard\n");
      return 1;
   }

   if(!init_gfx()) {
      allegro_message("Can't initialize graphic mode\n");
      return 1;
   }

   buffer = create_bitmap(SCREEN_W, SCREEN_H);
   if(!buffer) {
      allegro_message("Out of memory\n");
      return 1;
   }

   if(!load_datafiles()) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to load datafiles\n");
      return 1;
   }

   if(!load_map(argv[1])) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to load map file %s\n", argv[1]);
      return 1;
   }

   print_help();

   end = 0;
   while(!end) {
      draw_map(buffer);
      vsync(); /* So scrolling is a bit smoother (and slower) */
      blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      end = read_inputs();
   }

   uninit();

   return 0;
}
END_OF_MAIN();

