/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__TILES__
#define EME__TILES__

class BaseProperty;
class BaseCreator;

#include "size.h"

#ifdef EME__COMPATIBILITY
struct BITMAP;
class SelectedTiles;
class SparseTiles;
class Entry;
#include "creator.h"
#include "lutils.h"
#include <math.h>
#endif

/*
 * A buch of tiles, abstract class
 */
class Tiles {
protected:
  Tiles(BaseCreator *c, int i, int j, int w, int h);
  Tiles(BaseCreator *c);

public:
  virtual ~Tiles();

  virtual Tiles *clone() const = 0;

  Tiles &operator=(const Tiles &other);

  virtual int count() const = 0;

  const Size &size() const { return size_; }
  int begini() const { return size_.begini(); }
  int beginj() const { return size_.beginj(); }
  int endi() const { return size_.endi(); }
  int endj() const { return size_.endj(); }
  int width() const { return size_.width(); }
  int height() const { return size_.height(); }

  bool is_in(int i, int j) const { return size_.is_in(i, j); }

  const BaseCreator *creator() const { return creator_; }
  BaseCreator *creator() { return creator_; }

protected:
  Tiles(const Tiles &other);

  virtual BaseProperty *get_(int i, int j) const = 0;
  virtual void set_(int i, int j, BaseProperty *p) = 0;
  virtual void copy_(int i, int j, const BaseProperty *p) = 0;
  virtual void move_(int di, int dj) = 0;
  virtual void insert_col_(int i, int count) = 0;
  virtual void insert_row_(int j, int count) = 0;
  virtual void remove_col_(int i, int count) = 0;
  virtual void remove_row_(int j, int count) = 0;

public:
  /* Returns the property at (i, j) */
  const BaseProperty *get(int i, int j) const
    { return get_(i, j); }
  BaseProperty *get(int i, int j)
    { return get_(i, j); }

  /* Set the property at (i, j) to p.
   * Takes ownership of p */
  void set(int i, int j, BaseProperty *p)
    { set_(i, j, p); size_.update(i, j); }

  /* Set the property at (i, j) to the value of p */
  void copy(int i, int j, const BaseProperty *p)
    { copy_(i, j, p); size_.update(i, j); }

  /* Erase the property at (i, j).
   * Deletes the property if needed */
  virtual void clear(int i, int j) = 0;

  void move(int di, int dj);
  void resize(int i, int j, int w, int h);

  /* Insert count columns (rows) at i (j), moving left (down)
   * the next columns (rows) */
  void insert_col(int i, int count=1);
  void insert_row(int j, int count=1);

  /* Remove count columns (rows) at i (j), moving right (up)
   * the next columns (rows) */
  void remove_col(int i, int count=1);
  void remove_row(int j, int count=1);
  
private:
  Size size_;
  BaseCreator *creator_;

#ifdef EME__COMPATIBILITY
public:
  Tiles *Clone() const { return clone(); }

  /* Compatibility with Layer */
  void AddTile(int i, int j, BaseProperty *p) { DBG_ASSERT(!get(i, j)); set(i, j, p); }
  void SetTile(int i, int j, BaseProperty *p) { if (!get(i, j)) set(i, j, p); }

  const BaseCreator *GetCreator() const { return creator(); }
  //const BaseProperty *GetDefaultValue() const;
  const char *GetName() const { return creator()->GetName()->string(); }
  int GetWidth() const { return width(); } //
  int GetHeight() const { return height(); } //

  void DrawRect(
    BITMAP *bmp, int first_i, int first_j, int num_i, int num_j,
    int tile_w, int tile_h, float tile_odd, float scale, int l
  ) const {
    LayerUtils::DrawRect(this, bmp, first_i, first_j, num_i, num_j, tile_w, tile_h, tile_odd, scale, l);
  }
  void DrawDiamond(
    BITMAP *bmp, int first_i, int first_j, int num_i, int num_j,
    int tile_w, int tile_h, float tile_odd, float scale, int l
  ) const {
    LayerUtils::DrawRect(this, bmp, first_i, first_j, num_i, num_j, tile_w, tile_h, tile_odd, scale, l);
  }

  //void UpdateEntry(Entry *e);
  //void UpdateDefaultProperty(Entry *e);

  SelectedTiles *SelectAll() const { return LayerUtils::SelectAll(this); }
  SelectedTiles *SelectEmpty(const SelectedTiles *sel) const { return LayerUtils::SelectEmpty(this, sel); }
  SelectedTiles *SelectRect(int x1, int y1, int x2, int y2) const
  { return LayerUtils::SelectRect(this, x1, y1, x2, y2); }
  SelectedTiles *SelectCircle(int x, int y, int r2) const { return LayerUtils::SelectCircle(this, x, y, (int)sqrt((double)r2)); }
  SelectedTiles *SelectByProperty(int x, int y) const { return LayerUtils::SelectByProperty(this, x, y); }
  SelectedTiles *SelectByWand(int x, int y) const { return LayerUtils::SelectByWand(this, x, y); }

  SparseTiles *Copy(const SelectedTiles *sel) const { return LayerUtils::Copy(this, sel); }
  void Fill(const SelectedTiles *sel, const BaseProperty *value) { LayerUtils::Fill(this, sel, value); }
  void Fill(int x, int y, const BaseProperty *value) { LayerUtils::Fill(this, x, y, value); }
  void Cut(const SelectedTiles *sel) { LayerUtils::Cut(this, sel); }
  void Paste(int x, int y, const SparseTiles *tiles) { LayerUtils::Paste(this, x, y, tiles); }
  void Paste(int x, int y, const Entry *e) { LayerUtils::Paste(this, x, y, e); }

  void AppendColumn(int count) { insert_col(endi(), count); }
  void AppendRow(int count) { insert_row(endj(), count); }
  void InsertColumn(int i, int count) { insert_col(i, count); }
  void InsertRow(int j, int count) { insert_row(j, count); }
  void DeleteColumn(int i, int count) { remove_col(i, count); }
  void DeleteRow(int j, int count) { remove_row(j, count); }

  int Count() const { return count(); }

  // FIXME: add them to the FullTiles
#if 0
  const_iterator Begin() const { return begin(); }
  const_iterator Next(const_iterator t) const { return next(t); }
  const_iterator End() const { return end(); }
  iterator Begin() { return begin(); }
  iterator Next(iterator t) { return next(t); }
  iterator End() { return end(); }

  // Was BaseProperty *GetTile(const_iterator t) const;
  const BaseProperty *GetTile(const_iterator t) const { return get_tile(t); }
  BaseProperty *GetTile(iterator t) { return get_tile(t); }
  int GetJ(const_iterator t) const { return get_i(t); }
  int GetI(const_iterator t) const { return get_j(t); }
#endif

  int FirstI() const { return begini(); }
  int FirstJ() const { return beginj(); }
  int LastI() const { return endi()-1; }
  int LastJ() const { return endj()-1; }
  const BaseProperty *GetTileAt(int i, int j) const { return get(i, j); }
  BaseProperty *GetTileAt(int i, int j) { return get(i, j); }
#endif
};

#endif /* EME__TILES__ */

