/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "pool.h"


#include "debug.h"
#include "utils.h"

#include <stdlib.h>

//#define LOG_MEMORY
#ifdef LOG_MEMORY
#include <stdio.h>
#endif

MemoryPool::MemoryPool(size_t base_size, size_t nobjects):
  base_size(base_size), actual_base_size(base_size), nobjects(nobjects),
  head_of_free_list(0), num_used(0)
{
  if(base_size < sizeof(void*)) {
    actual_base_size = sizeof(void*);
  }
}


MemoryPool::~MemoryPool(void)
{
#ifdef LOG_MEMORY
  fprintf(stderr, "MemPool#%d: num_used %d\n", base_size, num_used);
#endif
}


void *MemoryPool::Alloc(size_t size)
{
  if(size != base_size) {
#ifdef LOG_MEMORY
    fprintf(stderr, "MemPool#%d: can't allocate %d bytes\n", base_size, size);
#endif
    return ::operator new(size);
  }

  void *ret = head_of_free_list;
  if(!ret) {
#ifdef LOG_MEMORY
    fprintf(stderr, "MemPool#%d: allocating new chuck\n", base_size);
#endif
    void *new_chunck = ::operator new(nobjects * actual_base_size);
    for(size_t i=0; i<nobjects-1; i++) {
      Next(new_chunck, i) = &Next(new_chunck, i+1);
    }
    Next(new_chunck, nobjects-1) = 0;
    ret = new_chunck;
  }

  head_of_free_list = Next(ret);

  num_used++;
#ifdef LOG_MEMORY
  fprintf(stderr, "MemPool#%d: %p allocated\n", base_size, ret);
#endif
  return ret;
}


void MemoryPool::Free(void *dead_object, size_t size)
{
  if(!dead_object) {
    return;
  }

  if(size != base_size) {
    ::operator delete(dead_object);
    return;
  }

  Next(dead_object) = head_of_free_list;
  head_of_free_list = dead_object;
#ifdef LOG_MEMORY
  fprintf(stderr, "MemPool#%d: %p deallocated\n", base_size, dead_object);
#endif
  num_used--;
}


