/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "help.h"

#include "debug.h"

#include "text.h"
#include "translat.h"
#include "ustring.h"
#include <altheme.h>


static TEXT *help = 0;



static COMMAND parser[] = {
  /* keyword        end      replace  prepend  append  overline underline */
  /* Begin section, 'til the end of line */
  { "\\begin",      "\n",    "",      NULL,    NULL,   NULL,    NULL },
  /* End section, 'til the end of line */
  { "\\end",        "\n",    "",      NULL,    NULL,   NULL,    NULL },
  /* Title, 'til the end of line */
  { "\\title",      "\n",    NULL,    "\n\n",  NULL,   "-",     "-"  },
  /* Paragraph beginning */
  { "\\parag",      "\\",    NULL,    "\n   ", NULL,   NULL,    NULL },
  /* Name, ends with \0 */
  { "\\name",       "\\",    NULL,    " ``",   "'' ",  NULL,    NULL },
  /* Key, ends with \0 */
  { "\\key",        "\\",    NULL,    " ",     " ",    NULL,    NULL },
  /* End name and key */
  { "\\0",          "\\",    NULL,    "",      NULL,   NULL,    NULL },
  /* \ */
  { "\\\\",         "\\",    NULL,    "\\",    NULL,   NULL,    NULL },
  /* non removeable new line */
  { "\\nl",         "\\",    NULL,    "\n",    NULL,   NULL,    NULL },
  /* non removeable space */
  { "\\_",          "\\",    NULL,    " ",     NULL,   NULL,    NULL },
  /* non removeable tab */
  { "\\tab",        "\\",    NULL,    "\t",    NULL,   NULL,    NULL },
  /* commend, 'til the end of line */
  { "\\comment",    "\n",    "",      NULL,    NULL,   NULL,    NULL },
  { "",             "\\",    NULL,    NULL,    NULL,   NULL,    NULL },
  {}
};


static char *uconvert_dup(const char *src, int src_encoding)
{
  char *ret = 0;
  if (src) {
    int size = uconvert_size(src, src_encoding, U_CURRENT);
    ret = (char*)malloc(size);
    do_uconvert(src, U_ASCII, ret, U_CURRENT, size);
  }
  return ret;
}


static void init_parser(void)
{
  static int done = 0;
  if (!done) {
    int n;
    for (n=0; parser[n].keyword; ++n) {
      parser[n].keyword = uconvert_dup(parser[n].keyword, U_ASCII);
      parser[n].end = uconvert_dup(parser[n].end, U_ASCII);
      parser[n].prepend = uconvert_dup(parser[n].prepend, U_ASCII);
      parser[n].append = uconvert_dup(parser[n].append, U_ASCII);
      parser[n].replace = uconvert_dup(parser[n].replace, U_ASCII);
      parser[n].overline = uconvert_dup(parser[n].overline, U_ASCII);
      parser[n].underline = uconvert_dup(parser[n].underline, U_ASCII);
    }
    done = 1;
  }
}

/* Returns msg */
int do_help(int msg, const char *id)
{
  if (msg==MSG_IDLE) return msg;
  if (msg==MSG_RPRESS || msg==MSG_RRELEASE) {
    return MSG_IDLE;
  }
  if (msg==MSG_CLICK && mouse_b&2) {
    while(mouse_b) {
    }
    popup_help(id);
    return MSG_IDLE;
  }
  return msg;
}


void popup_help(const char *id)
{
  if (help) {
    const char *text = find_text(help, UString(id));
    popup_text(Translation("Help").string(), text);
  }
}


void load_help(const char *fname)
{
  init_parser();
  help = load_texts(fname, parser);
}


void unload_help()
{
  if (help) {
    unload_texts(help);
    help = 0;
  }
}


