/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */

/*
 * Copied from Allegro
 */
#include "tiles.h"
#include "selected.h"
#include "prop.h"


typedef struct FLOODED_LINE      /* store segments which have been flooded */
{
   short flags;                  /* status of the segment */
   short lpos, rpos;             /* left and right ends of segment */
   short y;                      /* y coordinate of the segment */
   short next;                   /* linked list if several per line */
} FLOODED_LINE;


static int flood_count;          /* number of flooded segments */

/* Flags */
#define FLOOD_IN_USE             1
#define FLOOD_TODO_ABOVE         2
#define FLOOD_TODO_BELOW         4

#define FLOOD_LINE(c)            (((FLOODED_LINE *)_scratch_mem) + c)



static void *_scratch_mem = NULL;
static size_t _scratch_mem_size = 0;
static bool _grow_scratch_mem(size_t size)
{
  if (size>_scratch_mem_size) {
    void *tmp = realloc(_scratch_mem, size);
    if (!tmp) return false;
    _scratch_mem = tmp;
    _scratch_mem_size = size;
  }
  return true;
}



/* p2 can be 0, p1 cannot */
static bool are_different(const BaseProperty *p1, const BaseProperty *p2)
{
  return !(p2 && p1->IsEqualTo(p2));
}


static void fill_line(const Tiles *layer, int left, int right, int y, SelectedTiles *sel)
{
  for (int x=left; x<=right; ++x) {
    sel->Add(x, y);
  }
}



/* flooder:
 *  Fills a horizontal line around the specified position, and adds it
 *  to the list of drawn segments. Returns the first x coordinate after 
 *  the part of the line which it has dealt with.
 */
static int flooder(const Tiles *layer, int x, int y, const BaseProperty *src_prop, SelectedTiles *sel)
{
   FLOODED_LINE *p;
   int left = 0, right = 0;
   int c;

   /* check start pixel */
   if (are_different(src_prop, layer->get(x, y)))
      return x+1;

   /* work left from starting point */ 
   for (left=x-1; left>=0; left--)
      if (are_different(src_prop, layer->get(left, y)))
         break;

   /* work right from starting point */ 
   for (right=x+1; right<layer->endi(); right++)
      if (are_different(src_prop, layer->get(right, y)))
         break;

   left++;
   right--;

   /* draw the line */
   fill_line(layer, left, right, y, sel);

   /* store it in the list of flooded segments */
   c = y;
   p = FLOOD_LINE(c);

   if (p->flags) {
      while (p->next) {
         c = p->next;
         p = FLOOD_LINE(c);
      }

      p->next = c = flood_count++;
      if (!_grow_scratch_mem(sizeof(FLOODED_LINE) * flood_count)) {
        return layer->endi();
      }
      p = FLOOD_LINE(c);
   }

   p->flags = FLOOD_IN_USE;
   p->lpos = left;
   p->rpos = right;
   p->y = y;
   p->next = 0;

   if (y>0)
      p->flags |= FLOOD_TODO_ABOVE;

   if (y+1 < layer->endj())
      p->flags |= FLOOD_TODO_BELOW;

   return right+2;
}



/* check_flood_line:
 *  Checks a line segment, using the scratch buffer is to store a list of 
 *  segments which have already been drawn in order to minimise the required 
 *  number of tests.
 */
static bool check_flood_line(const Tiles *layer, int y, int left, int right, const BaseProperty *src_prop, SelectedTiles *sel)
{
   int c;
   FLOODED_LINE *p;
   bool ret = false;

   while (left <= right) {
      c = y;

      for (;;) {
         p = FLOOD_LINE(c);

         if ((left >= p->lpos) && (left <= p->rpos)) {
            left = p->rpos+2;
            break;
         }

         c = p->next;

         if (!c) {
            left = flooder(layer, left, y, src_prop, sel);
            ret = true;
            break; 
         }
      }
   }

   return ret;
}



/* floodfill:
 *  Select an enclosed area (starting at point x, y), returns the selected tiles
 */
SelectedTiles *FloodFill(const Tiles *layer, int x, int y)
{
   const BaseProperty *src_prop;
   int c;
   bool done;
   FLOODED_LINE *p;
   SelectedTiles *sel = new SelectedTiles();

   /* make sure we have a valid starting point */ 
   if (x<layer->begini() || x>=layer->endi() || y<layer->beginj() || y>=layer->endj())
      return sel;

   /* what prop to select? */
   src_prop = layer->get(x, y);
   if (!src_prop) {
     return sel;
   }

   /* set up the list of flooded segments */
   if (!_grow_scratch_mem(sizeof(FLOODED_LINE) * layer->endj())) {
     return sel;
   }
   flood_count = layer->endj();
   p = FLOOD_LINE(0);
   for (c=0; c<flood_count; c++) {
      p[c].flags = 0;
      p[c].lpos = SHRT_MAX;
      p[c].rpos = SHRT_MIN;
      p[c].y = y;
      p[c].next = 0;
   }

   /* start up the flood algorithm */
   flooder(layer, x, y, src_prop, sel);

   /* continue as long as there are some segments still to test */
   do {
      done = true;

      /* for each line on the screen */
      for (c=0; c<flood_count; c++) {

         p = FLOOD_LINE(c);

         /* check below the segment? */
         if (p->flags & FLOOD_TODO_BELOW) {
            p->flags &= ~FLOOD_TODO_BELOW;
            if (check_flood_line(layer, p->y+1, p->lpos, p->rpos, src_prop, sel)) {
               done = false;
               p = FLOOD_LINE(c);
            }
         }

         /* check above the segment? */
         if (p->flags & FLOOD_TODO_ABOVE) {
            p->flags &= ~FLOOD_TODO_ABOVE;
            if (check_flood_line(layer, p->y-1, p->lpos, p->rpos, src_prop, sel)) {
               done = false;
               /* special case shortcut for going backwards */
               if (c<layer->endj() && c>0)
                  c -= 2;
            }
         }
      }

   } while (!done);

   return sel;
}

