/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "editmap.h"

#include "utils.h"
#include "translat.h"
#include "command.h"
#include "map.h"
#include "cmdmap.h"
#include "cmdlayer.h"
#include "globals.h"
#include "gui.h" /* For Set/UnsetMouseBusy */
#include "help.h"

#include <altheme.h>
#include <allegro.h>


// FIXME: resizing before 0,0 -- see all the other FIXMEs


/*
 * Head
 */
#define HEAD_STRINGS(title) \
  static const Translation i18nTitle(title)

#define HEAD() \
    { altheme_shadow_box_proc, \
      0,   0, 320, 240,  0,  0,   0,     0,  0,  0, \
    0, 0, 0 }, \
    { altheme_ctext_proc, \
    160,   8, 160,  20,  0,  0,   0,     0,  0,  0, \
    i18nTitle.void_ptr(), 0, 0 }

/*
 * Tail
 */
#define TAIL_STRINGS() \
  static const Translation i18nOK("OK"); \
  static const Translation i18nCancel("Cancel");

#define TAIL() \
    { altheme_button_proc, \
      8, 212,  80,  20,  0,  0,'\r', D_EXIT, 0, 0, \
    i18nOK.void_ptr(), 0, 0 }, \
    { altheme_button_proc, \
    232, 212,  80,  20,  0,  0,   0, D_EXIT, 0, 0, \
    i18nCancel.void_ptr(), 0, 0 }, \
    {}

/*
 * Popup
 */
#define POPUP(ret, dialogs) \
  do { \
    set_dialog_color(dialogs, gui_fg_color, gui_bg_color); \
    centre_dialog(dialogs); \
    ret = popup_dialog(dialogs, -1); \
  } while(0)



/*
 * Input coord
 */
struct COORD {
  int value;
  char string[128];
  DIALOG *dialog;

  void init(int v, DIALOG *d) {
    value = v;
    dialog = d;
    uszprintf(string, 128, "%d", v);
  }
  void set(int v) {
    value = v;
    uszprintf(string, 128, "%d", v);
    dialog->flags |= D_DIRTY;
  }
  int get() const { return value; }
};

static int dist2(int x1, int y1, int x2, int y2)
{
  return (x2-x1)*(x2-x1) + (y2-y1)*(y2-y1);
}

static void clip_max(int &val, int max)
{
  val = val > max ? max : val;
}

static void clip_min(int &val, int min)
{
  val = val < min ? min : val;
}

/*
 * Data associated with a popup
 */
struct DATA {
  typedef enum {
    command_insert, command_delete, command_resize
  } command_type;

  const char *help_id;
  Map *map;
  Tiles *layer;
  DIALOG *map_dialog;
  int scale;
  COORD coord[4]; /* x, y, w, h */
  int default_coord[4];
  command_type type;
  int draw_x;
  int draw_y;

  void init(
    Map *m, int lidx, command_type t, DIALOG *md,
    int x, DIALOG *dx, int y, DIALOG *dy, int w, DIALOG *dw, int h, DIALOG *dh
  ) {
    map = m;
    layer = lidx>=0 ? map->GetLayer(lidx) : 0;
    map_dialog = md;
    default_coord[0] = x;
    default_coord[1] = y;
    default_coord[2] = w;
    default_coord[3] = h;
    coord[0].init(x, dx);
    coord[1].init(y, dy);
    coord[2].init(w, dw);
    coord[3].init(h, dh);
    type = t;
    update_scale();
  }

  void reset() {
    coord[0].set(default_coord[0]);
    coord[1].set(default_coord[1]);
    coord[2].set(default_coord[2]);
    coord[3].set(default_coord[3]);
    map_dialog->flags |= D_DIRTY;
    update_scale();
  }

  void force(int idx, int v) {
    coord[idx].set(v);
    map_dialog->flags |= D_DIRTY;
  }
  void set(int idx, int v) {
    if (v!=coord[idx].get()) {
      force(idx, v);
    }
  }
  void setx(int v) { set(0, v); }
  void sety(int v) { set(1, v); }
  void setw(int v) { set(2, v); }
  void seth(int v) { set(3, v); }

  int get(int idx) const { return coord[idx].value; }
  int getx() const { return get(0); }
  int gety() const { return get(1); }
  int getw() const { return get(2); }
  int geth() const { return get(3); }

  command_type gettype() const { return type; }
  int map_width() const { return map->GetWidth(); }
  int map_height() const { return map->GetHeight(); }

  void update_scale() {
    int mapw = map_width();
    int maph = map_height();
    if (gettype()==command_insert) {
      mapw += getw();
      maph += geth();
    }
    if (gettype()==command_resize) {
      if (mapw<getw()) mapw = getw();
      if (maph<geth()) maph = geth();
    }
    int s = 1;
    while (mapw/s>map_dialog->w || maph/s>map_dialog->h) {
      s*=2;
    }
    draw_x = (map_dialog->w-mapw/s)/2;
    draw_y = (map_dialog->h-maph/s)/2;
    scale = s;
  }

  void draw_insert(int mapx1, int mapy1, int mapx2, int mapy2, int x1, int y1, int x2, int y2)
  {
    mapx2 += getw()/scale;
    mapy2 += geth()/scale;
    rectfill(screen, mapx1, y1, mapx2, y2, gui_mg_color);
    rectfill(screen, x1, mapy1, x2, mapy2, gui_mg_color);
    circle(screen, x1, y1, 2, gui_fg_color);
    circle(screen, x2, y2, 2, gui_fg_color);
    if (layer) {
      int lx1 = mapx1 + layer->begini()/scale;
      int ly1 = mapy1 + layer->beginj()/scale;
      int lx2 = mapx1 + layer->endi()/scale + getw()/scale;
      int ly2 = mapy1 + layer->endj()/scale + geth()/scale;
      xor_mode(TRUE);
      rect(screen, lx1, ly1, lx2, ly2, gui_bg_color);
      xor_mode(FALSE);
    }
    rect(screen, mapx1, mapy1, mapx2, mapy2, gui_fg_color);
  }
  void draw_delete(int mapx1, int mapy1, int mapx2, int mapy2, int x1, int y1, int x2, int y2)
  {
    rectfill(screen, mapx1, y1, mapx2, y2, gui_mg_color);
    rectfill(screen, x1, mapy1, x2, mapy2, gui_mg_color);
    circle(screen, x1, y1, 2, gui_fg_color);
    circle(screen, x2, y2, 2, gui_fg_color);
    if (layer) {
      int lx1 = mapx1 + layer->begini()/scale;
      int ly1 = mapy1 + layer->beginj()/scale;
      int lx2 = mapx1 + layer->endi()/scale;
      int ly2 = mapy1 + layer->endj()/scale;
      xor_mode(TRUE);
      rect(screen, lx1, ly1, lx2, ly2, gui_bg_color);
      xor_mode(FALSE);
    }
    rect(screen, mapx1, mapy1, mapx2, mapy2, gui_fg_color);
  }
  void draw_resize(int mapx1, int mapy1, int mapx2, int mapy2, int x1, int y1, int x2, int y2)
  {
    int dx = getx()<0 ? getx() : 0;
    int dy = gety()<0 ? gety() : 0;
    rectfill(screen, x1-dx, y1-dy, x2-dx, y2-dy, gui_mg_color);
    circle(screen, x1-dx, y1-dy, 2, gui_fg_color);
    circle(screen, x2-dx, y2-dy, 2, gui_fg_color);
    if (layer) {
      int lx1 = mapx1 - dx + layer->begini()/scale;
      int ly1 = mapy1 - dy + layer->beginj()/scale;
      int lx2 = mapx1 - dx + layer->endi()/scale;
      int ly2 = mapy1 - dy + layer->endj()/scale;
      xor_mode(TRUE);
      rect(screen, lx1, ly1, lx2, ly2, gui_bg_color);
      xor_mode(FALSE);
    }
    rect(screen, mapx1-dx, mapy1-dy, mapx2-dx, mapy2-dy, gui_fg_color);
  }

  void draw() {
    int mapx1 = map_dialog->x+draw_x;
    int mapy1 = map_dialog->y+draw_y;
    int mapx2 = mapx1+map_width()/scale;
    int mapy2 = mapy1+map_height()/scale;
    int x1 = mapx1 + getx()/scale;
    int y1 = mapy1 + gety()/scale;
    int x2 = x1 + getw()/scale;
    int y2 = y1 + geth()/scale;
    switch (gettype()) {
      case command_insert:
        draw_insert(mapx1, mapy1, mapx2, mapy2, x1, y1, x2, y2);
        break;
      case command_delete:
        draw_delete(mapx1, mapy1, mapx2, mapy2, x1, y1, x2, y2);
        break;
      case command_resize:
        draw_resize(mapx1, mapy1, mapx2, mapy2, x1, y1, x2, y2);
        break;
    }
  }


  void clip_insert(int &x1, int &y1, int &x2, int &y2)
  {
    if (layer) {
      clip_min(x1, layer->begini());
      clip_max(x1, layer->endi());
      clip_min(y1, layer->beginj());
      clip_max(y1, layer->endj());
      clip_min(x2, layer->begini());
      clip_min(y2, layer->beginj());
    }
    else {
      clip_min(x1, 0);
      clip_max(x1, map_width());
      clip_min(y1, 0);
      clip_max(y1, map_height());
      clip_min(x2, 0);
      clip_min(y2, 0);
    }
  }
  void clip_delete(int &x1, int &y1, int &x2, int &y2)
  {
    if (layer) {
      clip_min(x1, layer->begini());
      clip_min(y1, layer->beginj());
      clip_max(x2, layer->endi());
      clip_max(y2, layer->endj());
    }
    else {
      clip_min(x1, 0);
      clip_min(y1, 0);
      clip_max(x2, map_width());
      clip_max(y2, map_height());
    }
  }
  void clip_resize(int &x1, int &y1, int &x2, int &y2)
  {
    clip_min(x1, 0); // FIXME
    clip_min(y1, 0); // FIXME
    if (layer) {
      clip_max(x1, layer->endi());
      clip_max(y1, layer->endj());
      clip_min(x2, layer->begini());
      clip_min(y2, layer->beginj());
    }
    else {
      clip_max(x1, map_width());
      clip_max(y1, map_height());
      clip_min(x2, 0);
      clip_min(y2, 0);
    }
  }

  void click(int mx, int my) {
    int userx = (mx-draw_x-map_dialog->x)*scale;
    int usery = (my-draw_y-map_dialog->y)*scale;
    int x1 = getx();
    int y1 = gety();
    int x2 = getx()+getw();
    int y2 = gety()+geth();
    if (dist2(x1, y1, userx, usery) < dist2(x2, y2, userx, usery)) {
      x1 = userx;
      y1 = usery;
    }
    else {
      x2 = userx;
      y2 = usery;
    }
    if (x2<x1) std::swap(x1, x2);
    if (y2<y1) std::swap(y1, y2);
    switch (type) {
      case command_insert:
        clip_insert(x1, y1, x2, y2);
        break;
      case command_delete:
        clip_delete(x1, y1, x2, y2);
        break;
      case command_resize:
        clip_resize(x1, y1, x2, y2);
        break;
    }
    setx(x1);
    sety(y1);
    setw(x2-x1);
    seth(y2-y1);
    update_scale();
  }
};


/*
 * Dialogs procs
 */
static int map_proc(int msg, DIALOG *d, int c)
{
  DATA *data = (DATA*)(d->dp3);

  msg = do_help(msg, data->help_id);

  switch (msg) {
    case MSG_DRAW:
      altheme_box_proc(msg, d, c);
      data->draw();
      return D_O_K;

    case MSG_CLICK:
      data->click(mouse_x, mouse_y);
      return D_O_K;

    default:
      return altheme_box_proc(msg, d, c);
  }
}

static int edit_proc(int msg, DIALOG *d, int c)
{
  DATA *data = (DATA*)(d->dp3);
  int coord = (int)(d->dp2);

  switch (msg) {
    case MSG_IDLE:
      if (d->flags & D_GOTFOCUS) {
        int v = ustrtol((char*)(d->dp), 0, 10);
        data->set(coord, v);
      }
      return D_O_K;

    case MSG_LOSTFOCUS:
    case MSG_END:
      if (d->flags & D_GOTFOCUS) {
        char *endp;
        int v = ustrtol((char*)(d->dp), &endp, 10);
        /* Make sure value is valid */
        if (v<0) { // FIXME
          data->force(coord, 0);
        }
        else if (*endp!=0) {
          data->force(coord, v);
        }
        else {
          data->set(coord, v);
        }
      }
      return altheme_edit_proc(msg, d, c);

    default:
      return altheme_edit_proc(msg, d, c);
  }
}

static int reset_proc(int msg, DIALOG *d, int c)
{
  DATA *data = (DATA*)(d->dp3);

  switch (msg) {
    case MSG_CLICK:
    case MSG_KEY:
      d->flags |= D_SELECTED;
      scare_mouse_area(d->x, d->y, d->w, d->h);
      altheme_button_proc(MSG_DRAW, d, c);
      unscare_mouse();
      while (mouse_b) {
      }
      d->flags &= ~D_SELECTED;
      data->reset();
      return D_REDRAWME;

    default:
      return altheme_button_proc(msg, d, c);
  }
}

#define TEXT(x, y, w, h, s, data) \
  { altheme_text_proc, \
    x, y, w, h, 0, 0, 0, 0, 0, 0, CAST_TO_VOID_PTR(s), 0, CAST_TO_VOID_PTR(data) }
#define EDIT(x, y, w, h, data, idx) \
  { edit_proc, \
    x, y, w, h, 0, 0, 0, 0, 4, 0, CAST_TO_VOID_PTR(&(data)->coord[idx].string), \
    CAST_TO_VOID_PTR(idx), CAST_TO_VOID_PTR(data) }
#define RESET(x, y, w, h, s, data) \
  { reset_proc, \
    x, y, w, h, 0, 0, 0, 0, 0, 0, CAST_TO_VOID_PTR(s), 0, CAST_TO_VOID_PTR(data) }
#define MAP(x, y, w, h, data) \
  { map_proc, \
    x, y, w, h, 0, 0, 0, 0, 0, 0, 0, 0, CAST_TO_VOID_PTR(data) }



typedef enum {
  DIALOG_BOX = 0,
  DIALOG_TITLE,

  DIALOG_X_TITLE, DIALOG_X_VALUE,
  DIALOG_Y_TITLE, DIALOG_Y_VALUE,
  DIALOG_W_TITLE, DIALOG_W_VALUE,
  DIALOG_H_TITLE, DIALOG_H_VALUE,

  DIALOG_RESET,

  DIALOG_MAP,

  DIALOG_OK, DIALOG_CANCEL,

  DIALOGS_COUNT
} DIALOG_INDEX;


template <typename CommandType>
void execute_command(
  Map *map, int layer_index, int x, int y, int count
)
{
  Command *c;
  if (layer_index>=0) {
    c = new CommandStack<CommandType>(map, layer_index, x, y, count);
  }
  else {
    c = new CommandStack<CommandType>(map, x, y, count);
  }
  map->GetCommandsManager()->Execute(c);
}


static void do_popup(
  const Translation &i18nTitle,
  const char *help_id,
  DATA::command_type type, Map *map, int layer_index,
  int default_x, int default_y, int default_w, int default_h
)
{
  TAIL_STRINGS();
  static const Translation i18nX("X:");
  static const Translation i18nY("Y:");
  static const Translation i18nW("W:");
  static const Translation i18nH("H:");
  static const Translation i18nReset("Reset");

  static DATA data;

  static DIALOG dialogs[] = {
    HEAD(),

    TEXT( 12,  88, 20, 20, i18nX.string(), &data),
    EDIT( 32,  88, 36, 20, &data, 0),
    TEXT( 12, 108, 20, 20, i18nY.string(), &data),
    EDIT( 32, 108, 36, 20, &data, 1),
    TEXT( 12, 128, 20, 20, i18nW.string(), &data),
    EDIT( 32, 128, 36, 20, &data, 2),
    TEXT( 12, 148, 20, 20, i18nH.string(), &data),
    EDIT( 32, 148, 36, 20, &data, 3),

    RESET( 8, 176, 56, 20, i18nReset.string(), &data),
        
    MAP(68, 28, 244, 168, &data),

    TAIL()
  };

  dialogs[DIALOG_TITLE].dp = i18nTitle.void_ptr();
  data.init(
    map, layer_index, type, dialogs+DIALOG_MAP,
    default_x, dialogs+DIALOG_X_VALUE, default_y, dialogs+DIALOG_Y_VALUE,
    default_w, dialogs+DIALOG_W_VALUE, default_h, dialogs+DIALOG_H_VALUE
  );
  data.help_id = help_id;

  int ret;
  POPUP(ret, dialogs);

  if (ret==DIALOG_OK) {
    GUI.SetMouseBusy();
    int x = data.getx();
    int y = data.gety();
    int w = data.getw();
    int h = data.geth();
    switch (type) {
      case DATA::command_insert:
        if (w>0) {
          execute_command<CommandInsertColumn>(map, layer_index, x, y, w);
        }
        if (h>0) {
          execute_command<CommandInsertRow>(map, layer_index, x, y, h);
        }
        break;

      case DATA::command_delete:
        if (w>0) {
          execute_command<CommandDeleteColumn>(map, layer_index, x, y, w);
        }
        if (h>0) {
          execute_command<CommandDeleteRow>(map, layer_index, x, y, h);
        }
        break;

      case DATA::command_resize:
        // FIXME: should move layers if adding before (0, 0) ??
        if (layer_index>=0) {
          Command *cmd = new CommandResize(map, layer_index, x, y, w, h);
          map->GetCommandsManager()->Execute(cmd);
          /*
          int dx = data.getx() < 0 ? -data.getx() : 0;
          int dy = data.gety() < 0 ? -data.gety() : 0;
          if (dx>0 || dy>0) {
            cmd = new CommandMoveLayer(map, layer_index, dx, dy);
            map->GetCommandsManager()->Execute(cmd);
          }
          */
        }
        else {
          CommandGroup *group = new CommandGroup();
          for (int l=0; l<map->GetNumLayers(); ++l) {
            Command *cmd = new CommandResize(map, l, x, y, w, h);
            group->Add(cmd);
          }
          map->GetCommandsManager()->Execute(group);
          /* FIXME: should handle clicking in the top and left parts
           * FIXME: size is bogus after this
          int dx = data.getx() < 0 ? -data.getx() : 0;
          int dy = data.gety() < 0 ? -data.gety() : 0;
          if (dx>0 || dy>0) {
            group = new CommandGroup();
            for (int l=0; l<map->GetNumLayers(); ++l) {
              Command *cmd = new CommandMoveLayer(map, l, dx, dy);
              group->Add(cmd);
            }
            map->GetCommandsManager()->Execute(group);
          }
          */
        }
        break;
    }
    GUI.UnsetMouseBusy();
  }
}



void resize_active_layer(Map *map)
{
  HEAD_STRINGS("Resize active layer");
  do_popup(
    i18nTitle, "Resize",
    DATA::command_resize, map, map->GetActiveLayerIndex(),
    0, 0, map->GetWidth(), map->GetHeight()
  );
}

void insert_in_active_layer(
  Map *map, int default_x, int default_y, int default_w, int default_h
)
{
  HEAD_STRINGS("Insert rows and columns in active layer");
  do_popup(
    i18nTitle, "Insert",
    DATA::command_insert, map, map->GetActiveLayerIndex(),
    default_x, default_y, default_w, default_h
  );
}

void delete_in_active_layer(
  Map *map, int default_x, int default_y, int default_w, int default_h
)
{
  HEAD_STRINGS("Delete rows and columns in active layer");
  do_popup(
    i18nTitle, "Delete",
    DATA::command_delete, map, map->GetActiveLayerIndex(),
    default_x, default_y, default_w, default_h
  );
}

void resize_all_layers(Map *map)
{
  HEAD_STRINGS("Resize all layers");
  do_popup(
    i18nTitle, "Resize",
    DATA::command_resize, map, -1,
    0, 0, map->GetWidth(), map->GetHeight()
  );
}

void insert_in_all_layers(
  Map *map, int default_x, int default_y, int default_w, int default_h
)
{
  HEAD_STRINGS("Insert rows and columns in all layer");
  do_popup(
    i18nTitle, "Insert",
    DATA::command_insert, map, -1,
    default_x, default_y, default_w, default_h
  );
}

void delete_in_all_layers(
  Map *map, int default_x, int default_y, int default_w, int default_h
)
{
  HEAD_STRINGS("Delete rows and columns in all layer");
  do_popup(
    i18nTitle, "Delete",
    DATA::command_delete, map, -1,
    default_x, default_y, default_w, default_h
  );
}

