/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cstring.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"
#include "alclean.h"

#include "entry.h"

#include <altheme.h>

#include <allegro.h>



/* Misc
   ---------------------------------*/
void String::Draw(const char *str, BITMAP *bmp, int x, int y, int w, int h)
{
  text_mode(gui_bg_color);
  textout_centre(bmp, font, str, x+w/2, y, gui_fg_color);
}



/* Property
   ---------------------------------*/
String::Property::~Property(void)
{
  Free(string_);
}


void String::Property::Set(const char *str)
{
  if(!string_ || ustrsize(string_)<ustrsize(str)) {
    Realloc(string_, (ustrlen(str)+1)*SIZEOF_CHARACTER);
  }
  ustrzcpy(string_, (ustrlen(str)+1)*SIZEOF_CHARACTER, str);
}


void String::Property::CopyFrom(const BaseProperty *p)
{
  const Property *ps = ::Cast<String>(p);
  Set(ps->string_);
}


int String::Property::IsEqualTo(const BaseProperty *p) const
{
  const Property *ps = ::Cast<String>(p);
  return ustrcmp(ps->string_, string_) == 0;
}


String::Property *String::Property::Clone(void) const
{
  return new Property(Get());
}



/* Creator
   ---------------------------------*/
String::Creator::Creator(StaticString *name, int max_len, const char *str):
  BaseCreator(name, new Property(str)), max_len_(max_len)
{
  DBG_SET_CREATOR_TYPE("String");
  Init(str);
}


void String::Creator::Init(const char *str)
{
  DBG_ASSERT(ustrlen(str) <= max_len_);
  Malloc(str_, (max_len_+1)*SIZEOF_CHARACTER);
  ustrzcpy(str_, (max_len_+1)*SIZEOF_CHARACTER, empty_string);
}


String::Creator::~Creator(void)
{
  Free(str_);
}


String::Creator *String::Creator::Clone(void) const
{
  return new Creator(this);
}


void String::Creator::Draw(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
  float scale, int l, int i, int j
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *ps = ::Cast<String>(p);
  /* Cannot draw it */
  String::Draw(ps->Get(), bmp, x, y, w, h);
}


int String::Creator::PrintValue(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int color
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *ps = ::Cast<String>(p);
  textprintf(
    bmp, font, x, y, color, Translation("%s: String %s"),
    GetName()->string(), ps->Get()
  );
  return 1;
}


void String::Creator::UpdateEntry(Entry *e, const BaseProperty *p) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *ps = ::Cast<String>(p);
  uszprintf(str_, (max_len_+1)*SIZEOF_CHARACTER, "%s", ps->Get());
  e->SetState(
    GetName()->string(), altheme_eedit_proc, max_len_, ustrlen(str_), str_,
    NULL, NULL, 0
  );
}


void String::Creator::UpdateProperty(BaseProperty *p, const Entry *e) const
{
  DBG_ASSERT(IsSameType(p));
  Property *ps = ::Cast<String>(p);
  const char *str_value = CAST_FROM_VOID_PTR(const char*, e->GetDPState());
  ps->Set(str_value);
}

