/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cmdselec.h"

#include "debug.h"
#include "utils.h"

#include "selected.h"
#include "map.h"
#include "lutils.h"


/* Adding tiles to the selection
   ---------------------------------*/
CommandSelectionAdd::CommandSelectionAdd(
  Map *map, const SelectedTiles *sel
): Command()
{
  PROFILE_BEGIN(sel_add_new)
  to_edit = map->GetSelectedTiles();
  to_add = sel->Clone();
  CHECK_POINTER(to_add);
  to_add->Remove(to_edit);
  PROFILE_END(sel_add_new)
}

CommandSelectionAdd::~CommandSelectionAdd(void)
{
  delete to_add;
}


void CommandSelectionAdd::UnExecute(void)
{
  to_edit->Remove(to_add);
}

void CommandSelectionAdd::Execute(void)
{
  LOG("CommandSelectionAdd::Execute\n");
  PROFILE_BEGIN(sel_add_exec)
  to_edit->Add(to_add);
  PROFILE_END(sel_add_exec)
}


int CommandSelectionAdd::IsEmpty(void) const
{
  return to_add->Empty();
}



/* Removing tiles from the selection
   ---------------------------------*/
CommandSelectionSuppress::CommandSelectionSuppress(
  Map *map, const SelectedTiles *sel
): Command()
{
  to_edit = map->GetSelectedTiles();
  to_suppress = LayerUtils::Intersection(to_edit, sel);
  CHECK_POINTER(to_suppress);
}

CommandSelectionSuppress::~CommandSelectionSuppress(void)
{
  delete to_suppress;
}


void CommandSelectionSuppress::UnExecute(void)
{
  to_edit->Add(to_suppress);
}

void CommandSelectionSuppress::Execute(void)
{
  LOG("CommandSelectionSuppress::Execute\n");
  to_edit->Remove(to_suppress);
}


int CommandSelectionSuppress::IsEmpty(void) const
{
  return to_suppress->Empty();
}



/* Replacing the selection by a new one
   ---------------------------------*/
CommandSelectionReplace::CommandSelectionReplace(
  Map *map, const SelectedTiles *sel
): Command()
{
  to_edit = map->GetSelectedTiles();
  new_selection = sel->Clone();
  CHECK_POINTER(new_selection);
  old_selection = to_edit->Clone();
  CHECK_POINTER(old_selection);
}

CommandSelectionReplace::~CommandSelectionReplace(void)
{
  delete new_selection;
  delete old_selection;
}


void CommandSelectionReplace::UnExecute(void)
{
  to_edit->Replace(old_selection);
}

void CommandSelectionReplace::Execute(void)
{
  LOG("CommandSelectionReplace::Execute\n");
  to_edit->Replace(new_selection);
}


int CommandSelectionReplace::IsEmpty(void) const
{
  return new_selection->IsSame(old_selection);
}


