/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cdataf.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include <typeinfo>

#include "entry.h"
#include "datafile.h"

#include <altheme.h>

#include <allegro.h>


/* Misc
   ---------------------------------*/
const Translation DatafileObject::i18nDatafile_not_found("Datafile not found");


int DatafileObject::display_callback(
  BITMAP *dest, int index, int x, int y, int w, DIALOG *d
)
{
  const char *fname = CAST_FROM_VOID_PTR(const char*, d->dp3);
  DBG_ASSERT(fname);
  if (index<0) {
    /* Returns the number of elements */
    return Datafiles.GetNumObjects(fname);
  }
  else {
    /* Returns the indexth element height */
    DATAFILE *data = Datafiles.Get(fname, index);
    if(data) {
      switch(data->type) {
        case DAT_BITMAP: {
          BITMAP *img = CAST_FROM_VOID_PTR(BITMAP*, data->dat);
          if(dest) {
            masked_blit(img, dest, 0, 0, x, y, w, img->h);
          }
          return img->h;
        }
        break;
        case DAT_RLE_SPRITE: {
          RLE_SPRITE *rle = CAST_FROM_VOID_PTR(RLE_SPRITE*, data->dat);
          if(dest) {
            draw_rle_sprite(dest, rle, x, y);
          }
          return rle->h;
        }
        break;
      }
    }
    /* Unknown object type, draw its name */
    const char *name = Datafiles.GetName(fname, index);
    if(!name) name = i18nDatafile_not_found;
    if(dest) {
      /* If bitmap is specified, also draw the element */
      int rtm = text_mode(-1);
      textout(dest, font, name, x, y, gui_fg_color);
      text_mode(rtm);
    }
    return text_height(font);
  }
}


void DatafileObject::Draw(
  const DATAFILE *data, int idx, const char *fname,
  BITMAP *bmp, int x, int y, int w, int h, float scale
)
{
  if(data) {
    switch(data->type) {
      case DAT_BITMAP:
      case DAT_RLE_SPRITE:
      {
        BITMAP *tmp = Datafiles.GetBitmap(fname, idx, scale);
        if(tmp) {
          masked_blit(tmp, bmp, 0, 0, x, y, tmp->w, tmp->h);
        }
      } break;
      default:
      {
        const char *name = Datafiles.GetName(fname, idx);
        if(name) {
          text_mode(gui_bg_color);
          textout(bmp, font, name, x, y, gui_fg_color);
        }
        else {
          char type_name[5]="    ";
          type_name[0] = (data->type>>24)&0xff;
          type_name[1] = (data->type>>16)&0xff;
          type_name[2] = (data->type>>8)&0xff;
          type_name[3] = data->type&0xff;
          text_mode(gui_bg_color);
          textout_centre(
            bmp, font, type_name, x+w/2, y+h/2-text_height(font)/2, gui_fg_color
          );
        }
      }
      break;
    }
  }
}



/* Property
   ---------------------------------*/
DatafileObject::Property::Property(
  StaticString *datafile, const DATAFILE *data, int index
):
  BaseProperty(), data_(data), fname_(datafile), index_(index)
{
  DBG_SET_PROP_TYPE("DatafileObject");
  fname_->Hold();
  //DBG_ASSERT(data_ == Datafiles.Get(fname_->string(), index_));
}


DatafileObject::Property::Property(StaticString *datafile, int idx):
  BaseProperty(), fname_(datafile)
{
  DBG_SET_PROP_TYPE("DatafileObject");
  fname_->Hold();
  Set(idx);
}


DatafileObject::Property::Property(StaticString *datafile, const char *name):
  BaseProperty(), fname_(datafile)
{
  DBG_SET_PROP_TYPE("DatafileObject");
  fname_->Hold();
  Set(name);
}


DatafileObject::Property *DatafileObject::Property::Clone(void) const
{
  return new Property(GetFname(), GetData(), Get());
}


DatafileObject::Property::~Property(void)
{
  fname_->Release();
}


void DatafileObject::Property::Set(int idx)
{
  data_ = Datafiles.Get(fname_->string(), idx);
  index_ = idx;
}


void DatafileObject::Property::Set(const char *name)
{
  index_ = Datafiles.GetIndex(fname_->string(), name);
  data_ = Datafiles.Get(fname_->string(), index_);
}


void DatafileObject::Property::CopyFrom(const BaseProperty *p)
{
  const Property *pd = ::Cast<DatafileObject>(p);
  index_ = pd->index_;
  data_ = pd->data_;
}


int DatafileObject::Property::IsEqualTo(const BaseProperty *p) const
{
  const Property *pd = ::Cast<DatafileObject>(p);
  return fname_->IsEqual(pd->fname_) && (pd->index_==index_);
}



/* Creator
   ---------------------------------*/
DatafileObject::Creator::Creator(
  StaticString *name, StaticString *datafile, int idx
):
  BaseCreator(name, new Property(datafile, idx)),
  fname_(datafile), drawing_cb_(0)
{
  DBG_SET_CREATOR_TYPE("DatafileObject");
  fname_->Hold();
}


DatafileObject::Creator::Creator(
  StaticString *name, StaticString *datafile, const char *objname
):
  BaseCreator(name, new Property(datafile, objname)),
  fname_(datafile), drawing_cb_(0)
{
  DBG_SET_CREATOR_TYPE("DatafileObject");
  fname_->Hold();
}


DatafileObject::Creator::Creator(
  StaticString *name, StaticString *datafile, int idx, Property *ref
):
  BaseCreator(name, ref), fname_(datafile), drawing_cb_(0)
{
  DBG_SET_CREATOR_TYPE("DatafileObject");
  fname_->Hold();
}


DatafileObject::Creator::Creator(
  StaticString *name, StaticString *datafile, const char *objname, Property *ref
):
  BaseCreator(name, ref), fname_(datafile), drawing_cb_(0)
{
  DBG_SET_CREATOR_TYPE("DatafileObject");
  fname_->Hold();
}


DatafileObject::Creator::Creator(const Creator *other):
  BaseCreator(other), fname_(other->fname_), drawing_cb_(other->drawing_cb_)
{
  DBG_SET_CREATOR_TYPE("DatafileObject");
  fname_->Hold();
}


DatafileObject::Creator::~Creator(void)
{
  fname_->Release();
}


DatafileObject::Creator *DatafileObject::Creator::Clone(void) const
{
  return new Creator(this);
}


void DatafileObject::Creator::Draw(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h, float scale,
  int l, int i, int j
) const
{
  const Property *pd = ::Cast<DatafileObject>(p);
  if(drawing_cb_) {
    drawing_cb_(pd, bmp, x, y, w, h, scale, l, i, j);
  }
  else {
    DatafileObject::Draw(
      pd->GetData(), pd->Get(), pd->GetFname()->string(), bmp, x, y, w, h, scale
    );
  }
}


int DatafileObject::Creator::PrintValue(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int color
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pd = ::Cast<DatafileObject>(p);
  const char *objname = Datafiles.GetName(pd->GetFname()->string(), pd->Get());
  if (!objname) objname = i18nDatafile_not_found;
  const char *fname = get_filename(pd->GetFname()->string());
  textprintf(
    bmp, font, x, y, color, Translation("%s: Datafile %s, %s(%d)"),
    GetName()->string(), fname, objname, pd->Get()
  );
  return 1;
}


void DatafileObject::Creator::UpdateEntry(
  Entry *e, const BaseProperty *p
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pd = ::Cast<DatafileObject>(p);
  e->SetState(
    GetName()->string(),
    altheme_elist_proc, pd->Get(), pd->Get(),
    CAST_TO_VOID_PTR(display_callback), NULL,
    CAST_TO_VOID_PTR(GetFname()->string()), 0
  );
}


void DatafileObject::Creator::UpdateProperty(
  BaseProperty *p, const Entry *e
) const
{
  DBG_ASSERT(IsSameType(p));
  Property *pd = ::Cast<DatafileObject>(p);
  pd->Set(e->GetD1State());
}


int DatafileObject::Creator::IsSameType(const BaseProperty *p) const
{
  /* Note that typeid(*current) can be different from Property,
     since it is possible to derive from DatafileObject::Creator
   */
  if(!(typeid(*GetReference())==typeid(*p))) {
    return 0;
  }
  const Property *pd = ::Cast<DatafileObject>(p);
  return fname_->IsEqual(pd->GetFname());
}

