/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "ccomposi.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "globals.h"
#include "gui.h" /* For GUI height */

#include <typeinfo>

#include "entry.h"

#include <altheme.h>

#include <allegro.h>



/* Misc
   ---------------------------------*/
typedef enum {
  BOX_ID = 0,
  BUTTON_ID = 1,
  NUM_DIALOGS
} DIALOG_ID;

#define POPUP_W  320
#define PADDING  8
#define BUTTON_H (text_height(font)+PADDING)


int Composite::popup_proc(int msg, DIALOG *d, int c)
{
  if(msg==MSG_IDLE) return D_O_K;
  if(msg==MSG_CLICK || msg==MSG_KEY) {
    DIALOG *popup = CAST_FROM_VOID_PTR(DIALOG*, d->dp2);
    while(mouse_b) {
    }
    centre_dialog(popup);
    set_dialog_color(popup, gui_fg_color, gui_bg_color);
    popup_dialog(popup, -1);
    return D_O_K;
  }
  return altheme_button_proc(msg, d, c);
}



/* Property
   ---------------------------------*/
Composite::Property::Property(int max_props):
  BaseProperty(),
  num_props_(0), max_props_(max_props)
{
  DBG_SET_PROP_TYPE("Composite");
  Calloc(props_, max_props);
}


Composite::Property::~Property(void)
{
  for (int n=0; n<num_props_; n++) {
    delete props_[n];
  }
  Free(props_);
}


Composite::Property *Composite::Property::Clone(void) const
{
  Property *p = new Property(max_props_);
  if (p) {
    for (int n=0; n<num_props_; n++) {
      if (props_[n]) {
        BaseProperty *child = props_[n]->Clone();
        CHECK_POINTER(child);
        p->Add(n, child);
      }
    }
  }
  return p;
}


void Composite::Property::Add(int index, BaseProperty *p)
{
  DBG_ASSERT(num_props_<=max_props_);
  props_[index] = p;
  if (index+1>num_props_) num_props_=index+1;
}


void Composite::Property::CopyFrom(const BaseProperty *p)
{
  const Property *pc = ::Cast<Composite>(p);
  for (int n=0; n<num_props_; n++) {
    if (props_[n] && pc->props_[n]) {
      props_[n]->CopyFrom(pc->props_[n]);
    }
  }
}


int Composite::Property::IsEqualTo(const BaseProperty *p) const
{
  int is_equal = 1;
  const Property *pc = ::Cast<Composite>(p);
  for (int n=0; n<num_props_; n++) {
    is_equal &= props_[n] && pc->props_[n] && props_[n]->IsEqualTo(pc->props_[n]);
  }
  return is_equal;
}



/* Creator
   ---------------------------------*/
Composite::Creator::Creator(StaticString *name, int max_props):
  BaseCreator(name, new Property(max_props)),
  max_props_(max_props), drawing_cb_(0)
{
  DBG_SET_CREATOR_TYPE("Composite");
  Init();
}


void Composite::Creator::Init(void)
{
  static const Translation i18nDone("Done");

  Calloc(popup_, max_props_*NUM_ENTRY_DIALOGS + (int)NUM_DIALOGS + 1);

  popup_[BOX_ID].proc = altheme_shadow_box_proc;
  popup_[BOX_ID].w = POPUP_W;
  {
    /* Make sure there is enough place to see all the entries */
    int min_height = text_height(font)*2*max_props_+BUTTON_H+4*PADDING;
    int gui_h = GUI.GetHeight();
    popup_[BOX_ID].h = min_height<gui_h ? min_height : gui_h;
  }

  popup_[BUTTON_ID].proc = altheme_button_proc;
  popup_[BUTTON_ID].dp = i18nDone.void_ptr();
  popup_[BUTTON_ID].key = '\r';
  popup_[BUTTON_ID].x = popup_[BOX_ID].x + PADDING;
  popup_[BUTTON_ID].y = popup_[BOX_ID].y + popup_[BOX_ID].h-BUTTON_H-PADDING;
  popup_[BUTTON_ID].w = popup_[BOX_ID].w - 2*PADDING;
  popup_[BUTTON_ID].h = BUTTON_H;
  popup_[BUTTON_ID].flags |= D_EXIT;
  //popup_[BUTTON_ID].key = '\r';

  if(max_props_ > 0) {
    int single_height = (popup_[BOX_ID].h - 4*PADDING - BUTTON_H) / max_props_;

    Malloc(entries_, max_props_);

    for(int n=0; n<max_props_; n++) {
      entries_[n] = new Entry(&(popup_[n*NUM_ENTRY_DIALOGS + NUM_DIALOGS]));
      CHECK_POINTER(entries_[n]);
      entries_[n]->SetSize(
        popup_[BOX_ID].x + PADDING,
        popup_[BOX_ID].y + PADDING + n*single_height,
        popup_[BOX_ID].w - 2*PADDING,
        single_height-4
      );
    }

    Calloc(creators_, max_props_);
  }
  else {
    entries_=0;
    creators_=0;
  }
}


#if 0
Composite::Creator *Composite::Creator::Clone(void) const
{
  /* Create a new Composite::Creator */
  Creator *cc = new Creator(GetName(), max_props_);

  /* Fill the new Composite::Creator with the children (i.e. creators_) */
  for(int n=0; n<max_props_; n++) {
    if (Get(n)) {
      BaseCreator *child = Get(n)->Clone();
      CHECK_POINTER(child);
      cc->Add(n, child);
    }
  }
  cc->SetDrawingCallback(GetDrawingCallback());
  return cc;
}
#else
Composite::Creator::Creator(const Creator *other):
  BaseCreator(other),
  max_props_(other->max_props_), drawing_cb_(other->drawing_cb_)
{
  Init();
  /* Fill with the children from the other creator */
  for (int n=0; n<max_props_; ++n) {
    if (other->Get(n)) {
      BaseCreator *child = other->Get(n)->Clone();
      CHECK_POINTER(child);
      Add(n, child);
    }
  }
}
Composite::Creator *Composite::Creator::Clone(void) const
{
  return new Creator(this);
}
#endif


Composite::Creator::~Creator(void)
{
  for(int n=0; n<max_props_; n++) {
    delete entries_[n];
    delete creators_[n];
  }
  if(entries_) Free(entries_);
  if(creators_) Free(creators_);
  Free(popup_);
}


int Composite::Creator::Add(int idx, BaseCreator *c)
{
  if(creators_[idx]) return 0;
  creators_[idx]=c;
  /* Clone prop so it can be deleted by creator (child) and creator (this) */
  BaseProperty *p = c->Create();
  CHECK_POINTER(p);
  ReferenceProperty()->Add(idx, p);
  return 1;
}


void Composite::Creator::Draw(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h, float scale,
  int l, int i, int j
) const
{
  const Property *pc = ::Cast<Composite>(p);
  if(drawing_cb_) {
    drawing_cb_(pc, bmp, x, y, w, h, scale, l, i, j);
  }
  else {
    for (int n=0; n<max_props_; n++) {
      if (pc->Get(n)) {
        creators_[n]->Draw(pc->Get(n), bmp, x, y, w, h, scale, l, i, j);
      }
    }
  }
}


int Composite::Creator::PrintValue(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int color
) const
{
  DBG_ASSERT(IsSameType(p));
  int lines = 2;
  const Property *pc = ::Cast<Composite>(p);
  textprintf(
    bmp, font, x, y, color,
    Translation("%s: Composite {"), GetName()->string()
  );
  for(int n=0; n<pc->Count(); n++) {
    if (pc->Get(n)) {
      lines += creators_[n]->PrintValue(
        pc->Get(n), bmp,
        x+PRINT_VALUE_HINDENT, y+(lines-1)*PRINT_VALUE_VINDENT, color
      );
    }
    else {
      textprintf(
        bmp, font, x+PRINT_VALUE_VINDENT, y+(lines-1)*PRINT_VALUE_VINDENT, color,
        Translation("<nil>")
      );
      lines++;
    }
  }
  textprintf(
    bmp, font, x, y+(lines-1)*PRINT_VALUE_VINDENT, color, Translation("}")
  );
  return lines;
}


void Composite::Creator::UpdateEntry(Entry *e, const BaseProperty *p) const
{
  static const Translation i18nMore("More...");

  DBG_ASSERT(IsSameType(p));
  const Property *pc = ::Cast<Composite>(p);
  if(max_props_>0) {
    e->SetState(
      GetName()->string(), popup_proc, 0, 0,
      i18nMore.void_ptr(), popup_,
      entries_, 0
    );
    for(int n=0; n<pc->Count(); n++) {
      if (pc->Get(n)) {
        creators_[n]->UpdateEntry(entries_[n], pc->Get(n));
      }
      else {
        entries_[n]->SetDefaultState();
      }
    }
  }
  else {
    e->SetDefaultState();
  }
}


void Composite::Creator::UpdateProperty(BaseProperty *p, const Entry *e) const
{
  DBG_ASSERT(IsSameType(p));
  Property *pc = ::Cast<Composite>(p);
  for(int n=0; n<pc->Count(); n++) {
    if (pc->Get(n)) {
      creators_[n]->UpdateProperty(pc->Get(n), entries_[n]);
    }
  }
}


int Composite::Creator::IsSameType(const BaseProperty *p) const
{
  /* Note that typeid(*current) can be different from Composite::Property,
     since it is possible to derive from Composite::Creator
   */
  int is_same = (typeid(*GetReference()) == typeid(*p));
  if(is_same) {
    const Property *pc = ::Cast<Composite>(p);
    const Property *def = ReferenceProperty();
    is_same = is_same && (def->Max()==pc->Max());
    is_same = is_same && (def->Count()==pc->Count());
    if(is_same) {
      for(int n=0; n<def->Count(); n++) {
        if (pc->Get(n)) {
          is_same = is_same && creators_[n]->IsSameType(pc->Get(n));
        }
      }
    }
  }
  return is_same;
}

