/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__COLOR_TYPE__
#define EME__COLOR_TYPE__
/*----------------------------------------------------------------------------
  Color Property

  - Types
  - Casts
  - Color packing and unpacking
  - Predefined colors
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

class Entry;


struct Color {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;
  typedef int Base;
  typedef void Param;


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Misc
     ---------------------------------*/
  static void Draw(int r,int g,int b, BITMAP *bmp, int x, int y, int w, int h);
  static void Draw(int color, BITMAP *bmp, int x, int y, int w, int h);


  /* Color packing and unpacking
     ---------------------------------*/
  static inline int pack(int r, int g, int b)
    { return (r&0xff) + ((g&0xff)<<8) + ((b&0xff)<<16); }
  static inline int red(int packed)
    { return packed&0xff; }
  static inline int green(int packed)
    { return (packed>>8)&0xff; }
  static inline int blue(int packed)
    { return (packed>>16)&0xff; }


  /* Predefined colors
     ---------------------------------*/
  static const int white;
  static const int grey;
  static const int black;
  static const int majik;


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(int r, int g, int b);
    Property(int packed_color=0);
    ~Property(void);
    Property *Clone(void) const;

    /* Accessors
       --------------------*/
    void Set(int r, int g, int b);
    void Set(int packed_color)
      {Set(red(packed_color), green(packed_color), blue(packed_color));}

    int GetRed(void) const
      {return red_;}
    int GetGreen(void) const
      {return green_;}
    int GetBlue(void) const
      {return blue_;}
    int GetColor(void) const /* Returns color in current Allegro format */
      {return color_;}
    int Get(void) const /* Returns a *packed* color */
      {return pack(GetRed(), GetGreen(), GetBlue());}

    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);
    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    int red_, green_, blue_; /* 0-255 values */
    int color_;              /* Color in the display format */
  }; /* class Color::Property */


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(StaticString *name, int r, int g, int b);
    Creator(StaticString *name, int packed_color);
    Creator *Clone(void) const;
    ~Creator(void);

    /* Properties
       --------------------*/
    Property *Create(int packed_color) const
      { return new Property(packed_color); }
    Property *Create(int r, int g, int b) const
      { return new Property(r, g, b); }

    virtual void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
      float scale, int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;


    /* Entry handlers
       --------------------*/
    void UpdateEntry(Entry *e, const BaseProperty *p) const;
    void UpdateProperty(BaseProperty *p, const Entry *e) const;

  protected:
    Creator(StaticString *name, BaseProperty *p):
      BaseCreator(name, p)
      {DBG_SET_CREATOR_TYPE("Color");}
    Creator(const Creator *other): BaseCreator(other)
      {DBG_SET_CREATOR_TYPE("Color");}

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<Color>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<Color>(GetVariableReference()); }
  }; /* class Color::Creator */
  

  /* Private
     ---------------------------------*/
private:
}; /* struct Color */

#ifdef DEBUG
typedef Color::Creator ColorCreator;
typedef Color::Property ColorProperty;
#endif

#endif /* EME__COLOR_TYPE__ */

