/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "semenew.h"

#include "sememngr.h"
#include "semeprop.h"
#include "sememap.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include <altheme.h>

#include <stdio.h>
#include <string.h>
#include <allegro.h>


#include "semetile.h"
#include "semeaddp.h"
#include "semeaddl.h"
#include "semedlg.h"


SemeMap *seme_map;




/*------------------------------------------------------------------------------
  Main Dialog
------------------------------------------------------------------------------*/

/* Indices to the dialog array 'get_map_dialog()' */
typedef enum {
  NEW_MAP_BOX,
  NEW_MAP_TITLE,

  /* The tiles shape, size, etc. */
  NEW_MAP_TILES,

  NEW_MAP_COMPRESS,
  NEW_MAP_TEXT2,

  /* The map properties */
  NEW_MAP_TEXT3,
  NEW_MAP_PROPS,         /* d->d1: index of the selected map property */
  NEW_MAP_PROPS_ADD,
  NEW_MAP_PROPS_REMOVE,

  /* The layers */
  NEW_MAP_TEXT4,
  NEW_MAP_LAYERS,        /* d->d1: index of the selected layer */
  NEW_MAP_LAYERS_ADD,
  NEW_MAP_LAYERS_REMOVE,

  NEW_MAP_OK,
  NEW_MAP_CANCEL,
  NUM_NEW_MAP_DIALOGS
} NEW_MAP_DIALOG;



/* Strings to enter the map size */
#define SIZE_LEN 10
char width_str[(SIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;
char height_str[(SIZE_LEN+1)*SIZEOF_CHARACTER]=EMPTY_STRING;

char *seme_props_cb(int index, int *listsize, DIALOG * /*d*/);
int seme_props_list(int msg, DIALOG *d, int c);
int seme_props_add(int msg, DIALOG *d, int c);
int seme_props_remove(int msg, DIALOG *d, int c);
char *seme_layers_cb(int index, int *listsize, DIALOG * /*d*/);
int seme_layers_list(int msg, DIALOG *d, int c);
int seme_layers_add(int msg, DIALOG *d, int c);
int seme_layers_remove(int msg, DIALOG *d, int c);

static const Translation i18nTiles("Tiles");
static const Translation i18nCompress("Compress");
static const Translation i18nProps("Map properties");
static const Translation i18nLayers("Layers");
static const Translation i18nAdd("Add");
static const Translation i18nRemove("Remove");
static const Translation i18nOK("OK");
static const Translation i18nCancel("Cancel");


static DIALOG *get_map_dialog()
{
  static DIALOG dlg[NUM_NEW_MAP_DIALOGS+1];
  static bool initialized = false;
  if (!initialized) {
    int i=0;

    box(dlg[i++], 0, 0, 320, 200);
    title(dlg[i++], 160, 8, 160, 20, 0);

    dialog(dlg[i++], 8, 28, 150, 20, TILE_BUTTON_HELP, (void*)(i18nTiles.string()), seme_tile_proc);

    check(dlg[i++], 162, 28, 16, 16, COMPRESS_HELP, D_SELECTED);
    text(dlg[i++], 182, 33, 130, 20, COMPRESS_HELP, i18nCompress);

    text(dlg[i++],    8,  56, 148, 20, PROP_LIST_HELP, i18nProps);
    dialog(dlg[i++],  8,  76, 148, 60, PROP_LIST_HELP, (void*)seme_props_cb, seme_props_list);
    dialog(dlg[i++],  8, 144,  74, 20, PROP_LIST_HELP, (void*)(i18nAdd.string()), seme_props_add);
    dialog(dlg[i++], 82, 144,  74, 20, PROP_LIST_HELP, (void*)(i18nRemove.string()), seme_props_remove);

    text(dlg[i++],   164,  56, 148, 20, LAYER_LIST_HELP, i18nLayers);
    dialog(dlg[i++], 164,  76, 148, 60, LAYER_LIST_HELP, (void*)seme_layers_cb, seme_layers_list);
    dialog(dlg[i++], 164, 144,  74, 20, LAYER_LIST_HELP, (void*)(i18nAdd.string()), seme_layers_add);
    dialog(dlg[i++], 238, 144,  74, 20, LAYER_LIST_HELP, (void*)(i18nRemove.string()), seme_layers_remove);

    button(dlg[i++],   8, 172, 80, 20, '\r', D_EXIT, i18nOK);
    button(dlg[i++], 232, 172, 80, 20,    0, D_EXIT, i18nCancel);

    end(dlg[i]);

    initialized = true;
  }
  return dlg;
}



int current_prop_index()
{
  return get_map_dialog()[NEW_MAP_PROPS].d1;
}


int current_layer_index()
{
  return get_map_dialog()[NEW_MAP_LAYERS].d1;
}



/* seme_props_cb:
 * Callback for getting the map property names, to display them in a list
 */
char *seme_props_cb(int index, int *listsize, DIALOG * /*d*/)
{
  if(index>=0) {
    return (char*)seme_map->GetProperty(index)->GetName()->string();
  }
  else {
    *listsize = seme_map->GetPropertyCount();
    return 0;
  }
}


/* seme_props_list:
 * List of the map properties, double click edit the selected map property
 */
int seme_props_list(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_DCLICK) {
    while(mouse_b) {
    }
    SemeProperty *to_edit = seme_map->GetProperty(current_prop_index());
    const SemeProperty *ret = popup_add_prop(to_edit);
    if (ret) {
      SemeProperty *p = SemePropertyManager::Create(get_prop_type());
      CHECK_POINTER(p);
      p->CopyFrom(ret);
      SemeProperty *old = seme_map->ReplaceProperty(current_prop_index(), p);
      delete old;
    }
  }
  return altheme_list_proc(msg, d, c);
}


/* seme_props_add:
 * Calls a popup to add a map property
 */
int seme_props_add(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_LPRESS || msg == MSG_KEY) {
    seme_button_proc(msg, d, c);
    scare_mouse_area(d->x, d->y, d->w, d->h);
    seme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while(mouse_b) {
    }
    if(IS_IN(d, mouse_x, mouse_y) || msg == MSG_KEY) {
      const SemeProperty *ret = popup_add_prop(0);
      if (ret) {
        SemeProperty *p = SemePropertyManager::Create(get_prop_type());
        CHECK_POINTER(p);
        p->CopyFrom(ret);
        seme_map->AddProperty(p);
      }
    }
    d->flags &= ~D_SELECTED;
    return D_REDRAW;
  }
  return seme_button_proc(msg, d, c);
}


/* seme_props_remove:
 * Remove the selected map property
 */
int seme_props_remove(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_LPRESS || msg == MSG_KEY) {
    seme_button_proc(msg, d, c);
    scare_mouse_area(d->x, d->y, d->w, d->h);
    seme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while(mouse_b) {
    }
    if(IS_IN(d, mouse_x, mouse_y) || msg == MSG_KEY) {
      SemeProperty *old = seme_map->RemoveProperty(current_layer_index());
      delete old;
    }
    d->flags &= ~D_SELECTED;
    return D_REDRAW;
  }
  return seme_button_proc(msg, d, c);
}


/* seme_layers_cb:
 * Callback for getting the layers names, to display them in a list
 */
char *seme_layers_cb(int index, int *listsize, DIALOG * /*d*/)
{
  if(index>=0) {
    return (char*)seme_map->GetLayer(index)->GetName()->string();
  }
  else {
    *listsize = seme_map->GetLayerCount();
    return 0;
  }
}


/* seme_layers_list:
 * List of the layers, double click edit the selected layer
 */
int seme_layers_list(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_DCLICK) {
    while(mouse_b) {
    }
    SemeProperty *to_edit = seme_map->GetLayer(current_layer_index());
    const SemeProperty *ret = popup_add_layer(to_edit);
    if (ret) {
      SemeProperty *p = SemePropertyManager::Create(get_layer_type());
      CHECK_POINTER(p);
      p->CopyFrom(ret);
      SemeProperty *old = seme_map->ReplaceLayer(current_layer_index(), p, 1);
      delete old;
    }
  }
  return altheme_list_proc(msg, d, c);
}


/* seme_layers_add:
 * Calls a popup to add a layer
 */
int seme_layers_add(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_LPRESS || msg == MSG_KEY) {
    seme_button_proc(msg, d, c);
    scare_mouse_area(d->x, d->y, d->w, d->h);
    seme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while(mouse_b) {
    }
    if(IS_IN(d, mouse_x, mouse_y) || msg == MSG_KEY) {
      const SemeProperty *ret = popup_add_layer(0);
      if (ret) {
        SemeProperty *p = SemePropertyManager::Create(get_layer_type());
        CHECK_POINTER(p);
        p->CopyFrom(ret);
        seme_map->AddLayer(p, 1);
      }
    }
    d->flags &= ~D_SELECTED;
    return D_REDRAW;
  }
  return seme_button_proc(msg, d, c);
}


/* seme_layers_remove:
 * Remove the selected layer
 */
int seme_layers_remove(int msg, DIALOG *d, int c)
{
  HANDLE_HELP(msg, d, c);
  if(msg == MSG_LPRESS || msg == MSG_KEY) {
    seme_button_proc(msg, d, c);
    scare_mouse_area(d->x, d->y, d->w, d->h);
    seme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while(mouse_b) {
    }
    if(IS_IN(d, mouse_x, mouse_y) || msg == MSG_KEY) {
      SemeProperty *old = seme_map->RemoveLayer(current_layer_index());
      delete old;
    }
    d->flags &= ~D_SELECTED;
    return D_REDRAW;
  }
  return seme_button_proc(msg, d, c);
}




/* seme_popup_new:
 * Popup to create a new type of map
 */
SemeMap *seme_popup_edit(SemeMap *previous)
{
  Translation i18nMapFormat("Map format: ");
  int ret;
  char *title;

  if(previous) {
    seme_map = new SemeMap(previous);
    CHECK_POINTER(seme_map);
    const char *fname = seme_map->GetFormatFname();
    int size = ustrsizez(fname) + ustrsizez(i18nMapFormat);
    title = (char*) malloc(size);
    CHECK_POINTER(title);
    ustrcpy(title, i18nMapFormat);

    int dw = 0;
    int len =
      text_length(font, fname) + text_length(font, i18nMapFormat) +
      text_length(font, SpaceString()) * 2;
    if(len > get_map_dialog()[NEW_MAP_BOX].w) {
      dw = (len-get_map_dialog()[NEW_MAP_BOX].w) / text_length(font, SpaceString());
    }
    ustrcat(title, fname+dw);

    get_map_dialog()[NEW_MAP_TITLE].dp = (void*) title;
  }
  else {
    seme_map = new SemeMap();
    CHECK_POINTER(seme_map);
    title = ustrdup(i18nMapFormat);
    CHECK_POINTER(title);
    get_map_dialog()[NEW_MAP_TITLE].dp = (void*) title;
  }
  if(seme_map->IsCompressed()) {
    get_map_dialog()[NEW_MAP_COMPRESS].flags |= D_SELECTED;
  }
  else {
    get_map_dialog()[NEW_MAP_COMPRESS].flags &= ~D_SELECTED;
  }

  set_dialog_color(get_map_dialog(), gui_fg_color, gui_bg_color);
  centre_dialog(get_map_dialog());
  ret = popup_dialog(get_map_dialog(), -1);

  free(title);

  if(ret == NEW_MAP_OK) {
    if(get_map_dialog()[NEW_MAP_COMPRESS].flags&D_SELECTED) {
      seme_map->Compress(1);
    }
    else {
      seme_map->Compress(0);
    }
    return seme_map;
  }
  else {
    return 0;
  }
}

