/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "semedata.h"

#include "semeprop.h"
#include "semefile.h"
#include "semevrsn.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include <string.h>
#include <allegro.h>

#include "cdataf.h"
typedef DatafileObject::Creator DatafileCreator;
typedef DatafileObject::Property DatafileProperty;


static const UString &usDatafile_object()
{
  static const UString str("Datafile object");
  return str;
}
SEME_REGISTER(SemeDatafile, usDatafile_object(), SemeDatafile::Create);


/* Constructor/destructor
   ---------------------------------*/
SemeProperty *SemeDatafile::Create(void)
{
  SemeProperty *ret = new SemeDatafile();
  CHECK_POINTER(ret);
  return ret;
}


SemeDatafile::SemeDatafile(void):
  SemeProperty(), fname(ustrdup(UNSPECIFIED_USTR)),
  ascii_fname(strdup(UNSPECIFIED_STR))
{
  if(!fname) ERROR_NOMEM();
  if(!ascii_fname) ERROR_NOMEM();
  /* Can't compute it now, Allegro is not initialized */
  num_objs = 0;
}


SemeDatafile::~SemeDatafile(void)
{
  free(fname);
  free(ascii_fname);
}


void SemeDatafile::SubCopyFrom(const SemeProperty *p)
{
  const SemeDatafile *d = cast(p);
  free(fname);
  fname = ustrdup(d->fname);
  if(!fname) ERROR_NOMEM();
  free(ascii_fname);
  ascii_fname = strdup(d->ascii_fname);
  if(!ascii_fname) ERROR_NOMEM();
  num_objs = ComputeNumObjs();
}


/* Additional data
   ---------------------------------*/
int SemeDatafile::AskAdditionalData(void)
{
  const char *p = AdditionalData();
  if(p && exists(p)) {
    free(fname);
    fname = ustrdup(p);
    if(!fname) ERROR_NOMEM();
    free(ascii_fname);
    ascii_fname = uconvert_dup(p, U_CURRENT, U_ASCII);
    if(!ascii_fname) ERROR_NOMEM();
    num_objs = ComputeNumObjs();
    return 1;
  }
  else {
    return 0;
  }
}


const char *SemeDatafile::AdditionalData(void)
{
  static char path[STRING_PATH_BASE_LENGTH*SIZEOF_CHARACTER];
  ustrzcpy(path, sizeof(path), fname);
  usetat(path, ustrlen(path), 0);
  const char *title = CreateTitle(EmptyString(), Translation(" : datafile"));
  if(file_select(title, path, 0) == TRUE) {
    return path;
  }
  else {
    return 0;
  }
}


const char *SemeDatafile::GetDataDescription(void)
{
  if(fname && exists(fname)) {
    return PrintDataDesc(Translation("Filename: %s"), fname);
  }
  else {
    return PrintDataDesc(Translation("Filename: %s"), UNSPECIFIED_USTR);
  }
}


/* Loading/saving format file
   ---------------------------------*/
int SemeDatafile::LoadFormat(FILE *f, int num_indent)
{
  char *str;
  str = ReadField(f, num_indent, "file_name");
  if(!str) {
    return 0;
  }

  free(ascii_fname);
  ascii_fname = derelativize(str);
  if(!ascii_fname) ERROR_NOMEM();

  free(fname);
  fname = uconvert_dup(ascii_fname, U_ASCII, U_CURRENT);
  if(!fname) ERROR_NOMEM();

  num_objs = ComputeNumObjs();
  free(str);
  return 1;
}


int SemeDatafile::SaveFormat(FILE *f, int num_indent) const
{
  char *rel_fname = relativize(ascii_fname);
  Field(f, num_indent, "file_name", "%s", rel_fname);
  free(rel_fname);
  return 1;
}


/* Creating/loading/saving map
   ---------------------------------*/
BaseCreator *SemeDatafile::CreateCreator(void)
{
  if(!fname || !exists(fname)) {
    if(!AskAdditionalData()) {
      return 0;
    }
  }
  BaseCreator *ret = new DatafileCreator(
    GetName(), new StaticString(fname), SEME_DEFAULT_DATA //??
  );
  return ret;
}


BaseCreator *SemeDatafile::LoadProperty(PACKFILE *pf, int version)
{
  int val = pack_mgetl(pf);
  if(val<0 || val>=num_objs) {
    return 0;
  }
  const char *f = fname;
  if(!f || !exists(f)) {
    f=AdditionalData();
    if(!f || !exists(f)) {
      return 0;
    }
  }
  DatafileCreator *creator=new DatafileCreator(GetName(), new StaticString(f), val); //??
  return creator;
}


BaseProperty *SemeDatafile::LoadTile(
  PACKFILE *pf, int version, const BaseCreator *creator
) const
{
  const DatafileCreator *dc = Cast<DatafileObject>(creator);
  int val = pack_mgetl(pf);
  if(val<0 || val>=num_objs) {
    return 0;
  }
  return dc->Create(val);
}


int SemeDatafile::Save(PACKFILE *pf, int version, const BaseProperty *p) const
{
  const DatafileProperty *pd = Cast<DatafileObject>(p);
  return pack_mputl(pd->Get(), pf) != EOF;
}


/* Saving C code
   ---------------------------------*/
int SemeDatafile::SaveCStructField(FILE *f, int num_indent) const
{
  char *str = CleanForC(GetName()->string());
  if(!str) {
    return 0;
  }
  Print(
    f, num_indent, "int %s; /* %s */", str,
    SemePropertyManager::GetAsciiName(GetType())
  );
  free(str);
  return 1;
}


int SemeDatafile::SaveCCode(
  FILE *f, int num_indent, const char *leading, const char * /*cleaning*/
) const
{
  char *str = CleanForC(GetName()->string());
  if(!str) {
    return 0;
  }
  Print(f, num_indent, "%s%s = pack_mgetl(file);", leading, str);
  free(str);
  return 1;
}


int SemeDatafile::SaveCCleanCode(
  FILE * /*f*/, int /*num_indent*/, const char * /*leading*/
) const
{
  return 1;
}


/* Internal function
   ---------------------------------*/
int SemeDatafile::ComputeNumObjs(void)
{
  DATAFILE *dat = load_datafile(fname);
  if(!dat) {
    return 0;
  }
  int n;
  for(n=0; dat[n].type!=DAT_END; n++) {
  }
  unload_datafile(dat);
  return n;
}

