/* ieme - a simple bitmap editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include <stdio.h>
#include <string.h>
#include <allegro.h>

#include <allegro/internal/aintern.h>

/*
 * FIXME: when reading a true color bitmap, should return 332 palette
 * Three functions are exported:
 *
 * BITMAP *load_xpm(const char *fname, RGB *pal)
 *  Reads an XPM from a file, suitable for 'register_bitmap_file_type'.
 *  Returns NULL on error. 'pal' can be NULL.
 *  Note; can handle only pixels coded on one character
 *  Note: in 8 bits, assumes color None (transparent) is the first one.
 *  Note: if the file format is bogus, may crash.
 *
 * BITMAP *convert_xpm(char *lines[], RGB *pal)
 *  Converts an included XPM to a BITMAP.
 *  Returns NULL on error. 'pal' can be NULL.
 *  Note; can handle only pixels coded on one character
 *  Note: in 8 bits, assumes color None (transparent) is the first one.
 *  Note: if the file format is bogus, may crash.
 *
 * int save_xpm(const char *fname, BITMAP *bmp, const RGB *pal)
 *  Saves a BITMAP to a file, suitable for 'register_bitmap_file_type'.
 *  Returns non-zero on error, 'pal' can be NULL.
 *  Note: can handle at most 94 colors.
 *
 * General Notes:
 *  If compiled with -DTEST_MAIN, produces a test executable.
 *  Never tested in 8 bits
 */



/* COLOR:
 *  Data structure to keep the link between character and color.
 *  An array of COLOR ends with { c=0; v=-1 }
 */
typedef struct COLOR {
  char c;         /* Character */
  unsigned int v; /* Value (color) */
} COLOR;



/* readline_:
 *  Reads a line in an internal buffer (not reentrant!)
 */
static const char *readline_(FILE *f)
{
  static int size = 256;
  static char *line = NULL;
  
  if (!line) line = (char*)malloc(size);

  {
    char *s = fgets(line, size, f);
    if (!s) {
      return 0;
    }
    else {
      int len = strlen(s);
      while (s[len-1]!='\n') {
        line = (char*)realloc(line, size*2);
        s = fgets(s+len, size, f);
        size *= 2;
      }
    }
  }

  return line;
}



/* find_color_:
 *  Returns the color identified by 'c' or -1 if not found
 */
static int find_color_(COLOR *colors, char c)
{
  while (colors->c && colors->c!=c) {
    colors++;
  }
  return colors->v;
}



/* find_character_:
 *  Returns the character associated with the color 'v'
 *  or 0 if not found
 */
static char find_character_(COLOR *colors, unsigned int v)
{
  while (colors->c && colors->v!=v) {
    colors++;
  }
  return colors->c;
}



/* load_colors_:
 *  Returns a newly allocated COLOR array, filled with data from 'from'
 *  terminated by character 0
 */
static COLOR *load_colors_(void *from, int ncols, int bpp,
    int begin, const char *(*get)(void *from, int n))
{
  int n;
  COLOR *colors = (COLOR*)malloc((ncols+1) * sizeof(COLOR));
  int majik = 0;

  if (!colors) {
    return NULL;
  }

  switch (bpp) {
    case 8: majik = MASK_COLOR_8; break;
    case 15: majik = MASK_COLOR_15; break;
    case 16: majik = MASK_COLOR_16; break;
    case 24: majik = MASK_COLOR_24; break;
    case 32: majik = makeacol_depth(32, 255, 0, 255, 0); break;
  }

  for (n=0; n<ncols; ++n) {
    unsigned int col;
    char c;
    const char *s = get(from, begin+n);
    int ret = sscanf(s, "%c\tc #%x", &c, &col);
    if (ret == 2) { /* c #xxxxxx */
      colors[n].c = c;
      colors[n].v =
        makeacol_depth(bpp, (col>>16)&0xff, (col>>8)&0xff, col&0xff, 255);
    }
    else if (ret == 1) { /* c None */
      colors[n].c = c;
      colors[n].v = majik;
    }
    else {
      free(colors);
      return NULL;
    }
  }

  colors[ncols].c = 0;
  colors[ncols].v = (unsigned int)-1;

  return colors;
}



/* get_from_file_:
 *  Helper function to get a line from an XPM file
 */
static const char *get_from_file_(void *from, int n)
{
  FILE *f = (FILE*)from;
  return readline_(f)+1; /* Eat the first " */
}



/* get_from_strings_:
 *  Helper function to get a line from an included XPM
 */
static const char *get_from_strings_(void *from, int n)
{
  char **strings = (char**)from;
  return strings[n];
}



/* insert_color_:
 *  Insert a color at the end of a COLOR array
 */
static COLOR *insert_color_(COLOR *colors, int count, char c, unsigned int v)
{
  colors = (COLOR*)realloc(colors, sizeof(COLOR)*(count+2));
  colors[count].c = c;
  colors[count].v = v;
  colors[count+1].c = 0;
  colors[count+1].v = (unsigned int)-1;
  return colors;
}



/* read_colors_:
 *  Returns a newly allocated COLOR array,
 *  filled with the colors from the bitmap,
 *  terminated by character 0
 */
static COLOR *read_colors_(BITMAP *bmp, int *color_count)
{
  static char characters[] =
    " .,-+=*@#$%&`~!^1234567890abcdefghijklmnopqrstuvwxyz"
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ()[]{}\\|/:;'\"<>?";
  int i, j;
  unsigned int majik = bitmap_color_depth(bmp)==32 ?
    makeacol_depth(32, 0, 0, 0, 0) :
    bitmap_mask_color(bmp);
  unsigned int count = 0;
  COLOR *colors = NULL;

  colors = insert_color_(colors, count, characters[count], majik);
  count++;

  for (j=0; j<bmp->h; ++j) {
    for (i=0; i<bmp->h; ++i) {
      int v = getpixel(bmp, i, j);
      int c = find_character_(colors, v);
      if (!c) {
        c = characters[count];
        colors = insert_color_(colors, count, c, v);
        count++;
        if (count>sizeof(characters)) {
          free(colors);
          return NULL;
        }
      }
    }
  }
  *color_count = count;
  return colors;
}



/* fill_bitmap_:
 *  Fills the bitmap with the data from the file
 */
static void fill_bitmap_(void *from, BITMAP *bmp, COLOR *colors,
    int begin, const char *(*get)(void *from, int n))
{
  int j;
  for (j=0; j<bmp->h; ++j) {
    int i;
    const char *s = get(from, begin+j);
    for (i=0; i<bmp->w; ++i) {
      putpixel(bmp, i, j, find_color_(colors, *s));
      s++;
    }
  }
}



/* xpm2bitmap:
 *  Converts an XPM to a bitmap, getting its data from 'get'
 */
static BITMAP *xpm2bitmap(void *from, RGB *pal,
    const char *(*get)(void *from, int n))
{
  BITMAP *bmp = NULL;
  COLOR *colors = NULL;
  const char *s;
  int w, h;
  int ncols, colsize;
  int ret;
  int bpp = _color_load_depth(32, FALSE);

  /* Header */
  s = get(from, 0);
  ret = sscanf(s, "%d %d %d %d", &w, &h, &ncols, &colsize);
  if (ret!=4) {
    return NULL;
  }
  if (colsize!=1) {
    return NULL;
  }

  /* Create the bitmap */
  bmp = create_bitmap_ex(bpp, w, h);
  if (!bmp) {
    return NULL;
  }

  /* Colors */
  colors = load_colors_(from, ncols, bpp, 1, get);
  if (!colors) {
    destroy_bitmap(bmp);
    return NULL;
  }
  /* Fill the palette if it exists */
  if (pal) {
    COLOR *c = colors;
    int n = 0;
    while (c->c) {
      pal[n].r = getr(c->v) >> 2;
      pal[n].g = getr(c->v) >> 2;
      pal[n].b = getr(c->v) >> 2;
      c++;
      n++;
    }
  }

  /* Convert the COLOR array to 8bits, if required */
  if (bpp==8) {
    COLOR *c = colors;
    int n = 0;
    while (c->c) {
      c->v = n;
      c++;
      n++;
    }
  }

  /* Image */
  fill_bitmap_(from, bmp, colors, 1+ncols, get);

  free(colors);

  return bmp;
}



/* load_xpm:
 *  Loads an XPM bitmap
 */
BITMAP *load_xpm(const char *fname, RGB *pal)
{
  BITMAP *bmp = NULL;
  FILE *f = fopen(fname, "rt");
  if (f) {
    const char *s = readline_(f);
    if (strcmp(s, "/* XPM */\n")==0) {
      s = readline_(f); /* static char * ... */
      bmp = xpm2bitmap(f, pal, get_from_file_);
    }
    fclose(f);
  }
  return bmp;
}



/* convert_xpm:
 *  Converts an includes XPM to a BITMAP
 */
BITMAP *convert_xpm(char *lines[], RGB *pal)
{
  return xpm2bitmap(lines, pal, get_from_strings_);
}



/* save_xpm:
 *  Saves an XPM bitmap
 */
int save_xpm(const char *fname, BITMAP *bmp, const RGB *pal)
{
  int i, j;
  int n;
  int color_count;
  COLOR *colors;
  FILE *f;
  int paletted = bitmap_color_depth(bmp)==8;

  colors = read_colors_(bmp, &color_count);
  if (!colors) {
    return -1;
  }

  f = fopen(fname, "wt");
  if (!f) {
    return -1;
  }

  /* Header */
  fprintf(f, "/* XPM */\n");
  fprintf(f, "static char * todo_xpm[] = {\n");
  fprintf(f, "\"%d %d %d 1\",\n", bmp->w, bmp->h, color_count);
  fprintf(f, "\" \tc None\",\n");

  /* Colors */
  if (paletted) {
    for (n=1; n<color_count; ++n) {
      fprintf(f, "\"%c\tc #%02x%02x%02x\",\n", colors[n].c,
          pal[colors[n].v].r, pal[colors[n].v].g, pal[colors[n].v].b);
    }
  }
  else {
    for (n=1; n<color_count; ++n) {
      fprintf(f, "\"%c\tc #%02x%02x%02x\",\n", colors[n].c,
          getr(colors[n].v), getg(colors[n].v), getb(colors[n].v));
    }
  }

  /* Image */
  for (j=0; j<bmp->h; ++j) {
    fprintf(f, "\"");
    for (i=0; i<bmp->w; ++i) {
      fprintf(f, "%c", find_character_(colors, getpixel(bmp, i, j)));
    }
    if (j!=bmp->h-1) {
      fprintf(f, "\",\n");
    }
    else {
      fprintf(f, "\"};\n");
    }
  }

  fclose(f);
  free(colors);

  return 0;
}



#ifdef TEST_MAIN

/* Alex the Allegator. thanks to Johan Peitz */
/* XPM */
static char * alex_xpm[] = {
"48 48 6 1",
" 	c None",
".	c #080808",
"+	c #FCFCFC",
"@	c #486028",
"#	c #587834",
"$	c #A4C47C",
"                ....     ....                   ",
"                ....     ....                   ",
"               .++++..  .++++..                 ",
"               .++++..  .++++..                 ",
"  ......       ...++.......++...                ",
"  ......       ...++.......++...                ",
"..@@####............@@###....##@..              ",
"..@@####............@@###....##@..              ",
"..###########################..#@@..            ",
"..###########################..#@@..            ",
".............................#####..            ",
".............................#####..            ",
"                      ...#########@@..          ",
"                ......#####..#######..          ",
"                ......#####..#######..          ",
"           .....######.....$$#######..          ",
"           .....######.....$$#######..          ",
"      .....#####......   ..$$#######..          ",
"      .....#####......   ..$$#######..          ",
"    ..#####.....        .$$##..###@@..          ",
"    ..#####.....        .$$##..###@@..          ",
"      .....           .......###@@..            ",
"      .....           .......###@@..            ",
"                    ..#########.@@..            ",
"                    ..##.......@##..            ",
"                    ..##.......@##..            ",
"                      ...$$$$#####..            ",
"                      ...$$$$#####..            ",
"                        .$$$$#####..         .. ",
"                        .$$$$#####..         .. ",
"                        .$$$$#####..       ..@@.",
"                        .$$$$#####..       ..@@.",
"                        .$$..#####@@..     ..@@.",
"                        .$$..#####@@..     ..@@.",
"                         ..####@..##..   ..##@@.",
"                         ..####@..##..   ..##@@.",
"                         ..####@..####...##@@$$.",
"                        .####@@.$$#######@@$$.. ",
"                        .####@@.$$#######@@$$.. ",
"                        .##@@.....$$$$$$$$$..   ",
"                        .##@@.....$$$$$$$$$..   ",
"                         ..##@@............     ",
"                         ..##@@............     ",
"                      ...######.@@..            ",
"                      ...######.@@..            ",
"                    ..#####@@###..@@..          ",
"                    ..#####@@###..@@..          ",
"                    ..................          "};



static void change_palette(BITMAP *bmp, RGB *bmp_pal)
{
  int i, j;
  RGB p[256];
  get_palette(p);
  for (j=0; j<bmp->h; ++j) for (i=0; i<bmp->w; ++i) {
    int c = getpixel(bmp, i, j);
    int new_c = makecol(bmp_pal[c].r, bmp_pal[c].g, bmp_pal[c].b);
    new_c = bestfit_color(p, bmp_pal[c].r, bmp_pal[c].g, bmp_pal[c].b);
    putpixel(bmp, i, j, new_c);
  }
}



int main(int argc, char **argv)
{
  BITMAP *bmp1 = 0;
  RGB pal1[256];
  BITMAP *bmp2 = 0;
  RGB pal2[256];
  const char *fin = 0;
  const char *fout = 0;
  int color_depth;
  int ret;

  if (argc!=2 && argc!=3 && argc!=4) {
    printf("Usage:\n$ test color_depth [input_file [output_file]]\n");
    return 1;
  }

  color_depth = strtol(argv[1], NULL, 10);
  if (color_depth!=8 && color_depth!=15 && color_depth!=16 &&
      color_depth!=24 && color_depth!=32) {
    printf("Color depth should be 8, 15, 16, 24 or 32\n");
    return 1;
  }

  if (argc==3 || argc==4) {
    fin = argv[2];
  }

  if (argc==4) {
    fout = argv[3];
  }

  /* Initialize Allegro */
  ret = allegro_init();
  if (ret!=0) return 1;
  set_color_depth(color_depth);
  ret = set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0);
  if (ret!=0) return 1;
  ret = install_keyboard();
  if (ret!=0) return 1;

  /* Register XPM load and save functions */
  register_bitmap_file_type("xpm", load_xpm, save_xpm);

  /* Convert the included XPM to a bitmap */
  bmp1 = convert_xpm(alex_xpm, pal1);
  /* Load the input file */
  if (fin) {
    bmp2 = load_bitmap(fin, pal2);
  }

  /* Convert the 2 bitmaps to use the default palette */
  if (color_depth==8) {
    change_palette(bmp1, pal1);
    change_palette(bmp2, pal2);
  }

  /* Blit the included XPM on the left of the screen */
  if (bmp1) {
    blit(bmp1, screen, 0, 0, 0, 0, bmp1->w, bmp1->h);
  }
  /* Blit the loaded bitmap on the right of the previous one */
  if (bmp2) {
    blit(bmp2, screen, 0, 0, 48, 0, bmp2->w, bmp2->h);
  }

  readkey();

  /* If an output file was given, save the input file */
  if (bmp2 && fout) {
    save_bitmap(fout, bmp1, NULL);
  }

  return 0;
}
END_OF_MAIN()
#endif

