#include <allegro.h>
#include "plugin.h"

#include "gui.h"
#include "globals.h" /* Define the variable GUI */

#include "cstring.h"
#include "cdataf.h"
#include "cbool.h"

#include "wrapper.h"

#include "sttcstr.h"

#include "map.h"

/* To be able to undo the menu */
#include "cmdselec.h"

#include "tiles.h"
#include "selected.h"

#include <string.h>


typedef enum {
  DESCRIPTION_INDEX = 0,
  PROPERTY_COUNT
} PROPERTY_INDEX;

#define DESCRIPTION_NAME "Description"
#define DESCRIPTION_DEFAULT_VALUE "No description"
#define DESCRIPTION_LENGTH 256



typedef enum {
  IMAGE_INDEX = 0,
  BLOCKED_INDEX,
  LAYER_COUNT
} LAYER_INDEX;

#define IMAGE_NAME "Image"
#define IMAGE_DEFAULT_VALUE 0
#define IMAGE_DATAFILE "images.dat"
#define BLOCKED_NAME "Blocked"
#define BLOCKED_DEFAULT_VALUE 0


/* Menu callback */
static void my_select(Map *map)
{
  /* Check if the map exists */
  if (!map) return;

  /* Create a wrapper to access the map */
  Wrapper wrapper(map);

  /* Get the active layer */
  int active_layer = wrapper.GetActiveLayerIndex();
  const Tiles *layer = wrapper.GetLayer(active_layer);
  if (layer) {
    /* Get the reference property */
    const BaseProperty *ref = wrapper.GetLayerReference(active_layer);

    /* Select the tiles */
    SelectedTiles *sel = new SelectedTiles();
    for (int i=layer->begini(); i<layer->endi(); ++i) {
      for (int j=layer->beginj(); j<layer->endj(); ++j) {
        /* If the tile exists and is equal to the reference */
        if (layer->get(i, j) && ref->IsEqualTo(layer->get(i, j))) {
          /* Select the tile */
          sel->Add(i, j);
        }
      }
    }

    /* Create the command and execute it */
    Command *cmd = new CommandSelectionReplace(map, sel);
    wrapper.Execute(cmd);
  }
}

int plugin_encoding(void)
{
  return U_ASCII;
}

const char *plugin_help(void)
{
  return "";
}

const char *plugin_about(void)
{
  return EMPTY_STRING; /* An empty string in the current encoding */
}

int plugin_init(int argc, const char **argv)
{
  int error = LS_NO_ERROR;
  /* Do some initializations, and set 'error' if needed */
  /* First, allow every features */
  GUI.AllowFeature(true, Features::Group::Everything);
  /* Then, disallow some */
  GUI.AllowFeature(false, Features::Group::VariableLayerCount);
  /* Add a user menu */
  GUI.SetUserMenu(0, "Select", my_select);
  return error;
}

void plugin_exit(void)
{
  /* Do some cleanup */
}

int plugin_new(void)
{
  int error = LS_NO_ERROR;
  /* Create a map, and set 'error' if needed */
  int width, height;
  if (popup_size("Map Width", &width, "Map Height", &height) == TRUE) {
    if (width>0 && height>0) {
      /* Create the map */
      Wrapper wrapper(width, height, LAYER_COUNT, PROPERTY_COUNT);

      wrapper.SetMapShape(MAP_SHAPE_RECT);
      wrapper.SetTileShape(TILE_SHAPE_RECT, 32, 32);
      wrapper.SetTileOffset(32, 32, ODD_NONE);

      wrapper.SetProperty<String>(
        DESCRIPTION_INDEX, DESCRIPTION_NAME, DESCRIPTION_DEFAULT_VALUE,
        DESCRIPTION_LENGTH
      );

      wrapper.SetLayer<DatafileObject>(
        IMAGE_INDEX, IMAGE_NAME, FULL_LAYER, IMAGE_DEFAULT_VALUE,
        new StaticString(IMAGE_DATAFILE)
      );
      wrapper.SetLayer<Boolean>(
        BLOCKED_INDEX, BLOCKED_NAME, FULL_LAYER, BLOCKED_DEFAULT_VALUE
      );

      wrapper.FillLayer(IMAGE_INDEX);
      wrapper.FillLayer(BLOCKED_INDEX);

      wrapper.SetMap();
    }
    else {
      error = LS_ERROR_INVALID;
    }
  }

  return error;
}

int plugin_load(const char *filename)
{
  int error = LS_NO_ERROR;
  /* Load the map from 'filename', and set 'error' if needed */
  PACKFILE *file = pack_fopen(filename, "r");

  if (file) {
    /* Load the map */
    int width = pack_igetl(file);
    int height = pack_igetl(file);
    if (width == EOF || height == EOF) {
      pack_fclose(file);
      return LS_ERROR_READ;
    }

    Wrapper wrapper(width, height, LAYER_COUNT, PROPERTY_COUNT);

    wrapper.SetMapShape(MAP_SHAPE_RECT);
    wrapper.SetTileShape(TILE_SHAPE_RECT, 32, 32);
    wrapper.SetTileOffset(32, 32, ODD_NONE);

    int length = pack_igetl(file);
    char description[DESCRIPTION_LENGTH];
    if (length == EOF) {
      pack_fclose(file);
      return LS_ERROR_READ;
    }
    if (pack_fread(description, length, file) < length) {
      pack_fclose(file);
      return LS_ERROR_READ;
    }
    wrapper.SetProperty<String>(
      DESCRIPTION_INDEX, DESCRIPTION_NAME, description, DESCRIPTION_LENGTH
    );

    wrapper.SetLayer<DatafileObject>(
      IMAGE_INDEX, IMAGE_NAME, FULL_LAYER, IMAGE_DEFAULT_VALUE,
      new StaticString(IMAGE_DATAFILE)
    );
    wrapper.SetLayer<Boolean>(
      BLOCKED_INDEX, BLOCKED_NAME, FULL_LAYER, BLOCKED_DEFAULT_VALUE
    );

    for (int i=0; i<width; i++) {
      for (int j=0; j<height; j++) {
        int image = pack_igetl(file);
        int blocked = pack_getc(file);

        if (image == EOF || blocked == EOF) {
          pack_fclose(file);
          return LS_ERROR_READ;
        }

        wrapper.SetTileValue<DatafileObject>(IMAGE_INDEX, i, j, image);
        wrapper.SetTileValue<Boolean>(BLOCKED_INDEX, i, j, blocked);
      }
    }

    pack_fclose(file);
    wrapper.SetMap();
  }
  else {
    error = LS_ERROR_FILE;
  }

  return error;
}

int plugin_save(const char *filename, const Map *map)
{
  int error = LS_NO_ERROR;
  /* Save 'map' in 'filename', and set 'error' if needed */
  const Wrapper wrapper(map);
  PACKFILE *file = pack_fopen(filename, "w");

  if (file) {
    /* Save the map */
    int width = wrapper.GetMapWidth();
    int height = wrapper.GetMapHeight();

    if (pack_iputl(width, file) == EOF) {
      pack_fclose(file);
      return LS_ERROR_WRITE;
    }
    if (pack_iputl(height, file) == EOF) {
      pack_fclose(file);
      return LS_ERROR_WRITE;
    }

    const char *description = wrapper.GetProperty<String>(DESCRIPTION_INDEX);
    int length = strlen(description) + 1;

    if (pack_iputl(length, file) == EOF) {
      pack_fclose(file);
      return LS_ERROR_WRITE;
    }
    if (pack_fwrite(description, length, file) < length) {
      pack_fclose(file);
      return LS_ERROR_WRITE;
    }

    for (int i=0; i<width; i++) {
      for (int j=0; j<height; j++) {
        /* Save the tile */
        int image = wrapper.GetTileValue<DatafileObject>(IMAGE_INDEX, i, j);
        int blocked = wrapper.GetTileValue<Boolean>(BLOCKED_INDEX, i, j);

        if (pack_iputl(image, file) == EOF) {
          pack_fclose(file);
          return LS_ERROR_WRITE;
        }
        if (pack_putc(blocked, file) == EOF) {
          pack_fclose(file);
          return LS_ERROR_WRITE;
        }
      }
    }

    pack_fclose(file);
  }
  else {
    error = LS_ERROR_FILE;
  }

  return error;
}

