#include <allegro.h>
#include "modules/ncds.h"
#include "level.h"
#include "modules/tilemap.h"
#include "modules/colormap.h"
#include "modules/gametime.h"
#include "character.h"
#include "data.h"
#include "main.h"
#include "title.h"

void vg_init(void)
{
	int i;

	if(allegro_init() != 0)
	{
		allegro_message("Allegro initialization failed!");
		exit(0);
	}
	
	/* set some window parameters */
	set_window_title("Video Game");
	if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) < 0)
	{
		allegro_message("Unable to set video mode.");
		exit(0);
	}
	set_display_switch_mode(SWITCH_BACKGROUND);
	
	if(install_timer() < 0)
	{
		allegro_message("Unable to initialize timer!");
		exit(0);
	}
	if(install_mouse() == -1)
	{
		allegro_message("Unable to initialize mouse!");
		exit(0);
	
	}
	if(install_keyboard() < 0)
	{
		allegro_message("Unable to initialize keyboard!");
		exit(0);
	}
	ncds_install();
	ncds_set_compressed_buffer_size(20000);
	vg_screen = create_bitmap(320, 240);
	if(!vg_screen)
	{
		allegro_message("Out of memory!");
		exit(0);
	}
	ncd_joy_install();
	if(num_joysticks > 0)
	{
		vg_controller.button[CONTROLLER_LEFT].key = KEY_J0_LEFT;
		vg_controller.button[CONTROLLER_RIGHT].key = KEY_J0_RIGHT;
		vg_controller.button[CONTROLLER_UP].key = KEY_J0_UP;
		vg_controller.button[CONTROLLER_DOWN].key = KEY_J0_DOWN;
		vg_controller.button[CONTROLLER_JUMP].key = KEY_J0_B0;
		vg_controller.button[CONTROLLER_FIRE].key = KEY_J0_B1;
		vg_controller.button[CONTROLLER_PAUSE].key = KEY_P;
	}
	else
	{
		vg_controller.button[CONTROLLER_LEFT].key = KEY_LEFT;
		vg_controller.button[CONTROLLER_RIGHT].key = KEY_RIGHT;
		vg_controller.button[CONTROLLER_UP].key = KEY_UP;
		vg_controller.button[CONTROLLER_DOWN].key = KEY_DOWN;
		vg_controller.button[CONTROLLER_JUMP].key = KEY_ALT;
		vg_controller.button[CONTROLLER_FIRE].key = KEY_LCONTROL;
		vg_controller.button[CONTROLLER_PAUSE].key = KEY_P;
	}
	
	vg_load_data();

	srand(time(0));
	for(i = 0; i < VG_CHARACTER_MAX_STATES; i++)
	{
		fmove_collision_map(&vg_player.sprite.cmap[i], vg_player.sprite.x, vg_player.sprite.y);
	}
	
}

void vg_exit(void)
{
	int i, j;

	if(vg_screen)
	{
		destroy_bitmap(vg_screen);
	}
	vg_free_data();
	ncd_joy_remove();
	ncds_remove();
}

void vg_player_set_state(int state)
{
	vg_player.sprite.state = state;
	fmove_collision_map(&vg_player.sprite.cmap[state], vg_player.sprite.x, vg_player.sprite.y);
}

void vg_logic(void)
{
	int i, new_step, was;
	int ok;

	if(key[KEY_ESC])
	{
		vg_done = 1;
	}
	switch(vg_state)
	{
		case VG_STATE_GAME:
		{
			vg_game_logic();
			break;
		}
		case VG_STATE_TITLE:
		{
			vg_title_logic();
			break;
		}
		case VG_STATE_LSELECT:
		{
			vg_lselect_logic();
			break;
		}
	}
}

void vg_update(void)
{
	int i, j;
	
	switch(vg_state)
	{
		case VG_STATE_GAME:
		{
			vg_game_render();
			break;
		}
		case VG_STATE_TITLE:
		{
			vg_title_render();
			break;
		}
		case VG_STATE_LSELECT:
		{
			vg_lselect_render();
			break;
		}
	}
}

void vg_draw(void)
{
	int i;
	
	vsync();
	stretch_blit(vg_screen, screen, 0, 0, 320, 240, 0, 0, SCREEN_W, SCREEN_H);
	switch(vg_state)
	{
		default:
		{
			break;
		}
	}
}

int main(int argc, char * argv[])
{
	int i;
	char new_name[128] = {0};
	int updated = 0, processed = 0;

	/* check arguments */
	if(argc > 1)
	{
		for(i = 1; i < argc; i++)
		{
		}
	}
	
	/* initialize the game */
	vg_init();
	set_palette(black_palette);
	
	gametime_init(60);
	vg_logic();
	vg_update();
	vg_draw();
	fade_in(vg_palette, 4);
	ncds_play_music("title.xm", 1);
	gametime_reset();
	
	/* the game loop */
	while(!vg_done)
	{
		int updated = 0;
		
		/* frame skip mode */
		while(gametime_get_frames() - gametime_tick > 0)
		{
			vg_logic();
			updated = 0;
			++gametime_tick;
		}
		
		/* update and draw the screen */
		if(!updated)
		{
			vg_update();
			vg_draw();
			updated = 1;
		}
		
		/* update the music */
		ncds_update_music();
	}
	
	/* stop music */
	ncds_stop_music();
	
	/* shut down program */
	vg_exit();

	/* success */
	return 0;
}
END_OF_MAIN()
