/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "toolsdlg.h"

#include "utils.h"
#include "debug.h"
#include "translat.h"

#include "help.h"
#include "icon.h"
#include "map.h"
#include "mapdlg.h"
#include "globals.h"
#include "selected.h"
#include "creator.h"
#include "stiles.h"
#include "brush.h"
#include "gui.h"
#include "cmdselec.h"
#include "cmdapply.h"
#include "cmdcut.h"
#include "cmdpaste.h"
#include "cmdbrush.h"
#include "cmdgroup.h"
#include "commands.h"
#include "lutils.h"
#include "emeftr.h"

#include <altheme.h>
#include <altheme/internal/theme.h>

#include <math.h>
#include <allegro.h>

#define IS_TOOL_ACTIVE(tool) (tools_[(tool)].flags & D_SELECTED)


static int tools_box_proc(int msg, DIALOG *d, int c)
{
  msg = do_help(msg, "Tools");
  if (msg==MSG_DRAW) {
    int ret = altheme_box_proc(msg, d, c);
    int rtm = text_mode(-1);
    char *str = (char*)(d->dp);
    int x = d->x+DIALOG_PADDING;
    int y = d->y+d->h-text_height(font)-DIALOG_PADDING;
    textout(screen, font, str, x, y, gui_fg_color);
    text_mode(rtm);
    return ret;
  }
  else {
    return altheme_box_proc(msg, d, c);
  }
}


static int tools_radio_proc(int msg, DIALOG *d, int c)
{
  msg = do_help(msg, "Tools");
  if (msg==MSG_DRAW) {
    altheme_draw_shadow_box(
      screen, d->x, d->y, d->x+d->w-1, d->y+d->h-1,
      d->flags&D_SELECTED, d->flags&D_GOTFOCUS
    );
    if (d->dp) {
      BITMAP *icon = (BITMAP*)d->dp;
      draw_sprite(screen, icon, d->x+(d->w-icon->w)/2, d->y+(d->h-icon->h)/2);
    }
    return D_O_K;
  }
  else {
    return altheme_radio_proc(msg, d, c);
  }
}


static void allow(bool yesno, DIALOG &d, BITMAP *(*icon_getter)(int))
{
  if (yesno) d.flags &= ~D_DISABLED;
  else d.flags |= D_DISABLED;
  if (icon_getter) d.dp = icon_getter(yesno);
}


ToolsDialog::ToolsDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d),
  count_(MAX_MERGE_ID), mouse_is_pressed_(false), saved_x_(-1), saved_y_(-1),
  clipboard_(0), brush_(0)
{
  Init(d);
}


ToolsDialog::ToolsDialog(DIALOG *d):
  Dialog(d),
  count_(MAX_MERGE_ID), mouse_is_pressed_(false), saved_x_(-1), saved_y_(-1),
  clipboard_(0), brush_(0)
{
  Init(d);
}


ToolsDialog::~ToolsDialog(void)
{
  Clear();
  ClearBrush();
}


void ToolsDialog::Init(DIALOG *d)
{
  box_ = &(d[TOOLS_DIALOG_BOX]);
  box_->proc = tools_box_proc;

  tools_=&(d[FIRST_TOOLS_DIALOG]);
  for(int i=0; i<NUM_TOOLS; i++) {
    tools_[i].proc = tools_radio_proc;
    tools_[i].d1 = TOOLS_GROUP;
  }

  tools_[DEFAULT_TOOL].flags |= D_SELECTED;

  slider_ = &(d[TOOLS_DIALOG_FILL_SLIDER]);
  slider_->proc = altheme_slider_proc;
  slider_->d1 = FILL_PROBA_MAX;
  slider_->d2 = FILL_PROBA_MAX;
}


void ToolsDialog::TranslationInit(void)
{
  static const Translation i18nDraw_A_Point      ("Draw a point (D)");
  static const Translation i18nSelect_Point      ("Select a point (P)");
  static const Translation i18nSelect_Rectangle  ("Select a rectangle (R)");
  static const Translation i18nSelect_Circle     ("Select a circle (C)");
  static const Translation i18nSelect_By_Property("Select by property value (B)");
  static const Translation i18nSelect_By_Wand    ("Select by a magic wand (W)");
  static const Translation i18nFill              ("Fill (F)");
  static const Translation i18nZoom              ("Zoom (Z) [Not yet done]");
  static const Translation i18nPaste             ("Paste (Ctrl-V)");
  static const Translation i18nPaste_Brush       ("Paste brush (Ctrl-P)");
  static const Translation i18nMove              ("Move [Not yet done]");
  static const Translation i18nUser              ("User [Not yet done]");
  static const Translation i18nFillDensity("Fill density");

  if (!box_->dp) {
    box_->dp = i18nFillDensity.void_ptr();

    tools_[TOOL_DRAW_POINT].dp2 = i18nDraw_A_Point.void_ptr();
    tools_[TOOL_SELECT_POINT].dp2 = i18nSelect_Point.void_ptr();
    tools_[TOOL_SELECT_RECT].dp2 = i18nSelect_Rectangle.void_ptr();
    tools_[TOOL_SELECT_CIRCLE].dp2 = i18nSelect_Circle.void_ptr();
    tools_[TOOL_SELECT_BY_PROP].dp2 = i18nSelect_By_Property.void_ptr();
    tools_[TOOL_SELECT_BY_WAND].dp2 = i18nSelect_By_Wand.void_ptr();
    tools_[TOOL_ZOOM].dp2 = i18nZoom.void_ptr();
    tools_[TOOL_FILL].dp2 = i18nFill.void_ptr();
    tools_[TOOL_PASTE].dp2 = i18nPaste.void_ptr();
    tools_[TOOL_PASTE_BRUSH].dp2 = i18nPaste_Brush.void_ptr();
    tools_[TOOL_MOVE].dp2 = i18nMove.void_ptr();
    tools_[TOOL_USER].dp2 = i18nUser.void_ptr();

    Allow(false, Features::ToolsBox::Everything);
    allow(false, tools_[TOOL_USER], 0); // TODO
    allow(false, tools_[TOOL_MOVE], get_move_icon); // TODO
    allow(false, tools_[TOOL_ZOOM], get_zoom_icon); // TODO
  }
}


void ToolsDialog::SizeInit(void)
{
  const int icon_size = 32;
  const int imax = 6;
  const int jmax = 2;
  for (int j=0; j<jmax; ++j) {
    for (int i=0; i<imax; ++i) {
      if (i*jmax+j < NUM_TOOLS) {
        DIALOG *t = &(tools_[i*jmax+j]);
        t->x = box_->x + i*icon_size + DIALOG_PADDING;
        t->y = box_->y + j*icon_size + DIALOG_PADDING;
        t->w = icon_size;
        t->h = icon_size;
      }
    }
  }
  char *str = (char*)(box_->dp);
  int len = text_length(font, str);
  int height = text_height(font);
  slider_->x = box_->x + DIALOG_PADDING + len;
  slider_->y = box_->y + box_->h - height - DIALOG_PADDING;
  slider_->w = box_->w - 2*DIALOG_PADDING - len;
  slider_->h = height;
}


void ToolsDialog::LazyInit(void)
{
  TranslationInit();
  SizeInit();
}


void ToolsDialog::Press(int tile_x, int tile_y, int /*shifts*/)
{
  if(!GetMap()) return;

  GUI.SetMouseBusy();

  mouse_is_pressed_ = true;

  saved_x_ = tile_x;
  saved_y_ = tile_y;

  count_ = (count_+1) % MAX_MERGE_ID;

  //Change the mouse pointer ?

  GUI.UnsetMouseBusy();
}


void ToolsDialog::Release(int tile_x, int tile_y, int shifts)
{
  if(!GetMap()) return;

  SelectedTiles *sel = NULL;

  GUI.SetMouseBusy();

  if(IS_TOOL_ACTIVE(TOOL_SELECT_RECT)) {
    int begin_x = MIN(saved_x_, tile_x);
    int begin_y = MIN(saved_y_, tile_y);
    int end_x = MAX(saved_x_, tile_x);
    int end_y = MAX(saved_y_, tile_y);

    /* Remove the graphical feedback */
    GUI.GetMapDialog()->Rect();
    sel = LayerUtils::SelectRect(
      GetMap()->GetActiveLayer(), begin_x, begin_y, end_x, end_y
    );
    CHECK_POINTER(sel);
  }
  else if(IS_TOOL_ACTIVE(TOOL_SELECT_CIRCLE)) {
    int radius2 =
      (tile_x-saved_x_)*(tile_x-saved_x_) + (tile_y-saved_y_)*(tile_y-saved_y_);
    /* Remove the graphical feedback */
    GUI.GetMapDialog()->Circle();
#if 0
    sel = LayerUtils::SelectCircle(
      GetMap()->GetActiveLayer(), saved_x_, saved_y_, radius2
    );
#else
    sel = LayerUtils::SelectCircle(
      GetMap()->GetActiveLayer(), saved_x_, saved_y_, (int)sqrt((double)(radius2))
    );
#endif
    CHECK_POINTER(sel);
  }
  else if(IS_TOOL_ACTIVE(TOOL_SELECT_BY_PROP)) {
    sel = LayerUtils::SelectByProperty(
      GetMap()->GetActiveLayer(), tile_x, tile_y
    );
    CHECK_POINTER(sel);
  }
  else if(IS_TOOL_ACTIVE(TOOL_SELECT_BY_WAND)) {
    sel = LayerUtils::SelectByWand(
      GetMap()->GetActiveLayer(), tile_x, tile_y
    );
    CHECK_POINTER(sel);
  }
  else if(IS_TOOL_ACTIVE(TOOL_PASTE)) {
    Paste();
  }
  else if(IS_TOOL_ACTIVE(TOOL_PASTE_BRUSH)) {
    PasteBrush();
  }

  UpdateSelection(sel, shifts);

  mouse_is_pressed_ = false;

  GUI.UnsetMouseBusy();
}


void ToolsDialog::Click(int tile_x, int tile_y, int shifts)
{
  if(!GetMap()) return;

  SelectedTiles *sel = NULL;

  GUI.SetMouseBusy();

  if(IS_TOOL_ACTIVE(TOOL_SELECT_POINT)) {
    sel = LayerUtils::SelectRect(
      GetMap()->GetActiveLayer(), tile_x, tile_y, tile_x, tile_y
    );
    CHECK_POINTER(sel);
  }
  else if(IS_TOOL_ACTIVE(TOOL_DRAW_POINT)) {
    Map *map = GetMap();
    if(
      map->GetSelectedTiles()->Empty() ||
      map->GetSelectedTiles()->IsIn(tile_x, tile_y)
    ) {
      Command *cmd = new CommandApply(
        map, map->GetActiveLayerIndex(), tile_x, tile_y,
        map->GetActiveLayerReference(), count_
      );
      map->GetCommandsManager()->Execute(cmd);
    }
  }
  else if(IS_TOOL_ACTIVE(TOOL_FILL)) {
    /* If there is a selection, fill it. Else do a flood fill */
    Map *map = GetMap();
    SelectedTiles *s = map->GetSelectedTiles();
    if (s->Empty()) {
      s = LayerUtils::SelectByWand(map->GetActiveLayer(), tile_x, tile_y);
    }
    Command *cmd = new CommandApply(
      map, map->GetActiveLayerIndex(), s, map->GetActiveLayerReference(),
      MAX_MERGE_ID, slider_->d2
    );
    if (s!=map->GetSelectedTiles()) {
      delete s;
    }
    map->GetCommandsManager()->Execute(cmd);
  }
  else if (IS_TOOL_ACTIVE(TOOL_SELECT_RECT)) {
    /* Draw a ggraphical feedback */
    GUI.GetMapDialog()->Rect(saved_x_, saved_y_, tile_x, tile_y);
  }
  else if (IS_TOOL_ACTIVE(TOOL_SELECT_CIRCLE)) {
    /* Draw a ggraphical feedback */
    GUI.GetMapDialog()->Circle(saved_x_, saved_y_, tile_x, tile_y);
  }

  UpdateSelection(sel, shifts);

  GUI.UnsetMouseBusy();
}


void ToolsDialog::UpdateSelection(SelectedTiles *sel, int shifts)
{
  if(sel) {
    if(shifts & KB_SHIFT_FLAG) { /* Add */
      Command *cmd = new CommandSelectionAdd(GetMap(), sel);
      GetMap()->GetCommandsManager()->Execute(cmd);
    }
    else if(shifts & KB_CTRL_FLAG) { /* Suppress */
      Command *cmd = new CommandSelectionSuppress(GetMap(), sel);
      GetMap()->GetCommandsManager()->Execute(cmd);
    }
    else { /* No shift, replace */
      Command *cmd = new CommandSelectionReplace(GetMap(), sel);
      GetMap()->GetCommandsManager()->Execute(cmd);
    }
    delete sel;
  }
}


void ToolsDialog::Clear(void)
{
  if(clipboard_) {
    delete clipboard_;
  }
  clipboard_ = 0;
}


void ToolsDialog::ClearBrush(void)
{
  if(brush_) {
    delete brush_;
  }
  brush_ = 0;
}


void ToolsDialog::Copy(void)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  Clear();
  clipboard_ = LayerUtils::Copy(
    map->GetActiveLayer(), map->GetSelectedTiles()
  );
  CHECK_POINTER(clipboard_);
}


void ToolsDialog::CopyBrush(void)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  ClearBrush();
  brush_ = new Brush(
    map->GetLayers(), &(map->GetViewedLayers()), map->GetSelectedTiles()
  );
}


void ToolsDialog::Cut(void)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  Copy();
  Command *cmd = new CommandCut(
    map, map->GetActiveLayerIndex(), map->GetSelectedTiles()
  );
  map->GetCommandsManager()->Execute(cmd);
}


void ToolsDialog::Paste(void)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  if(clipboard_) {
    int x = GUI.GetMapDialog()->GetTileX(mouse_x, mouse_y);
    int y = GUI.GetMapDialog()->GetTileY(mouse_x, mouse_y);
    Command *cmd = new CommandPaste(
      map, map->GetActiveLayerIndex(), clipboard_, x, y
    );
    map->GetCommandsManager()->Execute(cmd);
  }
}


void ToolsDialog::PasteBrush(void)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  if(brush_) {
    int x = GUI.GetMapDialog()->GetTileX(mouse_x, mouse_y);
    int y = GUI.GetMapDialog()->GetTileY(mouse_x, mouse_y);
    Command *cmd = new CommandPasteBrush(map, brush_, x, y);
    map->GetCommandsManager()->Execute(cmd);
  }
}


void ToolsDialog::Paste(SparseTiles *tiles)
{
  Map *map = GetMap();
  DBG_ASSERT(map);
  Command *cmd = new CommandPaste(
    map, map->GetActiveLayerIndex(), tiles, tiles->begini(), tiles->beginj()
  );
  map->GetCommandsManager()->Execute(cmd);
}


void ToolsDialog::ChangeTool(int new_tool)
{
  DBG_ASSERT(new_tool+1>=1 && new_tool+1<NUM_TOOLS_DIALOGS);

  /* Don't allow changing tool during a user input */
  if (!mouse_is_pressed_) {
    scare_mouse();
    for(int i=0; i<NUM_TOOLS; i++) {
      tools_[i].flags &= ~D_SELECTED;
      tools_[i].flags |= D_DIRTY;
    }
    tools_[new_tool].flags |= D_SELECTED;
    tools_[new_tool].flags |= D_DIRTY;
    unscare_mouse();
  }
}


void ToolsDialog::SetBrush(const Brush *brush)
{
  ClearBrush();
  brush_ = brush->Clone();
  CHECK_POINTER(brush_);
}


const char *ToolsDialog::HelpText() const
{
  static const Translation i18nToolDialog("Tools");
  static const Translation i18nFillDensity("Fill density (0 to 100%)");

  int mx = mouse_x;
  int my = mouse_y;
  for (int n=0; n<NUM_TOOLS; ++n) {
    DIALOG *t = &(tools_[n]);
    if (mx>=t->x && mx<t->x+t->w && my>=t->y && my<t->y+t->h) {
      return (char*)(t->dp2);
    }
  }
  if (my>slider_->y) {
    return i18nFillDensity.string();
  }
  return i18nToolDialog.string();
}


void ToolsDialog::Allow(bool yesno, unsigned int features)
{
  if (features&Features::ToolsBox::Pencil) {
    allow(yesno, tools_[TOOL_DRAW_POINT], get_draw_icon);
  }
  if (features&Features::ToolsBox::Fill) {
    allow(yesno, tools_[TOOL_FILL], get_fill_icon);
    allow(yesno, *slider_, 0);
  }
  if (features&Features::ToolsBox::Point) {
    allow(yesno, tools_[TOOL_SELECT_POINT], get_point_icon);
  }
  if (features&Features::ToolsBox::Rectangle) {
    allow(yesno, tools_[TOOL_SELECT_RECT], get_rect_icon);
  }
  if (features&Features::ToolsBox::Circle) {
    allow(yesno, tools_[TOOL_SELECT_CIRCLE], get_circle_icon);
  }
  if (features&Features::ToolsBox::ByProperty) {
    allow(yesno, tools_[TOOL_SELECT_BY_PROP], get_prop_icon);
  }
  if (features&Features::ToolsBox::Wand) {
    allow(yesno, tools_[TOOL_SELECT_BY_WAND], get_wand_icon);
  }
  if (features&Features::ToolsBox::Zoom) {
    //allow(yesno, tools_[TOOL_ZOOM], get_zoom_icon); // TODO
  }
  if (features&Features::ToolsBox::Paste) {
    allow(yesno, tools_[TOOL_PASTE], get_paste_icon);
  }
  if (features&Features::ToolsBox::PasteBrush) {
    allow(yesno, tools_[TOOL_PASTE_BRUSH], get_pasteb_icon);
  }
  if (features&Features::ToolsBox::Move) {
    //allow(yesno, tools_[TOOL_MOVE], get_move_icon); // TODO
  }
  if (features&Features::ToolsBox::User) {
    //allow(yesno, tools_[TOOL_USER], 0); // TODO
  }
}

