/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "text.h"

#include <allegro.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>



/*
 * Helpers
 */

/* Returns a pointer to the first occurence in 'string' of a character in 'c'
 * */
static const char *next_(const char *string, const char *c)
{
  const char *ret = string;
  if (ugetc(c)) {
    ret = 0;
    while (ugetc(c)) {
      const char *r = ustrchr(string, ugetxc(&c));
      if (r && (!ret || r-string<ret-string)) ret = r;
    }
    if (!ret) ret = ustrchr(string, 0);
  }
  return ret;
}

static char *uconvert_dup(const char *src, int src_encoding)
{
  char *ret = 0;
  if (src) {
    int size = uconvert_size(src, src_encoding, U_CURRENT);
    ret = (char*)malloc(size);
    do_uconvert(src, U_ASCII, ret, U_CURRENT, size);
  }
  return ret;
}


static int ustrzcmp(const char *s1, int size, const char *s2)
{
  /* Assume there is only one encoding */
  while (size && *s1 && *s2 && *s1==*s2) {
    s1++;
    s2++;
    size--;
  }
  return size ? *s1-*s2 : 0;
}

static int my_ustrlen(const char *s, int size)
{
  int ret = 0;
  const char *end = s+size;
  while(s!=end && ugetxc(&s)) {
    ret ++;
  }
  return ret;
}




static int is(COMMAND *cmd, const char *string)
{
  return ustrzcmp(string, ustrsize(cmd->keyword), cmd->keyword)==0;
}


static int is_begin(COMMAND *commands, const char *string)
{
  return is(commands, string);
}


static int is_end(COMMAND *commands, const char *string)
{
  return is(commands+1, string);
}


static COMMAND *is_type(COMMAND *commands, const char *string)
{
  while (!is(commands, string)) {
    commands++;
  }
  return commands;
}


static const char *next(COMMAND *cmd, const char *s)
{
  /* End of lines are eaten by the parser */
  return next_(s+ustrsize(cmd->keyword), cmd->end);
}


/*
 * Returns the id (newly allocated) from a string beginning by \\begin
 */
static char *get_id(COMMAND *cmd, const char *string)
{
  static char *blanks = 0;

  int n;
  char *ret = NULL;
  const char *begin = string+ustrsize(cmd->keyword);
  int len = next(cmd, string)-begin;
  char *s;

  if (!blanks) blanks = uconvert_dup(" \t\n", U_ASCII);

  ret = (char*)malloc(len+ucwidth(0));
  s = ret;
  /* Trim blanks before */
  while (ustrchr(blanks, *begin)) {
    begin += ucwidth(*begin);
    len --;
  }
  { /* Trim blanks after */
    char *last_non_blank = 0;
    for (n=0; n<len; ++n) {
      int c = ugetxc(&begin);
      if (!ustrchr(blanks, c)) last_non_blank = s;
      s += usetc(s, c);
    }
    if (last_non_blank) {
      usetc(last_non_blank+1, 0);
    }
    else {
      usetc(s, 0);
    }
  }
  return ret;
}


static const char *text_(COMMAND *cmd, const char *begin, int size)
{
  static char *text = NULL;

  char *out;
  int len = my_ustrlen(begin, size);
  int append_size = cmd->append ? ustrsize(cmd->append) : 0;
  int prepend_size = cmd->prepend ? ustrsize(cmd->prepend) : 0;
  int base_size, total_size;
  if (cmd->replace) {
    begin = cmd->replace;
    size = ustrsize(begin);
    len = ustrlen(begin);
  }
  base_size = size + prepend_size + append_size + ucwidth(0);
  total_size = base_size;
  if (cmd->overline) {
    int overline_size = ustrsize(cmd->overline);
    total_size += size*overline_size + ucwidth('\n');
  }
  if (cmd->underline) {
    int underline_size = ustrsize(cmd->underline);
    total_size += size*underline_size + ucwidth('\n');
  }
  text = (char*) realloc(text, total_size);
  out = text;
  /* Prepend */
  if (cmd->prepend) {
    const char *in;
    for (in=cmd->prepend; ugetc(in); ) {
      out += usetc(out, ugetxc(&in));
    }
  }
  /* Overline */
  if (cmd->overline) {
    int n;
    for (n=0; n<size; ++n) out += usetc(out, ugetc(cmd->overline));
    out += usetc(out, '\n');
  }
  /* Input text or replace */
  if (!cmd->replace) { /* Input text */
    const char *in;
    int n;
    unsigned int space = 0;
    for (in=begin, n=0; n<len; ++n) {
      int c = ugetxc(&in);
      switch (c) {
        case ' ':
        case '\n':
        case '\t':
          c=' ';
          space++;
          break;

        default:
          space=0;
          break;
      }
      if (space<=1) out += usetc(out, c);
    }
  }
  else { /* Replace */
    const char *in;
    int n;
    for (in=begin, n=0; n<len; ++n) out += usetc(out, ugetxc(&in));
  }
  /* Underline */
  if (cmd->underline) {
    int n;
    out += usetc(out, '\n');
    for (n=0; n<size; ++n) out += usetc(out, ugetc(cmd->underline));
  }
  /* Append */
  if (cmd->append) {
    const char *in;
    for (in=cmd->append; ugetc(in); ) out += usetc(out, ugetxc(&in));
  }
  usetc(out, 0);
  return text;
}


static const char *text(COMMAND *cmd, const char *s)
{
  static char *blanks = 0;

  const char *begin = s+ustrsize(cmd->keyword);
  int size = next(cmd, s)-begin;

  if (!blanks) blanks = uconvert_dup(" \t\n", U_ASCII);

  /* Trim blanks before */
  while (size!=0 && ustrchr(blanks, *begin)) {
    begin += ucwidth(*begin);
    size -= ucwidth(*begin);
  }
  { /* Trim blanks after */
    int last_blanks_size = 0;
    const char *s = begin;
    while (s!=begin+size && ugetc(s)) {
      int c = ugetxc(&s);
      if (ustrchr(blanks, c)) last_blanks_size+=ucwidth(c);
      else last_blanks_size = 0;
    }
    size -= last_blanks_size;
  }
  return text_(cmd, begin, size);
}



static void append(char **text, int *len, const char *word)
{
  int word_len = ustrsize(word);
  *text = (char*)realloc(*text, *len+word_len+ucwidth(0));
  ustrcpy(*text+*len, word);
  *len += word_len;
}



/*
 * Parse a text
 */
static int parse_one(COMMAND *commands, char **input, TEXT *t)
{
  /* Initialize */
  int len = 0;
  t->text = NULL;

  /* Find the beginning of the text (or the end of file) */
  while (ugetc(*input) && !is_begin(commands, *input)) {
    *input += uwidth(*input);
  }

  /* If beginning found, parse the text */
  if (ugetc(*input)) {
    /* Get the text id */
    t->id = get_id(commands, *input);

    /* Get the text */
    while (*input) {
      COMMAND *type = is_type(commands, *input);
      const char *w = text(type, *input);
      append(&(t->text), &len, w);
      if (is_end(commands, *input)) break;
      *input = (char*)next(type, *input);
    }
  }

  /* Return true if there's still text to read */
  return ugetc(*input);
}


static char *fread_all(const char *fname)
{
  char *ret = NULL;

  PACKFILE *f = pack_fopen(fname, "rt");
  if (f) {
    int len = file_size(fname);
    char *file = (char*)malloc(len+ucwidth(0));
    ret = file;
    pack_fread(file, len, f);
    file[len] = 0;
    pack_fclose(f);
    if (need_uconvert(file, U_UTF8, U_CURRENT)) {
      ret = (char*)uconvert_dup(file, U_UTF8);
      free(file);
    }
  }
  return ret;
}



TEXT *load_texts(const char *fname, COMMAND *cmd)
{
  TEXT *items = NULL;
  char *file;

  file = fread_all(fname);
  if (file) {

    /* Parse file */
    int count = 0;
    char *input = file;
    items = (TEXT*) malloc((count+1)*sizeof(TEXT));
    while (parse_one(cmd, &input, items+count)) {
      count++;
      items = (TEXT*) realloc(items, (count+1)*sizeof(TEXT));
    }
    items[count].id = NULL;
    items[count].text = NULL;

    free(file);
  }

  return items;
}


void unload_texts(TEXT *texts)
{
  TEXT *t = texts;
  while (t->id) {
    free(t->id);
    free(t->text);
    t++;
  }
  free(texts);
}


const char *find_text(TEXT *texts, const char *id)
{
  TEXT *t = texts;
  while (t->id) {
    if (ustrcmp(t->id, id)==0) {
      return t->text;
    }
    t++;
  }
  return NULL;
}

