/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "stiles.h"

#include "prop.h"
#include "creator.h"

#include "debug.h"

/*
 * SparseTiles
 */
SparseTiles::SparseTiles(BaseCreator *c, int i, int j, int w, int h):
  Tiles(c, i, j, w, h), tiles_()
{
}


SparseTiles::SparseTiles(BaseCreator *c):
  Tiles(c), tiles_()
{
}


SparseTiles::SparseTiles(const SparseTiles &other):
  Tiles(other)
{
  for (const_iterator i=other.begin(); i!=other.end(); ++i) {
    tiles_.insert(Data((*i).first, (*i).second->Clone()));
  }
}


SparseTiles::~SparseTiles()
{
  for (iterator i=begin(); i!=end(); ++i) {
    delete (*i).second;
  }
}


SparseTiles *SparseTiles::clone() const
{
  return new SparseTiles(*this);
}


SparseTiles &SparseTiles::operator=(const SparseTiles &other)
{
  Tiles::operator=(other);
  for (iterator i=begin(); i!=end(); ++i) {
    delete (*i).second;
  }
  tiles_.clear();
  for (const_iterator i=other.begin(); i!=other.end(); ++i) {
    tiles_.insert(Data((*i).first, (*i).second->Clone()));
  }
  return *this;
}


BaseProperty *SparseTiles::get_(int i, int j) const
{
  Imp::const_iterator it = tiles_.find(pack(i, j));
  return it==tiles_.end() ? 0 : (*it).second;
}


void SparseTiles::set_(int i, int j, BaseProperty *p)
{
  Imp::iterator it = tiles_.find(pack(i, j));
  if (it != tiles_.end()) {
    BaseProperty *tmp = (*it).second;
    (*it).second = p;
    delete tmp;
  }
  else {
    /* No need to insert an empty tile */
    if (p) {
      tiles_.insert(Data(pack(i, j), p));
    }
  }
}


void SparseTiles::copy_(int i, int j, const BaseProperty *p)
{
  DBG_ASSERT(p);
  Imp::iterator it = tiles_.find(pack(i, j));
  if (it != tiles_.end()) {
    DBG_ASSERT((*it).second);
    (*it).second->CopyFrom(p);
  }
  else {
    tiles_.insert(Data(pack(i, j), p->Clone()));
  }
}


BaseProperty *SparseTiles::clear_(int i, int j)
{
  BaseProperty *ret = 0;
  Imp::iterator t = tiles_.find(pack(i, j));
  if (t != tiles_.end()) {
    ret = (*t).second;
    tiles_.erase(t);
  }
  return ret;
}


void SparseTiles::move_(int dx, int dy)
{
  Imp tmp;
  for (const_iterator t=begin(); t!=end(); ++t) {
    int i = get_i(t);
    int j = get_j(t);
    BaseProperty *p = (*t).second;
    tmp.insert(Data(pack(i+dx, j+dy), p));
  }
  std::swap(tiles_, tmp);
}


void SparseTiles::insert_col_(int x, int count)
{
  Imp tmp;
  for (int j=beginj(); j<endj(); ++j) {
    int i=begini();
    for ( ; i<x; ++i)
      if (get(i, j)) tmp.insert(Data(pack(i, j), get(i, j)));
    for ( ; i<x+count; ++i)
      ;
    for ( ; i<endi()+count; ++i)
      if (get(i-count, j)) tmp.insert(Data(pack(i, j), get(i-count, j)));
  }
  std::swap(tiles_, tmp);
}


void SparseTiles::remove_col_(int x, int count)
{
  Imp tmp;
  for (int j=beginj(); j<endj(); ++j) {
    int i=begini();
    for ( ; i<x; ++i)
      if (get(i, j)) tmp.insert(Data(pack(i, j), get(i, j)));
    for ( ; i<x+count; ++i)
      ;
    for ( ; i<endi(); ++i)
      if (get(i, j)) tmp.insert(Data(pack(i-count, j), get(i, j)));
  }
  std::swap(tiles_, tmp);
}


void SparseTiles::insert_row_(int y, int count)
{
  Imp tmp;
  int j=beginj();
  for ( ; j<y; ++j)
    for (int i=begini(); i<endi(); ++i)
      if (get(i, j)) tmp.insert(Data(pack(i, j), get(i, j)));
  for ( ; j<y+count; ++j)
    for (int i=begini(); i<endi(); ++i)
      ;
  for ( ; j<endj()+count; ++j)
    for (int i=begini(); i<endi(); ++i)
      if (get(i, j-count)) tmp.insert(Data(pack(i, j), get(i, j-count)));
  std::swap(tiles_, tmp);
}


void SparseTiles::remove_row_(int y, int count)
{
  Imp tmp;
  int j=beginj();
  for ( ; j<y; ++j)
    for (int i=begini(); i<endi(); ++i)
      if (get(i, j)) tmp.insert(Data(pack(i, j), get(i, j)));
  for ( ; j<y+count; ++j)
    for (int i=begini(); i<endi(); ++i)
      ;
  for ( ; j<endj(); ++j)
    for (int i=begini(); i<endi(); ++i)
      if (get(i, j)) tmp.insert(Data(pack(i, j-count), get(i, j)));
  std::swap(tiles_, tmp);
}

