/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "settheme.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include <altheme.h>

#include <allegro.h>

static const UString usDefaultThemeName("default");
static const UString usPaperThemeName("paper");
static const UString usThreeDThemeName("3D");
static const UString usRoundThemeName("round");


static void select_menu(MENU *menu)
{
  for(int n=0; n<NUM_THEMES; n++) {
    theme_menu()[n].flags = 0;
  }
  menu->flags |= D_SELECTED;
}

#define SET_THEME(name) \
  static int name##_theme(void) \
  { \
    select_menu(active_menu); \
    altheme_init_##name(gui_fg_color, gui_bg_color, gui_mg_color); \
    return D_REDRAW; \
  }

SET_THEME(default)
SET_THEME(paper)
SET_THEME(threed)
SET_THEME(round)



MENU *theme_menu(void)
{
  static const Translation i18nDefault("&Default");
  static const Translation i18nPaper  ("&Paper");
  static const Translation i18n3_D    ("&3 D");
  static const Translation i18nRound  ("&Round");

  static MENU theme_menu[] = {
  /* char *text             , int (*proc)(), child,  flags, dp */
    {i18nDefault.char_ptr() , default_theme, 0, D_SELECTED, 0},
    {i18nPaper.char_ptr()   ,   paper_theme, 0,          0, 0},
    {i18n3_D.char_ptr()     ,  threed_theme, 0,          0, 0},
    {i18nRound.char_ptr()   ,   round_theme, 0,          0, 0},
    {}
  };

  return theme_menu;
}

static const char *theme_names[] = {
  usDefaultThemeName,
  usPaperThemeName,
  usThreeDThemeName,
  usRoundThemeName,
  0
};


void SetTheme(const char *theme_name)
{
  int n;
  for(n=0; n<NUM_THEMES; n++) {
    if(ustricmp(theme_name, theme_names[n]) == 0) {
      break;
    }
  }
  if(n<NUM_THEMES) {
    MENU *old_active = active_menu;
    active_menu = &(theme_menu()[n]);
    theme_menu()[n].proc();
    active_menu = old_active;
  }
}

const char *GetThemeName(void)
{
  int n;
  for(n=0; n<NUM_THEMES; n++) {
    if(theme_menu()[n].flags&D_SELECTED) {
      break;
    }
  }
  if(n<NUM_THEMES) {
    return theme_names[n];
  }
  return GetDefaultThemeName();
}

const char *GetDefaultThemeName(void)
{
  return theme_names[DEFAULT_THEME];
}

