/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__POOL_USER__H__
#define EME__POOL_USER__H__
/*----------------------------------------------------------------------------
  Base class for objects that need a memory pool

  Pools for object with size: 1*4, 2*4, ... num_pools*4 bytes

  Usage (if you want num_pools=8):
    class Managed: public MultiPoolUser<8> {
      ...
    }
  You must call Managed::init() before allocating a Managed object.

  You can also use:
    class Managed MEMORY_POOL_CLASS(8) {
      ...
    }
  You must call MEMORY_POOL_INIT(Managed) before allocating a Managed object.
  Be careful if a Managed object can be created before entering main.

  If your compiler does not support static template member (i.e. if the linker
  complains because it can't find 'is_initialized' and 'mem_pools'), put
  in a c++ file:
    MEMORY_POOL_INSTANCE(8)

  If you use the defines (MEMORY_POOL_CLASS, MEMORY_POOL_INIT and
  MEMORY_POOL_INSTANCE), you'll need to compile with -DUSE_MEMORY_POOL.
----------------------------------------------------------------------------*/


#include "pool.h"

#include "debug.h"


#ifdef USE_MEMORY_POOL

#define MEMORY_POOL_INIT(a_class) a_class::init()
#define MEMORY_POOL_EXIT(a_class) a_class::exit()
#define MEMORY_POOL_CLASS(num_pools) : public MultiPoolUser<num_pools>
#define MEMORY_POOL_INSTANCE(num_pools) \
  template <> MemoryPool *MultiPoolUser<num_pools>::mem_pools[num_pools]; \
  template <> int MultiPoolUser<num_pools>::is_initialized = 0;

#else /* !USE_MEMORY_POOL */

#define MEMORY_POOL_INIT(a_class)
#define MEMORY_POOL_EXIT(a_class)
#define MEMORY_POOL_CLASS(num_pools)
#define MEMORY_POOL_INSTANCE(num_pools)

#endif /* !USE_MEMORY_POOL */


template <int num_pools>
class MultiPoolUser {

public:

  static void *operator new(size_t size)
    {
      if(size && size%4==0 && size/4<=num_pools) {
        return mem_pools[size/4-1]->Alloc(size);
      }
      return ::operator new(size);
    }

  static void operator delete(void *p, size_t size)
    {
      if(size && size%4==0 && size/4<=num_pools) {
        mem_pools[size/4-1]->Free(p, size);
      }
      else {
        ::operator delete(p);
      }
    }

  static void init(void)
    {
      if(!is_initialized) {
        for(int i=0; i<num_pools; i++) {
          mem_pools[i] = new MemoryPool((i+1)*4);
        }
        is_initialized = 1;
      }
    }

  static void exit(void)
    {
      if (is_initialized) {
        for (int i=0; i<num_pools; ++i) {
          delete mem_pools[i];
        }
        is_initialized = 0;
      }
    }

  virtual ~MultiPoolUser()
    {}

private:

  static int is_initialized;
  static MemoryPool *mem_pools[num_pools];

};


template <int num_pools>
MemoryPool *MultiPoolUser<num_pools>::mem_pools[num_pools];

template <int num_pools>
int MultiPoolUser<num_pools>::is_initialized = 0;


#endif /* EME__POOL_USER__H__ */

