/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__MAP_DIALOG__
#define EME__MAP_DIALOG__
/*----------------------------------------------------------------------------
  Dialog for displaying the map
----------------------------------------------------------------------------*/

#include "debug.h"

#include "dialog.h"

typedef enum {
  MAP_DIALOG_BOX = 0,
  MAP_DIALOG_HSLIDER,
  MAP_DIALOG_VSLIDER,
  NUM_MAP_DIALOGS
} MAP_DIALOG;

typedef enum {
  ODD_NONE = 0,
  ODD_RIGHT = 1,
  ODD_LEFT = 2
} ODD;

typedef enum {
  MAP_SHAPE_RECT = 0,
  MAP_SHAPE_DIAMOND = 1
} MAP_SHAPE;

typedef enum {
  TILE_SHAPE_RECT = 0,
  TILE_SHAPE_DIAMOND = 1
    // TILE_SHAPE_HEXA
} TILE_SHAPE;

struct BITMAP;

class MapDialog: public Dialog {

public:
  /* Constructor/destructor
     ---------------------------------*/
  MapDialog(int x, int y, int w, int h, DIALOG *d);
  MapDialog(DIALOG *d);
  ~MapDialog(void);

  void LazyInit(void);

  void InitMap(void);

  void DrawGrid(int yesno);
  int IsGridDrawn(void) const
    {return draw_grid;}

  void SetMapShape(MAP_SHAPE shape);
  void SetTileShape(TILE_SHAPE shape, int w, int h);
  void SetTileOffset(int dx, int dy, ODD odd=ODD_NONE);
  float GetScaledTileDx(void) const
    {return GetScale()*tile_dx;}
  float GetScaledTileDy(void) const
    {return GetScale()*tile_dy;}
  float GetScaledTileWidth(void) const
    {return GetScale()*tile_w;}
  float GetScaledTileHeight(void) const
    {return GetScale()*tile_h;}
  float GetScale(void) const;

  int GetTileX(int mousex, int mousey) const;
  int GetTileY(int mousex, int mousey) const;

  void Rect(int i1=-1, int j1=-1, int i2=-1, int j2=-1);
  void Circle(int i1=-1, int j1=-1, int i2=-1, int j2=-1);

  void Unzoom(void);
  void ZoomIn(void);
  void ZoomOut(void);

  void Redraw(void) const;

protected:
  void Init(DIALOG *d);

  const char *HelpText() const;

  void Draw(void) const;
  void PrintCurrentTileValue(int mx, int my) const;

  void (*ChooseDrawDotted(void) const)(BITMAP*,int,int,int,int);
  void (*ChooseDrawGrid(void) const)(BITMAP*,int,int,int,int);

private:
  MapDialog(const MapDialog&);
  MapDialog &operator=(const MapDialog&);

  int GetDX(void) const;
  int GetDY(void) const;
  void DrawRect(void) const;
  void DrawDiamond(void) const;
  void UpdateScrollbars(void) const;

  DIALOG *box;   /* Where to draw the map */
  DIALOG *h_sb;  /* Horizontal scrollbar */
  DIALOG *v_sb;  /* Vertical scrollbar */

  BITMAP *map_bitmap;    /* Draw in memory and blit, so there is no flicker */
  int tile_dx;           /* "Theorical" tile width */
  int tile_dy;           /* "Theorical" tile height */
  int tile_odd;          /* Tile dx for odd rows */
  int scale;             /* Scale value */
  MAP_SHAPE map_shape;   /* Diamond or Rectangle */
  TILE_SHAPE tile_shape; /* Diamond or Rectangle [or Hexagon] */
  int tile_w;            /* Actual tile width */
  int tile_h;            /* Actual tile height */

  int draw_grid; /* True if the grid must be drawn */

  static int MapProc(int msg, DIALOG *d, int c);
};

#endif /* EME__MAP_DIALOG__ */

