/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__MAP__
#define EME__MAP__
/*----------------------------------------------------------------------------
  A map

- Constructor/destructor
- Miscellanei
- Layers
- Active layer (the one that is currently worked on)
- Properties
- Active property (the one that is currently worked on)
- Internal data
----------------------------------------------------------------------------*/

#include "debug.h"

struct BITMAP;
class Entry;
class SelectedTiles;
#include "tstack.h"
#include "ccomposi.h"
#include "config.inc"
#include <vector>
#include "selected.h"
#include "commands.h"
#include "lviewed.h"

class Map {

public:
  /* Constructor/destructor
     --------------------*/
  Map(
    int w, int h,             /* Map size (int number of tiles) */
    int num_layers            /* Number of layers */
  );

  Map(int w, int h);

  ~Map(void);


  /* Miscellanei
     --------------------*/
  int GetWidth(void) const /* Returns the map width (in number of tiles) */
    { return layers_.width(); }

  int GetHeight(void) const /* Returns the map height (in number of tiles) */
    { return layers_.height(); }

  bool IsIn( /* Returns true if the coords are in the map */
    int i, int j              /* Coords (in tiles) */
  ) const
    { return layers_.is_in(i, j); }

  void SetUserdata(void *userdata, void (*destroyer)(void *userdata)=0) {
    if (userdata_ && userdata_destroyer_) {
      userdata_destroyer_(userdata_);
    }
    userdata_=userdata;
    userdata_destroyer_=destroyer;
  }
  void *GetUserdata() const
    { return userdata_; }


  /* Layers
     --------------------*/
  const TilesStack *GetLayers() const { return &layers_; }
  TilesStack *GetLayers() { return &layers_; }

  int SetLayer( /* Sets a layer, n must be < num layers */
    int n,                    /* Layer index */
    Tiles *layer              /* Layer to add */
  );

  Tiles *RemoveLayer(int l);
  void InsertLayer(int l, Tiles *layer);

  int GetNumLayers(void) const /* Returns the number of layers */
    { return layers_.depth(); }

  const ViewedLayers &GetViewedLayers() const
    { return viewed_layers_; }

  const Tiles *GetLayer(int l) const /* Returns the nth layer */
    { return layers_.get(l); }
  Tiles *GetLayer(int l) /* Returns the nth layer */
    { return layers_.get(l); }

  /* Returns the index of the first layer with name 'name' or GetNumLayers() if
   * not found */
  int FindLayerIndex(const char *name) const;

  const char *GetLayerName(int l) const; /* Returns the nth layer name */


  /* Active layer (the one that is currently worked on)
     --------------------*/
  const Tiles *GetActiveLayer() const; /* Returns the active layer */
  Tiles *GetActiveLayer(); /* Returns the active layer */

  int GetActiveLayerIndex(void) const; /* Returns the active layer index */

  const char *GetActiveLayerName(void) const; /*Returns the active layer name*/

  void SetActiveLayer( /* Changes the active layer */
    int n,                    /* New active layer index */
    Entry *e                  /* Entry to update */
  );
  void SetActiveLayer(int l);

  void SetViewedLayers(const ViewedLayers *v);

  void UpdateActiveLayer(
    Entry *e
  );

  void IncrActiveLayer( /* Incr the active layer index */
    Entry *e                  /* Entry to update */
  );

  void DecrActiveLayer( /* Decr the active layer index */
    Entry *e                  /* Entry to update */
  );


  /* Reference properties (i.e. default value for layers)
     --------------------*/
  const BaseProperty *GetLayerReference(int l) const;
  const BaseProperty *GetActiveLayerReference() const
    { return GetLayerReference(GetActiveLayerIndex()); }


  /* Properties
     --------------------*/
  void SetProperties( /* Set the list of map properties */
    Composite::Creator *propcreator     /* */
  );

  Composite::Creator *GetProperties(void) const
    {return propcreator;}

  int GetNumProperties(void) const; /* Returns the number of properties */

  const char *GetPropertyName(int n) const; /* Returns the nth property name */

  const BaseProperty *GetProperty(int n) const;


  /* Active property (the one that is currently worked on)
     --------------------*/
  void SetActiveProperty(
    int prop,                 /* New active property index */
    Entry *e                  /* Entry to update */
  );

  void UpdateActiveProperty(
    Entry *e
  );


  const SelectedTiles *GetSelectedTiles() const
    { return &selected_tiles_; }
  SelectedTiles *GetSelectedTiles()
    { return &selected_tiles_; }

  const Commands *GetCommandsManager() const
    { return &commands_manager_; }
  Commands *GetCommandsManager()
    { return &commands_manager_; }

protected:


private:
  Map(const Map&);
  Map &operator=(const Map&);

  int width, height;          /* Map size (in tiles) */
  TilesStack layers_;
  int active_layer;           /* Active layer index */
  ViewedLayers viewed_layers_;
  Composite::Creator *propcreator;   /* */
  int active_property;        /* Active property index */

  void *userdata_;
  void (*userdata_destroyer_)(void *userdata);
  SelectedTiles selected_tiles_;
  Commands commands_manager_;
};

#endif /* EME__MAP__ */

