/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"
#include "alclean.h"
#include "globals.h"
#include "mapdlg.h"
#include "map.h"
#include "gui.h"
#include "commands.h"

#include "help.h"

#include "plugin.h"
#include "settheme.h"
#include "font.h"


/* For memory pool initialization */
#include "prop.h"

#include <stdlib.h>
#include <stdio.h>
#include <allegro.h>
#include <string.h>


#define HELP_BEGIN \
  "eme, version " EME_VERSION_STR ", Copyright " EME_COPYRIGHT "\n" \
  "\n" EME_INFO

#define HELP_CORE \
  "\n\n" \
  "$ eme --help\n" \
  "$ eme <filename> OPTIONS\n" \
  "$ eme --file=<filename> OPTIONS\n" \
  "\n" \
  "OPTIONS can be:\n" \
  "--help              This help\n" \
  "--file=<filename>   Loads or create filename\n" \
  "--cdepth=<number>   Color depth (8, 15, 16, 24, 32) [default 16]\n" \
  "--width=<number>    Screen width [default 640]\n" \
  "--height=<number>   Screen height [default 480]\n" \
  "--undo=<number>     Maximum undo levels [default 10]\n" \
  "--save              Just save after loading (filename must be specified;\n" \
  "                      destroys the previous map)\n" \
  ""

#define HELP_STR   "--help"
#define FILE_STR   "--file="
#define WIDTH_STR  "--width="
#define HEIGHT_STR "--height="
#define CDEPTH_STR "--cdepth="
#define UNDO_STR   "--undo="
#define SAVE_STR   "--save"

/* eme section in config file */
#define EME_SECTION UString("[eme]")

/* All sections in config file */
#define CONFIG_SCREEN_WIDTH  UString("screen_width")
#define CONFIG_SCREEN_HEIGHT UString("screen_height")
#define CONFIG_SCREEN_CDEPTH UString("screen_cdepth")
#define CONFIG_UNDO_LEVELS   UString("undo_levels")
#define CONFIG_THEME         UString("theme")
#define CONFIG_FG_COLOR      UString("fg_color")
#define CONFIG_BG_COLOR      UString("bg_color")
#define CONFIG_GRID          UString("grid")
#define CONFIG_SCREEN_TYPE   UString("screen_type")
#define CONFIG_FONT_TYPE     UString("font")
#define CONFIG_HELP_FILE     UString("help_file")

/* The default values given to get_config_string cannot simply be an oject
 * constructed on the stack because get_config_string may return the default
 * string */
#define SCREEN_AUTODETECT            "none"
#define SCREEN_AUTODETECT_FULLSCREEN "fullscreen"
#define SCREEN_AUTODETECT_WINDOWED   "windowed"

#define FONT_TYPE_SMALL "small"
#define FONT_TYPE_BIG   "big"

/* Default values for the config */
#define DEFAULT_SCREEN_TYPE SCREEN_AUTODETECT
#define DEFAULT_WIDTH  -1
#define DEFAULT_HEIGHT -1
#define DEFAULT_CDEPTH -1
#define DEFAULT_UNDO   32
#define DEFAULT_SAVE   0
#define DEFAULT_THEME    GetDefaultThemeName()
#define DEFAULT_FG_COLOR "32 32 32"
#define DEFAULT_BG_COLOR "192 192 192"
#define DEFAULT_GRID 1
#define DEFAULT_FONT_TYPE FONT_TYPE_SMALL
#define DEFAULT_FONT_SIZE FONT_SIZE_SMALL
#define DEFAULT_HELP_FILE "emehelp.en"


/* Ascii messages: used before Allegro is installed */
#define NO_UNICODE_SUPPORT_ERROR() \
  "Error: not compiled with unicode support\n"

#define ALLEGRO_INIT_ERROR(str) \
  "Error: unable to initialize Allegro:\n  %s\n", str

/* Converted messages */
#define SEVERAL_MAP_ERROR() \
  Translation("Error: you can specify only one map file")

#define INSTALL_MOUSE_ERROR(str) \
  Translation("Error: unable to install mouse: %s"), str

#define INSTALL_KEYBOARD_ERROR(str) \
  Translation("Error: unable to install keyboard: %s"), str

#define INSTALL_GFX_ERROR() \
  Translation("Error: unable to initialize a graphic mode")

#define PLUGIN_INIT_ERROR(str) \
  Translation("Error: unable to initialize plugin: %s"), str

#define NO_MAP_ERROR() \
  Translation("Error: no map loaded")


static int screen_type;

void save_config(void)
{
  set_config_int(EME_SECTION, CONFIG_SCREEN_WIDTH, GUI.GetWidth());
  set_config_int(EME_SECTION, CONFIG_SCREEN_HEIGHT, GUI.GetHeight());
  set_config_int(EME_SECTION, CONFIG_SCREEN_CDEPTH, bitmap_color_depth(screen));
  set_config_int(
    EME_SECTION, CONFIG_UNDO_LEVELS, Commands::GetMaxUndoLevels()
  );
  set_config_int(EME_SECTION, CONFIG_GRID, GUI.IsGridDrawn());

  if (font_is_small(font)) {
    set_config_string(EME_SECTION, CONFIG_FONT_TYPE, UString(FONT_TYPE_SMALL));
  }
  else {
    set_config_string(EME_SECTION, CONFIG_FONT_TYPE, UString(FONT_TYPE_BIG));
  }

  set_config_string(EME_SECTION, CONFIG_THEME, GetThemeName());
  char str[(STRING_INT_LENGTH+1) * 3];
  uszprintf(
    str, sizeof(str), UString("%d %d %d"),
    getr(gui_fg_color), getg(gui_fg_color), getb(gui_fg_color)
  );
  set_config_string(EME_SECTION, CONFIG_FG_COLOR, str);
  uszprintf(
    str, sizeof(str), UString("%d %d %d"),
    getr(gui_bg_color), getg(gui_bg_color), getb(gui_bg_color)
  );
  set_config_string(EME_SECTION, CONFIG_BG_COLOR, str);

  switch(screen_type) {
    case GFX_AUTODETECT:
      set_config_string(EME_SECTION, CONFIG_SCREEN_TYPE, UString(SCREEN_AUTODETECT));
    break;
    case GFX_AUTODETECT_FULLSCREEN:
      set_config_string(
        EME_SECTION, CONFIG_SCREEN_TYPE, UString(SCREEN_AUTODETECT_FULLSCREEN)
      );
    break;
    case GFX_AUTODETECT_WINDOWED:
      set_config_string(
        EME_SECTION, CONFIG_SCREEN_TYPE, UString(SCREEN_AUTODETECT_WINDOWED)
      );
    break;
  }
}


void load_config(void)
{
  static const UString config_fname("eme.cfg");
  char *exe_path = al_clean_get_executable_name();

  { /* Load config file */
    int len = ustrsizez(exe_path) + ustrsizez(config_fname);
    char *config;
    Malloc(config, len);
    replace_filename(config, exe_path, config_fname, len);
    override_config_file(config);
    Free(config);
  }

  { /* Load help file */
    UString default_help(DEFAULT_HELP_FILE);
    const char *help_fname =
      get_config_string(EME_SECTION, CONFIG_HELP_FILE, default_help);
    int len = ustrsizez(exe_path) + ustrsizez(help_fname);
    char *help;
    Malloc(help, len);
    replace_filename(help, exe_path, help_fname, len);
    load_help(help);
    Free(help);
  }

  free(exe_path);
}


int get_color(const char *ground, const char *def_value)
{
  const char *color=get_config_string(EME_SECTION, ground, def_value);
  int r, g, b;
  const char *end = color;
  r = ustrtol(end, (char**)&end, 10);
  g = ustrtol(end, (char**)&end, 10);
  b = ustrtol(end, (char**)&end, 10);
  return makecol(r, g, b);
}


const char *get_error_string(int err)
{
  /* Create them here, so it is possible to have a translation, because they
     are created after allegro_init is called */
  static const Translation error_unknown(LS_ERROR_UNKNOWN_STR);
  static const Translation error_mem(LS_ERROR_MEM_STR);
  static const Translation error_file(LS_ERROR_FILE_STR);
  static const Translation error_read(LS_ERROR_READ_STR);
  static const Translation error_write(LS_ERROR_WRITE_STR);
  static const Translation error_option(LS_ERROR_OPTION_STR);
  static const Translation error_invalid(LS_ERROR_INVALID_STR);

  const char *err_str = error_unknown;
  switch (err) {
    case LS_ERROR_MEM:
      err_str = error_mem;
    break;
    case LS_ERROR_FILE:
      err_str = error_file;
    break;
    case LS_ERROR_READ:
      err_str = error_read;
    break;
    case LS_ERROR_WRITE:
      err_str = error_write;
    break;
    case LS_ERROR_OPTION:
      err_str = error_option;
    break;
    case LS_ERROR_INVALID:
      err_str = error_invalid;
    break;
  }
  return err_str;
}


int autodetect_screen(int type, int *width, int *height, int *depth)
{
  static int screen_widths[] = { 1600, 1280, 1024, 800, 640, 320, -1 };
  static int screen_heights[] = { 1200, 1024, 768, 600, 480, 240, -1 };
  static int screen_depthes[] = { 32, 24, 16, 15, 8, -1 };

  int w = *width;
  int h = *height;
  int d = *depth;
  int n;
  for (n=0; screen_depthes[n]; ++n) {
    if (d==screen_depthes[n]) break;
  }
  d=screen_depthes[n];
  for (int n=0; screen_widths[n]>0; n++) {
    *width = w<=0 ? screen_widths[n] : w;
    *height = h<=0 ? screen_heights[n] : h;
    for (int nn=0; screen_depthes[nn]>0; nn++) {
      *depth = d<=0 ? screen_depthes[nn] : d;

      set_color_depth(*depth);
      int err = set_gfx_mode(type, *width, *height, 0, 0);
      if (err==0) {
        return 0;
      }
    }
  }
  set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
  return -1;
}


int initialize_screen(int *asked_type, int *asked_width, int *asked_height, int *asked_depth)
{
  int err;
  int type = *asked_type;
  int width = *asked_width;
  int height = *asked_height;
  int depth = *asked_depth;
  err = autodetect_screen(type, &width, &height, &depth);
  if (err<0 && type!=GFX_AUTODETECT) {
    type = GFX_AUTODETECT;
    err = autodetect_screen(type, &width, &height, &depth);
  }
  if (err<0) {
    width = -1;
    height = -1;
    depth = -1;
    err = autodetect_screen(type, &width, &height, &depth);
  }
  if(err<0) {
    allegro_message(INSTALL_GFX_ERROR());
    allegro_message(NewLineString());
    return -1;
  }
  if (type!=*asked_type || width!=*asked_width || height!=*asked_height || depth!=*asked_depth) {
    *asked_type = type;
    *asked_width = width;
    *asked_height = height;
    *asked_depth = depth;
  }
  RGB palette332[256];
  generate_332_palette(palette332);
  set_palette(palette332);
  return 0;
}


#ifdef DEBUG
#include "cbool.h"
#include "ccolor.h"
#include "ccomposi.h"
#include "cdataf.h"
#include "clist.h"
#include "cnumber.h"
#include "cpercent.h"
#include "cstring.h"
#endif
int main(int argc, char **argv)
{
  int width=DEFAULT_WIDTH;
  int height=DEFAULT_HEIGHT;
  int cdepth=DEFAULT_CDEPTH;
  int undo=DEFAULT_UNDO;
  int save=DEFAULT_SAVE;
  FONT_SIZE font_size=DEFAULT_FONT_SIZE;
  char *file=0;
  int plugin_argc = 1;
  const char **plugin_argv;
  Malloc(plugin_argv, argc);

  MEMORY_POOL_INIT(BaseProperty);
#ifdef DEBUG
  fprintf(stderr, "sizeof(Boolean::Property):        %d\n", sizeof(Boolean::Property));
  fprintf(stderr, "sizeof(Number::Property):         %d\n", sizeof(Number::Property));
  fprintf(stderr, "sizeof(Percent::Property):        %d\n", sizeof(Percent::Property));
  fprintf(stderr, "sizeof(String::Property):         %d\n", sizeof(String::Property));
  fprintf(stderr, "sizeof(List::Property):           %d\n", sizeof(List::Property));
  fprintf(stderr, "sizeof(Composite::Property):      %d\n", sizeof(Composite::Property));
  fprintf(stderr, "sizeof(DatafileObject::Property): %d\n", sizeof(DatafileObject::Property));
  fprintf(stderr, "sizeof(Color::Property):          %d\n", sizeof(Color::Property));
#endif

  /* Allegro Initialization */
  int encoding = plugin_encoding();
  UString::InitEncoding(encoding);
  int err;
  err = allegro_init();
  if(err) {
    fprintf(stderr, ALLEGRO_INIT_ERROR(allegro_error));
    return 1;
  }

  load_config();

  width = get_config_int(EME_SECTION, CONFIG_SCREEN_WIDTH, width);
  height = get_config_int(EME_SECTION, CONFIG_SCREEN_HEIGHT, height);
  int desktop_cdepth = desktop_color_depth();
  if (desktop_cdepth) {
    cdepth = desktop_cdepth;
  }
  else {
    cdepth = get_config_int(EME_SECTION, CONFIG_SCREEN_CDEPTH, cdepth);
  }

  undo = get_config_int(EME_SECTION, CONFIG_UNDO_LEVELS, undo);

  {
    UString default_screen(DEFAULT_SCREEN_TYPE);
    UString fullscreen(SCREEN_AUTODETECT_FULLSCREEN);
    UString windowed(SCREEN_AUTODETECT_WINDOWED);
    const char *screen_type_name =
      get_config_string(EME_SECTION, CONFIG_SCREEN_TYPE, default_screen);
    if(ustricmp(screen_type_name, windowed) == 0) {
      screen_type = GFX_AUTODETECT_WINDOWED;
    }
    else if(ustricmp(screen_type_name, fullscreen) == 0) {
      screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    else {
      screen_type = GFX_AUTODETECT;
    }
  }

  {
    UString default_font(DEFAULT_FONT_TYPE);
    UString big(FONT_TYPE_BIG);
    const char *font_type_name =
      get_config_string(EME_SECTION, CONFIG_FONT_TYPE, default_font);
    if (ustricmp(font_type_name, big) == 0) {
      font_size = FONT_SIZE_BIG;
    }
  }

  /* Set the executable name */
  plugin_argv[0] = argv[0];

  /* Command line options */
  for(int i=1; i<argc; i++) {
    if(strncmp(argv[i], WIDTH_STR, strlen(WIDTH_STR))==0) {
      width = atoi(argv[i]+strlen(WIDTH_STR));
    }
    else if(strncmp(argv[i], FILE_STR, strlen(FILE_STR))==0) {
      if(!file) {
        file = argv[i]+strlen(FILE_STR);
      }
      else {
        allegro_message(SEVERAL_MAP_ERROR());
        allegro_message(NewLineString());
        return 1;
      }
    }
    else if(strncmp(argv[i], HEIGHT_STR, strlen(HEIGHT_STR))==0) {
      height = atoi(argv[i]+strlen(HEIGHT_STR));
    }
    else if(strncmp(argv[i], CDEPTH_STR, strlen(CDEPTH_STR))==0) {
      cdepth = atoi(argv[i]+strlen(CDEPTH_STR));
    }
    else if(strncmp(argv[i], UNDO_STR, strlen(UNDO_STR))==0) {
      undo = atoi(argv[i]+strlen(UNDO_STR));
    }
    else if(strcmp(argv[i], SAVE_STR)==0) {
      save = 1;
    }
    else if(strcmp(argv[i], HELP_STR)==0) {
      allegro_message(UString(HELP_BEGIN));
      allegro_message(Translation(HELP_CORE));
      allegro_message(UString("%s"), UString(plugin_help()).string());
      return 0;
    }
    else {
      if(argv[i][0] == '-') {
        plugin_argv[plugin_argc] = argv[i];
        plugin_argc++;
      }
      else if(!file) {
        file=argv[i];
      }
      else {
        allegro_message(SEVERAL_MAP_ERROR());
        allegro_message(NewLineString());
        return 1;
      }
    }
  } /* for(int i=1; i<argc; i++) */

  initialize_screen(&screen_type, &width, &height, &cdepth);

  /* Allegro I/O Initializations */
  if (!save) {
    /* install_mouse should be done after screen init so the mouse pointer
     * colors are ok */
    err = install_mouse();
    if(err<0) {
      allegro_message(INSTALL_MOUSE_ERROR(allegro_error));
      allegro_message(NewLineString());
      return 1;
    }
    err = install_keyboard();
    if(err) {
      allegro_message(INSTALL_KEYBOARD_ERROR(allegro_error));
      allegro_message(NewLineString());
      return 1;
    }
  } /* if (!save) */

  if(undo<0) {
    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
    allegro_message(ERROR_BAD_UNDO_LEVEL_STR);
    allegro_message(NewLineString());
    return 1;
  }
  Commands::ChangeMaxUndoLevels(undo);

  if (!save) {
    /* TODO: Toggle small/big font */
    font = get_font(font_size);

    {
      UString default_fg(DEFAULT_FG_COLOR);
      UString default_bg(DEFAULT_BG_COLOR);
      GUI.SetTheme(
        get_config_string(EME_SECTION, CONFIG_THEME, DEFAULT_THEME),
        get_color(CONFIG_FG_COLOR, default_fg.string()),
        get_color(CONFIG_BG_COLOR, default_bg.string())
      );
    }

    /* GUI Initialization */
    GUI.SetSize(width, height);

    GUI.AllowFeature(false, Features::Group::Everything);

    if(get_config_int(EME_SECTION, CONFIG_GRID, DEFAULT_GRID)) {
      GUI.ShowGrid();
    }
    else {
      GUI.HideGrid();
    }
  } /* if (!save) */

  err = plugin_init(plugin_argc, plugin_argv);
  if(err!=LS_NO_ERROR) {
    const char *err_str = get_error_string(err);
    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
    allegro_message(PLUGIN_INIT_ERROR(err_str));
    allegro_message(NewLineString());
    return 1;
  }

  if(file) {
    int err;
    int create_map = ! exists(UString(file));
    if(create_map) {
      PROFILE_BEGIN(plugin_new)
      err = plugin_new();
      PROFILE_END(plugin_new)
    }
    else {
      PROFILE_BEGIN(plugin_load)
      err = plugin_load(UString(file));
      PROFILE_END(plugin_load)
    }
    if (err) {
      const char *err_str = get_error_string(err);
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      if(create_map) {
        allegro_message(ERROR_CREATING_STR);
        allegro_message(NewLineString());
        allegro_message(UString(file));
        allegro_message(NewLineString());
        allegro_message(err_str);
        allegro_message(NewLineString());
      }
      else {
        allegro_message(ERROR_READING_STR);
        allegro_message(NewLineString());
        allegro_message(UString(file));
        allegro_message(NewLineString());
        allegro_message(err_str);
        allegro_message(NewLineString());
      }
      return 1;
    }
    GUI.SetPath(UString(file));
  } /* if (file) */

  if(save) {
    if(!file) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message(NO_MAP_ERROR());
      allegro_message(NewLineString());
    }
    /* Just save */
    PROFILE_BEGIN(plugin_save)
    int err = plugin_save(UString(file), GUI.GetMap());
    PROFILE_END(plugin_save)
    if (err) {
      const char *err_str = get_error_string(err);
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message(ERROR_WRITING_STR);
      allegro_message(NewLineString());
      allegro_message(UString(file));
      allegro_message(NewLineString());
      allegro_message(err_str);
      allegro_message(NewLineString());
      return 1;
    }
  }
  else {
    /* Main Loop */
    GUI.SetWindowTitle();
    GUI.DoDialog ();
  }

  /* Force cleanup or MemoryPool is unhappy */
  GUI.ClearClipboard();
  GUI.ClearBrush();
  GUI.SetMap(0);

  unload_help();

  plugin_exit();

  Free(plugin_argv);

  set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);

  MEMORY_POOL_EXIT(BaseProperty);

  LOGEXEC();

  return 0;
}
END_OF_MAIN();

