/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "lutils.h"

#include "entry.h"
#include "prop.h"
#include "creator.h"
#include "selected.h"
#include "tiles.h"
#include "stiles.h"
#include "flood.h"
#include "lviewed.h"
#include "tstack.h"

#include "debug.h"

#include <math.h>


namespace LayerUtils {

/* Displays the layer on a bitmap */
void DrawRect(
  const Tiles *tiles,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale,
  int l                     /* Layer index */
)
{
  int i, j; /* tile indices */
  int x, y; /* tile position in pixels */

  const int last_i = MIN(first_i+num_i, tiles->endi()-1);
  const int last_j = MIN(first_j+num_j, tiles->endj()-1);

  const int dx = (int) (tile_odd * tile_w);

  const BaseCreator *c = tiles->creator();

  for (j=first_j, y=0; j<=last_j; j++, y+=tile_h) {
    for (i=first_i, x=0; i<=last_i; i++, x+=tile_w) {
      if (tiles->get(i, j)) {
        c->Draw(tiles->get(i, j), bmp, x+(j&1)*dx, y, tile_w, tile_h, scale, l, i, j);
      }
    }
  }
}


/* Displays the layer on a bitmap */
void DrawDiamond(
  const Tiles *tiles,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale,
  int l                     /* Layer index */
)
{
  int i, j; /* tile indices */

  const int dx = tile_w * first_i + tile_w/2;
  const int dy = tile_h * first_j + tile_h/2;

  const int w = num_i*tile_w; /* width of the display */
  const int h = num_j*tile_h; /* height of the display */

  const BaseCreator *c = tiles->creator();

  const int width = tiles->width();
  const int height = tiles->height();
  for (j=0; j<height; ++j) {
    for (i=0; i<width; ++i) {
      const int x = -dx + tile_w * (height+i-j) / 2;
      const int y = -dy + tile_h * (i+j+1) / 2;
      /* if the position is on the display */
      if (x+tile_w>=0 && y+tile_h>=0 && x<=w && y<=h) {
        if (tiles->get(i, j)) {
          c->Draw(tiles->get(i, j), bmp, x, y, tile_w, tile_h, scale, l, i, j);
        }
      }
    }
  }
}


/* Displays the layer stack on a bitmap */
void DrawRect(
  const TilesStack *layers,
  const ViewedLayers *viewed_layers,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale
)
{
  for (int n=0; n<layers->depth(); ++n) {
    if (viewed_layers->at(n) && layers->get(n)) {
      DrawRect(
        layers->get(n), bmp, first_i, first_j, num_i, num_j,
        tile_w, tile_h, tile_odd, scale, n
      );
    }
  }
}


/* Displays the layer stack on a bitmap */
void DrawDiamond(
  const TilesStack *layers,
  const ViewedLayers *viewed_layers,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale
)
{
  for (int n=0; n<layers->depth(); ++n) {
    if (viewed_layers->at(n) && layers->get(n)) {
      DrawDiamond(
        layers->get(n), bmp, first_i, first_j, num_i, num_j,
        tile_w, tile_h, tile_odd, scale, n
      );
    }
  }
}




/* Selections
   ---------------------------------*/

template <typename T>
SelectedTiles *select_(const Tiles *tiles, int i1, int j1, int i2, int j2, T test)
{
  SelectedTiles *s = new SelectedTiles();

  if (s) {
    for (int j=j1; j<=j2; ++j) {
      for (int i=i1; i<=i2; ++i) {
        if (test(tiles, i, j)) {
          s->Add(i, j);
        }
      }
    }
  }

  return s;
}

static inline bool is_empty_(const Tiles *tiles, int i, int j)
{
  return !tiles->get(i, j);
}

static inline bool exists_(const Tiles *tiles, int i, int j)
{
  return tiles->get(i, j);
}


class in_circle_ {
  int i_, j_, r2_;
public:
  in_circle_(int i, int j, int r2): i_(i), j_(j), r2_(r2) {}
  bool operator()(const Tiles *tiles, int i, int j) {
    return tiles->get(i, j) && (i-i_)*(i-i_)+(j-j_)*(j-j_)<=r2_;
  }
};


class is_equal_ {
  const BaseProperty *p_;
public:
  is_equal_(const BaseProperty *p): p_(p) {}
  bool operator()(const Tiles *tiles, int i, int j) {
    return tiles->get(i, j) && p_->IsEqualTo(tiles->get(i, j));
  }
};


/* Select all the existing tiles */
SelectedTiles *SelectAll(const Tiles *tiles)
{
  return SelectRect(tiles, tiles->begini(), tiles->beginj(), tiles->endi(), tiles->endj());
}


/* Returns a SelectedTiles object with all the empty tiles in 'sel' */
SelectedTiles *SelectEmpty(
  const Tiles *tiles,
  const SelectedTiles *sel
)
{
  SelectedTiles *s = new SelectedTiles();
  if (s) {
    for (SelectedTiles::const_iterator t=sel->Begin(); t!=sel->End(); ++t) {
      if (!tiles->get(sel->GetX(t), sel->GetY(t))) {
        s->Add(sel->GetX(t), sel->GetY(t));
      }
    }
  }
  return s;
}


SelectedTiles *SelectRect(
  const Tiles *tiles,
  int i1, int j1,           /* Top left corner indices (inclusive) */
  int i2, int j2            /* Bottom right corner (inclusive) */
)
{
  return select_(tiles, i1, j1, i2, j2, exists_);
}


SelectedTiles *SelectCircle(
  const Tiles *tiles,
  int i, int j,             /* Circle center indices */
  int r                     /* Circle radius (in tiles, inclusive) */
)
{
  int r2 = r*r;
  int i1 = MAX(tiles->begini(), i-r);
  int j1 = MAX(tiles->beginj(), j-r);
  int i2 = MIN(tiles->endi(), i+r);
  int j2 = MIN(tiles->endj(), j+r);
  return select_(tiles, i1, j1, i2, j2, in_circle_(i, j, r2));
}


SelectedTiles *SelectByProperty(
  const Tiles *tiles,
  int i, int j              /* Reference tile indices */
)
{
  const BaseProperty *ref = tiles->get(i, j);
  if (ref) {
    return select_(tiles, tiles->begini(), tiles->beginj(), tiles->endi(), tiles->endj(), is_equal_(ref));
  }
  else {
    return new SelectedTiles();
  }
}

SelectedTiles *SelectByWand(
  const Tiles *tiles,
  int i, int j    /* Reference tile indices */
)
{
  return FloodFill(tiles, i, j);
}


SelectedTiles *Intersection(const SelectedTiles *s1, const SelectedTiles *s2)
{
  SelectedTiles *ret = new SelectedTiles();
  if (ret) {
    SelectedTiles::const_iterator id;
    for (id=s1->Begin(); id!=s1->End(); ++id) {
      if (s2->IsIn(s1->GetX(id), s1->GetY(id))) {
        ret->Add(s1->GetX(id), s1->GetY(id));
      }
    }
  }
  return ret;
}


SelectedTiles *GrowSelection(const SelectedTiles *sel)
{
  SelectedTiles *ret = new SelectedTiles();
  if (ret) {
    for (
      SelectedTiles::iterator id=sel->Begin();
      id!=sel->End();
      ++id
    ) {
      ret->Add(sel->GetX(id)+0, sel->GetY(id)+1);
      ret->Add(sel->GetX(id)+0, sel->GetY(id)-1);
      ret->Add(sel->GetX(id)+1, sel->GetY(id)+0);
      ret->Add(sel->GetX(id)-1, sel->GetY(id)+0);
    }
    ret->Add(sel);
  }
  return ret;
}


SelectedTiles *ShrinkSelection(const SelectedTiles *sel, const Tiles *tiles)
{
  SelectedTiles *ret = 0;
  SelectedTiles *s1 = InvertSelection(sel, tiles);
  if (s1) {
    SelectedTiles *s2 = GrowSelection(s1);
    if (s2) {
      ret = InvertSelection(s2, tiles);
      delete s2;
    }
    delete s1;
  }
  return ret;
}


SelectedTiles *InvertSelection(const SelectedTiles *sel, const Tiles *tiles)
{
  SelectedTiles *ret = SelectAll(tiles);
  if (ret) {
    ret->Remove(sel);
  }
  return ret;
}



/* Copy/cut/paste
   ---------------------------------*/
SparseTiles *Copy(
  const Tiles *tiles,
  const SelectedTiles *sel      /* Which tiles to copy */
)
{
  SparseTiles *ret = new SparseTiles(tiles->creator()->Clone());

  if (ret) {
    const SelectedTiles::const_iterator last = sel->End();
    for (SelectedTiles::const_iterator t=sel->Begin(); t!=last; ++t) {
      const int i = sel->GetX(t);
      const int j = sel->GetY(t);
      const BaseProperty *p = tiles->get(i, j);
      if (p) {
        DBG_ASSERT(!ret->get(i, j));
        ret->set(i, j, p->Clone());
      }
    }
  }

  return ret;
}


TilesStack *Copy(
  const TilesStack *layers,
  const ViewedLayers *viewed,
  const SelectedTiles *sel
)
{
  int count = layers->depth()<viewed->count() ? layers->depth() : viewed->count();
  TilesStack *ret = new TilesStack();

  for (int l=0; l<count; ++l) {
    const Tiles *layer = layers->get(l);
    if (layer && viewed->at(l)) {
      DBG_ASSERT(!ret->get(l));
      Tiles *new_layer = Copy(layer, sel);
      ret->set(l, new_layer);
    }
  }

  return ret;
}



void Fill(
  Tiles *tiles,
  const SelectedTiles *sel,     /* Which tiles to copy */
  const BaseProperty *value       /* Which value to paste */
)
{
  const SelectedTiles::const_iterator last = sel->End();
  for (SelectedTiles::const_iterator t=sel->Begin(); t!=last; ++t) {
    const int i = sel->GetX(t);
    const int j = sel->GetY(t);
    if (tiles->is_in(i, j)) {
      tiles->copy(i, j, value);
    }
  }
}


void Fill(
  Tiles *tiles,
  int i, int j,
  const BaseProperty *value
)
{
  if (tiles->is_in(i, j)) {
    tiles->copy(i, j, value);
  }
}


void Cut(
  Tiles *tiles,
  const SelectedTiles *sel      /* Which tiles to cut */
)
{
  const SelectedTiles::const_iterator last = sel->End();
  for (SelectedTiles::const_iterator t=sel->Begin(); t!=last; ++t) {
    const int i = sel->GetX(t);
    const int j = sel->GetY(t);
    tiles->clear(i, j);
  }
}


void Paste(
  Tiles *tiles,
  int i, int j,             /* Where to paste (top left corner, inclusive) */
  const SparseTiles *to_paste    /* Which tiles to paste */
)
{
  /* can copy only if are the same type */
  if (tiles->creator()->IsSameType(to_paste->creator()->GetReference())) {
    for (SparseTiles::const_iterator t=to_paste->begin(); t!=to_paste->end(); ++t) {
      int ti = to_paste->get_i(t) - to_paste->begini() + i;
      int tj = to_paste->get_j(t) - to_paste->beginj() + j;
      if (tiles->is_in(ti, tj)) {
        const BaseProperty *p = to_paste->get_tile(t);
        if (p) {
          tiles->copy(ti, tj, p);
        }
      }
    }
  }
}


void Paste(
  TilesStack *layers,
  int i, int j,
  const TilesStack *to_paste
)
{
  int count = layers->depth()<to_paste->depth() ? layers->depth() : to_paste->depth();
  for (int l=0; l<count; ++l) {
    Tiles *to = layers->get(l);
    const SparseTiles *from = to_paste->get<SparseTiles>(l);
    if (to && from) {
      Paste(to, i, j, from);
    }
  }
}



void Paste(
  Tiles *tiles,
  int i, int j,             /* Where to paste */
  const Entry *e            /* Which value to paste */
)
{
  BaseProperty *p = tiles->get(i, j);
  if (p) {
    tiles->creator()->UpdateProperty(p, e);
  }
}



/* Insert/Delete row/column
   ---------------------------------*/
void Move(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int di, int dj
)
{
  for (int l=0; l<layers->depth(); ++l) {
    if (viewed->at(l)) {
      Tiles *layer = layers->get(l);
      if (layer) {
        layer->move(di, dj);
      }
    }
  }
}


void InsertColumns(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
)
{
  for (int l=0; l<layers->depth(); ++l) {
    if (viewed->at(l)) {
      Tiles *layer = layers->get(l);
      if (layer) {
        layer->insert_col(i, count);
      }
    }
  }
  layers->update();
}


void InsertRows(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
)
{
  for (int l=0; l<layers->depth(); ++l) {
    if (viewed->at(l)) {
      Tiles *layer = layers->get(l);
      if (layer) {
        layer->insert_row(j, count);
      }
    }
  }
  layers->update();
}


void DeleteColumns(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
)
{
  for (int l=0; l<layers->depth(); ++l) {
    if (viewed->at(l)) {
      Tiles *layer = layers->get(l);
      if (layer) {
        layer->remove_col(i, count);
      }
    }
  }
  layers->update();
}


void DeleteRows(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
)
{
  for (int l=0; l<layers->depth(); ++l) {
    if (viewed->at(l)) {
      Tiles *layer = layers->get(l);
      if (layer) {
        layer->remove_row(j, count);
      }
    }
  }
  layers->update();
}

} /* namespace LayerUtils */

