/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "linedlg.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "gui.h"
#include "mapdlg.h"
#include "map.h"
#include "commands.h"
#include "globals.h"

#include <altheme.h>
#include <altheme/internal/theme.h>
#include <stdio.h>



int LineDialog::LineProc(int msg, DIALOG *d, int c)
{
  LineDialog *linedlg = CAST_FROM_VOID_PTR(LineDialog*, d->dp);

  switch (msg) {
    case MSG_IDLE:
      linedlg->Idle();
      return D_O_K;
    case MSG_DRAW:
      linedlg->Draw();
      return D_O_K;
  }
  return D_O_K;
}


LineDialog::LineDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d)
{
  Init(d);
}


LineDialog::LineDialog(DIALOG *d): Dialog(d)
{
  Init(d);
}


LineDialog::~LineDialog()
{
}


void LineDialog::Init(DIALOG *d)
{
  box_ = d;
  box_->proc = LineProc;
  box_->dp = this;

  mouse_tile_x_ = 0;
  mouse_tile_y_ = 0;
  scale_ = 1;
  mouse_tile_x_ = 0;
  mouse_tile_y_ = 0;
  scale_ = 1;
  map_width_ = 0;
  map_height_ = 0;

  help_ = 0;
}


void LineDialog::InitMap()
{
  if (GetMap()) {
    const MapDialog *mapdlg = GUI.GetMapDialog();

    scale_ = mapdlg->GetScale();
    int mx = mouse_x;
    int my = mouse_y;
    mouse_tile_x_ = mapdlg->GetTileX(mx, my);
    mouse_tile_y_ = mapdlg->GetTileY(mx, my);
    map_width_ = GetMap()->GetWidth();
    map_height_ = GetMap()->GetHeight();
    map_is_modified_ = false;
  }
}


const char *LineDialog::GetHelp() const
{
  return help_;
}


void LineDialog::Idle()
{
  int mx = mouse_x;
  int my = mouse_y;

  const char *new_help = GUI.Help(mx, my);
  if (new_help != help_) {
    help_ = new_help;
    box_->flags |= D_DIRTY;
  }

  if (GetMap()) {
    const MapDialog *mapdlg = GUI.GetMapDialog();

    float new_scale = mapdlg->GetScale();
    if (new_scale!=scale_) {
      scale_ = new_scale;
      box_->flags |= D_DIRTY;
    }

    int new_mouse_tile_x = mapdlg->GetTileX(mx, my);
    int new_mouse_tile_y = mapdlg->GetTileY(mx, my);
    if (new_mouse_tile_x!=mouse_tile_x_ || new_mouse_tile_y!=mouse_tile_y_) {
      mouse_tile_x_ = new_mouse_tile_x;
      mouse_tile_y_ = new_mouse_tile_y;
      box_->flags |= D_DIRTY;
    }

    int new_map_width = GetMap()->GetWidth();
    int new_map_height = GetMap()->GetHeight();
    if (new_map_width!=map_width_ || new_map_height!=map_height_) {
      map_width_ = new_map_width;
      map_height_ = new_map_height;
      box_->flags |= D_DIRTY;
    }

    bool new_map_is_modified = GetMap()->GetCommandsManager()->HasChanged();
    if (new_map_is_modified!=map_is_modified_) {
      map_is_modified_ = new_map_is_modified;
      box_->flags |= D_DIRTY;
    }
  }
}


static const char *create_scale_string(float scale)
{
  static char string[256];
  if (scale>=1) sprintf(string, "%d:1", (int)scale);
  else sprintf(string, "1:%d", (int)(1/scale));
  return string;
}


void LineDialog::Draw()
{
  static const Translation i18nNoMap("No map");
  static const Translation i18nMapInfoUnmodified(
    "Map %dx%d %s - Mouse %dx%d"
  );
  static const Translation i18nMapInfoModified(
    "Map %dx%d %s - Mouse %dx%d - Modified"
  );

  altheme_draw_box(screen, box_->x, box_->y, box_->x+box_->w-1, box_->y+box_->h-1);
  int rtm = text_mode(-1);
  int y = box_->y + (box_->h-text_height(font))/2;
  const char *help = GetHelp();
  if (help) {
    int x = box_->x + box_->w - altheme_frame_size;
    textout_right(screen, font, help, x, y, gui_fg_color);
  }
  int x = box_->x + altheme_frame_size*2;
  if (GetMap()) {
    if (map_is_modified_) {
      textprintf(
        screen, font, x, y, gui_fg_color, i18nMapInfoModified.string(),
        map_width_, map_height_, UString(create_scale_string(scale_)).string(),
        mouse_tile_x_, mouse_tile_y_
      );
    }
    else {
      textprintf(
        screen, font, x, y, gui_fg_color, i18nMapInfoUnmodified.string(),
        map_width_, map_height_, UString(create_scale_string(scale_)).string(),
        mouse_tile_x_, mouse_tile_y_
      );
    }
  }
  else {
    textprintf(screen, font, x, y, gui_fg_color, i18nNoMap.string());
  }
  text_mode(rtm);
}

