/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__GUI__
#define EME__GUI__
/*----------------------------------------------------------------------------
  The GUI

- GFX
- Map
- Path for loading/saving the map
- Commands
- Brushes
- User menu
- Layers
- Tile dialog entry
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h" /* For STRING_PATH_BASE_LENGTH */

struct DIALOG;
struct BITMAP;

class Dialog;
class Map;

class MapDialog;
class LayerDialog;
class PropertyDialog;
class TileDialog;
class ToolsDialog;
class LineDialog;
class ClipBoard;
class MenuDialog;

class Entry;

class Brush;

class Tiles;
class SelectedTiles;
class ViewedLayers;

struct DIALOG;

typedef enum {
  GUI_DIALOG_SHORTCUTS = 0,
  NUM_GUI_DIALOGS
} GUI_DIALOG;

#include "emeftr.h"


class GraphicalUserInterface {

public:
  GraphicalUserInterface(void);
  ~GraphicalUserInterface(void);

  int DoDialog(void);


  /* GFX
     ---------------------------------*/
  bool IsInteractive() const {
    return is_interactive_;
  }

  void SetTheme(
    const char *theme_name, /* Name defined in settheme.h */
    int fg, int bg          /* Fore and back ground colors, current cdepth */
  );

  /* App name, used in the title bar */
  void SetAppName(const char *name);

  void SetSize(
    int w, int h /* GUI size in pixels */
  );

  /* Sets the callback for creating commands. If 'create_layer' is 0, disallow
   * user to add/remove/move layers. 'create_layer' should return the new layer
   * to insert. If it returns 0, no layer is inserted */
  void SetCreateLayer(
    void (*create_layer)(Map *map, int active_layer_index)
  );

  /* GUI size in pixels */
  int GetWidth(void) const
    {return width;}
  int GetHeight(void) const
    {return height;}

  /* Grid */
  void ShowGrid(void);
  void HideGrid(void);
  int IsGridDrawn(void) const;

  /* Clear the area under the menu */
  void ClearMenu(void) const;

  /* Window title */
  void SetWindowTitle(void) const;

  /* Mouse */
  void SetMouseBusy(void);
  void UnsetMouseBusy(void);


  /* Help */
  const char *Help(int mx, int my) const;


  /* AllowFeature
     ---------------------------------*/
  void AllowFeature(bool yesno, Features::Group::Groups group, unsigned int features);
  void AllowFeature(bool yesno, Features::Group::Groups group);


  /* Map
     ---------------------------------*/
  void SetMap(Map *map);
  Map *GetMap(void) const
    {return map;}
  MapDialog *GetMapDialog(void) const
    {return mapdlg;}
  void ClearMap(void);
  void RedrawMap(void);


  /* Path for loading/saving the map
     ---------------------------------*/
  void SetPath(const char *path);
  const char *GetPath(void) const;


  /* Commands
     ---------------------------------*/
  void Apply(void);
  void LayerDel(void);
  void LayerIns(void);
  void LayerUp(void);
  void LayerDown(void);
  void Copy(void);
  void CopyBrush(void);
  void Cut(void);
  void Paste(void);
  void PasteBrush(void);
  void Click(int tilex, int tiley, int shifts);
  void Press(int tilex, int tiley, int shifts);
  void Release(int tilex, int tiley, int shifts);
  void SelectAll(void);
  void SelectNone(void);
  void SelectInvert(void);
  void SelectGrow(void);
  void SelectShrink(void);
  void Undo(void);
  void Redo(void);

  void UpdateSelection(SelectedTiles *sel, int shifts);


  /* Brushes and clipboard
     ---------------------------------*/
  Brush *GetBrush(void) const;
  void SetBrush(Brush *brush);
  void ClearBrush();
  void ClearClipboard();


  /* User menu
     ---------------------------------*/
  void SetUserMenuTitle(const char *title);
  void SetUserMenu(
    int menu_index,
    const char *menu_name,
    void (*menu_callback)(Map *map),
    int hotkey=0
  );
  void CallUserMenu(int menu_index);
  bool MenuHotKey(int key);
  void SetLoadBrush(Brush *(*load_brush)(const char *));
  void SetSaveBrush(bool (*save_brush)(const char *, const Brush *));


  /* User data
     ---------------------------------*/
  void SetDrawUserdata(void (*draw)(void *data, BITMAP *bmp, int i, int j, float scale));
  void DrawUserdata(void *data, BITMAP *bmp, int i, int j, float scale);


  /* Layers and properties
     ---------------------------------*/
  void IncrActiveLayer(void);
  void DecrActiveLayer(void);

  void SetActiveProperty(int prop_index);

  void UpdateActiveLayer() const;
  void UpdateActiveProperty() const;
  void UpdateActiveLayerAndProperty() const;


  /* Tile dialog entry
     ---------------------------------*/
  Entry *GetTileEntry(void) const;

protected:


private:
  GraphicalUserInterface(const GraphicalUserInterface&);
  GraphicalUserInterface &operator=(const GraphicalUserInterface*);

  static int Shortcuts(int msg, DIALOG *d, int c);

  bool is_interactive_;
  void (*create_layer_)(Map*, int);
  void (*draw_userdata_)(void *data, BITMAP *bmp, int i, int j, float scale);
  int width, height;
  DIALOG *dialogs;
  Map *map;
  MapDialog *mapdlg;
  LayerDialog *layerdlg;
  PropertyDialog *propdlg;
  TileDialog *tiledlg;
  ToolsDialog *toolsdlg;
  LineDialog *linedlg;
  MenuDialog *menu;
  BITMAP *mouse_pointer;
  BITMAP *mouse_pointer_busy;

  char path[STRING_PATH_BASE_LENGTH];
  char *app_name;
};

#endif /* EME__GUI__ */

