/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "gui.h"

#include "debug.h"
#include "utils.h"

#include "settheme.h"
#include "mouse.h"

#include "map.h"
#include "mapdlg.h"
#include "layerdlg.h"
#include "propdlg.h"
#include "tiledlg.h"
#include "toolsdlg.h"
#include "linedlg.h"
#include "menudlg.h"

#include "globals.h"
#include "commands.h"

#include "selected.h"
#include "cmdselec.h"
#include "cmdapply.h"
#include "cmdlayer.h"
#include "lutils.h"

#include <altheme.h>
#include <altheme/internal/theme.h> /* For SB_SIZE */

#include <allegro.h>


#define MENU_HEIGHT 16


/* c must be a character, eg 'f' */
#define CTRL(c) ((c)-'a'+1)


int GraphicalUserInterface::Shortcuts(int msg, DIALOG * /*d*/, int c)
{
  if (msg==MSG_IDLE) return D_O_K;

  if (msg==MSG_XCHAR) {
    /* Eat the ESC */
    if ((c&0xff)==27) return D_USED_CHAR;

    switch (c&0xff) {
      /* Copy/Cut/Paste */
      case CTRL('c'):
        GUI.Copy();
      return D_USED_CHAR;
      case CTRL('v'):
        GUI.Paste();
      return D_USED_CHAR;
      case CTRL('p'):
        GUI.PasteBrush();
      return D_USED_CHAR;
      case CTRL('x'):
        GUI.Cut();
      return D_USED_CHAR;
      case CTRL('b'):
        GUI.CopyBrush();
      return D_USED_CHAR;

      /* Undo/Redo */
      case CTRL('z'):
        GUI.Undo();
      return D_USED_CHAR;
      case CTRL('r'):
        GUI.Redo();
      return D_USED_CHAR;

      /* Selection */
      case CTRL('a'):
        GUI.SelectAll();
      return D_USED_CHAR;
      case CTRL('n'):
        GUI.SelectNone();
      return D_USED_CHAR;
      case CTRL('i'):
        if((c>>8) == KEY_TAB) {
          return D_O_K;
        }
        GUI.SelectInvert();
      return D_USED_CHAR;

      /* Select new tool */
      case 'P':
      case 'p':
        GUI.toolsdlg->ChangeTool(TOOL_SELECT_POINT);
      return D_USED_CHAR;
      case 'R':
      case 'r':
        GUI.toolsdlg->ChangeTool(TOOL_SELECT_RECT);
      return D_USED_CHAR;
      case 'C':
      case 'c':
        GUI.toolsdlg->ChangeTool(TOOL_SELECT_CIRCLE);
      return D_USED_CHAR;
      case 'B':
      case 'b':
        GUI.toolsdlg->ChangeTool(TOOL_SELECT_BY_PROP);
      return D_USED_CHAR;
      case 'W':
      case 'w':
        GUI.toolsdlg->ChangeTool(TOOL_SELECT_BY_WAND);
      return D_USED_CHAR;
      case 'Z':
      case 'z':
        GUI.toolsdlg->ChangeTool(TOOL_ZOOM);
      return D_USED_CHAR;
      case 'D':
      case 'd':
        GUI.toolsdlg->ChangeTool(TOOL_DRAW_POINT);
      return D_USED_CHAR;
      case 'F':
      case 'f':
        GUI.toolsdlg->ChangeTool(TOOL_FILL);
      return D_USED_CHAR;

      /* Change active layer */
      case '>':
      case '.':
        GUI.IncrActiveLayer();
      return D_USED_CHAR;
      case '<':
      case ',':
        GUI.DecrActiveLayer();
      return D_USED_CHAR;

      /* Zoom in/out */
      case '+':
        GUI.GetMapDialog()->ZoomIn();
      return D_USED_CHAR;
      case '-':
        GUI.GetMapDialog()->ZoomOut();
      return D_USED_CHAR;
      case '=':
        GUI.GetMapDialog()->Unzoom();
      return D_USED_CHAR;
    }

    switch (c>>8) {
      case KEY_F1:
        popup_help();
      break;

      /* Zoom in/out */
      case KEY_PLUS_PAD:
        GUI.GetMapDialog()->ZoomIn();
      return D_USED_CHAR;
      case KEY_MINUS_PAD:
        GUI.GetMapDialog()->ZoomOut();
      return D_USED_CHAR;
      case KEY_EQUALS:
        GUI.GetMapDialog()->Unzoom();
      return D_USED_CHAR;
    }
  }

  if (GUI.MenuHotKey(c&0xff)) {
    return D_USED_CHAR | D_REDRAW;
  }

  return D_O_K;
}


GraphicalUserInterface::GraphicalUserInterface(void):
  is_interactive_(false), create_layer_(0), draw_userdata_(0),
  width(0), height(0), map(0),
  mouse_pointer(0), mouse_pointer_busy(0), app_name(0)
{
  int num_dialogs =
    NUM_GUI_DIALOGS +
    NUM_MAP_DIALOGS + NUM_TILE_DIALOGS + NUM_PROPERTY_DIALOGS +
    NUM_LAYER_DIALOGS + NUM_TOOLS_DIALOGS +
    NUM_LINE_DIALOGS +
    NUM_MENU_DIALOGS + 1;

  Calloc(dialogs, num_dialogs);

  dialogs[GUI_DIALOG_SHORTCUTS].proc = GraphicalUserInterface::Shortcuts;

  int curr_dialog = NUM_GUI_DIALOGS;

  mapdlg = new MapDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(mapdlg);
  curr_dialog += NUM_MAP_DIALOGS;

  tiledlg = new TileDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(tiledlg);
  curr_dialog += NUM_TILE_DIALOGS;

  propdlg = new PropertyDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(propdlg);
  curr_dialog += NUM_PROPERTY_DIALOGS;

  layerdlg = new LayerDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(layerdlg);
  curr_dialog += NUM_LAYER_DIALOGS;

  toolsdlg = new ToolsDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(toolsdlg);
  curr_dialog += NUM_TOOLS_DIALOGS;

  linedlg = new LineDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(linedlg);
  curr_dialog += NUM_LINE_DIALOGS;

  menu = new MenuDialog(&(dialogs[curr_dialog]));
  CHECK_POINTER(menu);

  /* The global EmptyString may not be already initialized */
  SetPath(EMPTY_STRING);
}


GraphicalUserInterface::~GraphicalUserInterface(void)
{
  free(app_name);
  delete mapdlg;
  delete propdlg;
  delete tiledlg;
  delete layerdlg;
  delete toolsdlg;
  delete linedlg;
  delete menu;
  PROFILE_BEGIN(delete_map)
  delete map;
  PROFILE_END(delete_map)
  Free(dialogs);
  LOGEXEC();
}


int GraphicalUserInterface::DoDialog(void)
{
  clear_to_color(screen, gui_mg_color);
  if(!mouse_pointer) {
    mouse_pointer = mouse_sprite;
    mouse_pointer_busy = get_mouse_busy_sprite();
  }

  menu->LazyInit();

#ifndef PARALYTIC_MOUSE
  return do_dialog(dialogs, -1);
#else
  return paralytic_do_dialog(dialogs, -1);
#endif
}


/* GFX
   ---------------------------------*/
void GraphicalUserInterface::SetTheme(const char *theme_name, int fg, int bg)
{
  gui_fg_color = fg;
  gui_bg_color = bg;
  gui_mg_color = makecol(
    (getr(fg)+getr(bg))/2, (getg(fg)+getg(bg))/2, (getb(fg)+getb(bg))/2
  );
  set_dialog_color(dialogs, fg, bg);
  dotted_init(fg, bg);
  if(theme_name) {
    ::SetTheme(theme_name);
  }
}


void GraphicalUserInterface::SetAppName(const char *name)
{
  if (app_name) free(app_name);
  app_name = ustrdup(name);
}


void GraphicalUserInterface::SetSize(int w, int h)
{
  is_interactive_ = true;
  width = w;
  height = h;

  int base_width = width/6;
  int total_height = (height-2*MENU_HEIGHT-2*DIALOG_PADDING);
  int base_height = (total_height)/5;

  mapdlg->SetSize(
    DIALOG_PADDING,
    MENU_HEIGHT+DIALOG_PADDING,
    base_width*4 - 2*DIALOG_PADDING - SB_SIZE,
    total_height - DIALOG_PADDING - SB_SIZE
  );

  tiledlg->SetSize(
    base_width*4+DIALOG_PADDING,
    MENU_HEIGHT+DIALOG_PADDING,
    base_width - DIALOG_PADDING - DIALOG_PADDING/2,
    base_height*4 - DIALOG_PADDING
  );

  layerdlg->SetSize(
    base_width*5 + DIALOG_PADDING/2,
    MENU_HEIGHT+DIALOG_PADDING,
    base_width - DIALOG_PADDING - DIALOG_PADDING/2,
    base_height*2 - DIALOG_PADDING
  );

  propdlg->SetSize(
    base_width*5+DIALOG_PADDING/2,
    base_height*2+MENU_HEIGHT+DIALOG_PADDING,
    base_width - DIALOG_PADDING - DIALOG_PADDING/2,
    base_height*2 - DIALOG_PADDING
  );

  toolsdlg->SetSize(
    base_width*4+DIALOG_PADDING,
    base_height*4+MENU_HEIGHT+DIALOG_PADDING,
    base_width*2 - 2*DIALOG_PADDING,
    base_height - DIALOG_PADDING
  );

  linedlg->SetSize(
    DIALOG_PADDING,
    height - MENU_HEIGHT,
    base_width*6 - 2*DIALOG_PADDING,
    MENU_HEIGHT
  );

  /* We don't care of the menu size, since it's automatically computed. */
}


void GraphicalUserInterface::SetCreateLayer(void (create_layer)(Map *, int))
{
  create_layer_ = create_layer;
}


void GraphicalUserInterface::ShowGrid(void)
{
  mapdlg->DrawGrid(1);
  menu->DrawGrid(1);
}


void GraphicalUserInterface::HideGrid(void)
{
  mapdlg->DrawGrid(0);
  menu->DrawGrid(0);
}


int GraphicalUserInterface::IsGridDrawn(void) const
{
  return mapdlg->IsGridDrawn();
}


void GraphicalUserInterface::ClearMenu(void) const
{
  rectfill(screen, 0, 0, SCREEN_W, MENU_HEIGHT+DIALOG_PADDING, gui_mg_color);
}


void GraphicalUserInterface::SetWindowTitle(void) const
{
  char text[1024];
  const char *fname = get_filename(GetPath());

  if (!ugetc(fname)) {
    fname = "Untitled";
  }

  if (app_name) {
    uszprintf(text, sizeof(text), "%s - %s", app_name, fname);
  }
  else {
    uszprintf(text, sizeof(text), "%s", fname);
  }
  set_window_title(text);
}


void GraphicalUserInterface::SetMouseBusy(void)
{
  if(mouse_pointer_busy) set_mouse_sprite(mouse_pointer_busy);
}


void GraphicalUserInterface::UnsetMouseBusy(void)
{
  if(mouse_pointer) set_mouse_sprite(mouse_pointer);
}


const char *GraphicalUserInterface::Help(int mx, int my) const
{
  const char *help = 0;

  if (!help) help = mapdlg->Help(mx, my);
  if (!help) help = tiledlg->Help(mx, my);
  if (!help) help = propdlg->Help(mx, my);
  if (!help) help = layerdlg->Help(mx, my);
  if (!help) help = toolsdlg->Help(mx, my);
  if (!help) help = linedlg->Help(mx, my);

  return help;
}


/* AllowFeature
   ---------------------------------*/
void GraphicalUserInterface::AllowFeature(bool yesno, Features::Group::Groups group, unsigned int features)
{
  using namespace Features;
  switch (group) {
    case Group::MapMenu:
      menu->AllowMap(yesno, features);
      break;
    case Group::EditMenu:
      menu->AllowEdit(yesno, features);
      break;
    case Group::SelectMenu:
      menu->AllowSelect(yesno, features);
      break;
    case Group::BrushMenu:
      menu->AllowBrush(yesno, features);
      break;
    case Group::UserMenu:
      menu->AllowUser(yesno, features);
      break;
    case Group::TileBox:
      tiledlg->Allow(yesno, features);
      break;
    case Group::LayersBox:
      layerdlg->Allow(yesno, features);
      break;
    case Group::ToolsBox:
      toolsdlg->Allow(yesno, features);
      break;
    default:
      DBG_ASSERT(0);
  }
}


void GraphicalUserInterface::AllowFeature(bool yesno, Features::Group::Groups group)
{
  using namespace Features;
  switch (group) {
    case Group::MapMenu:
      AllowFeature(yesno, Group::MapMenu, MapMenu::Everything);
      break;
    case Group::EditMenu:
      AllowFeature(yesno, Group::EditMenu, EditMenu::Everything);
      break;
    case Group::SelectMenu:
      AllowFeature(yesno, Group::SelectMenu, SelectMenu::Everything);
      break;
    case Group::BrushMenu:
      AllowFeature(yesno, Group::BrushMenu, BrushMenu::Everything);
      break;
    case Group::UserMenu:
      AllowFeature(yesno, Group::UserMenu, UserMenu::Everything);
      break;
    case Group::TileBox:
      AllowFeature(yesno, Group::TileBox, TileBox::Everything);
      break;
    case Group::LayersBox:
      AllowFeature(yesno, Group::LayersBox, LayersBox::Everything);
      break;
    case Group::ToolsBox:
      AllowFeature(yesno, Group::ToolsBox, ToolsBox::Everything);
      break;
    case Group::VariableLayerSize:
      AllowFeature(yesno, Group::MapMenu, MapMenu::VariableLayerSize);
      AllowFeature(yesno, Group::ToolsBox, ToolsBox::VariableLayerSize);
      break;
    case Group::VariableLayerCount:
      AllowFeature(yesno, Group::MapMenu, MapMenu::VariableLayerCount);
      AllowFeature(yesno, Group::LayersBox, LayersBox::VariableLayerCount);
      break;
    case Group::VariableMapSize:
      AllowFeature(yesno, Group::MapMenu, MapMenu::VariableMapSize);
      break;
    case Group::Select:
      AllowFeature(yesno, Group::SelectMenu, SelectMenu::Select);
      AllowFeature(yesno, Group::ToolsBox, ToolsBox::Select);
      break;
    case Group::Brushes:
      AllowFeature(yesno, Group::BrushMenu, BrushMenu::Brushes);
      AllowFeature(yesno, Group::EditMenu, EditMenu::Brushes);
      AllowFeature(yesno, Group::ToolsBox, ToolsBox::Brushes);
      break;
    case Group::Everything:
      AllowFeature(yesno, Group::MapMenu, MapMenu::Everything);
      AllowFeature(yesno, Group::EditMenu, EditMenu::Everything);
      AllowFeature(yesno, Group::SelectMenu, SelectMenu::Everything);
      AllowFeature(yesno, Group::BrushMenu, BrushMenu::Everything);
      AllowFeature(yesno, Group::UserMenu, UserMenu::Everything);
      AllowFeature(yesno, Group::TileBox, TileBox::Everything);
      AllowFeature(yesno, Group::LayersBox, LayersBox::Everything);
      AllowFeature(yesno, Group::ToolsBox, ToolsBox::Everything);
      break;
  }
}


/* Map
   ---------------------------------*/
void GraphicalUserInterface::SetMap(Map *new_map)
{
  if(map) delete map;
  map = new_map;
  mapdlg->SetMap(map);
  propdlg->SetMap(map);
  /* Layer dialog must be initialized *before* tile dialog, because tile dialog
     calls layer dialog functions */
  layerdlg->SetMap(map);
  tiledlg->SetMap(map);
  toolsdlg->SetMap(map);
  linedlg->SetMap(map);
  menu->SetMap(map);
  if (map) {
    map->SetActiveLayer(0, tiledlg->GetEntry());
    layerdlg->SetActiveLayer(map->GetActiveLayerIndex());
  }
}


void GraphicalUserInterface::ClearMap(void)
{
  SetMap(0);
}


void GraphicalUserInterface::RedrawMap(void)
{
  mapdlg->Redraw();
}


void GraphicalUserInterface::SetPath(const char *new_path)
{
  ustrzcpy(path, sizeof(path), new_path);
  usetat(path, ustrlen(path), 0);
}


const char *GraphicalUserInterface::GetPath(void) const
{
  return path;
}


/* Commands
   ---------------------------------*/
void GraphicalUserInterface::Apply(void)
{
  if(map) {
    SetMouseBusy();
    Command *cmd = new CommandApply(
      map, map->GetActiveLayerIndex(), map->GetSelectedTiles(),
      map->GetActiveLayerReference()
    );
    GetMap()->GetCommandsManager()->Execute(cmd);
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
}


void GraphicalUserInterface::LayerDel(void)
{
  if (map) {
    SetMouseBusy();
    Command *cmd = new CommandRemoveLayer(
      map, map->GetActiveLayerIndex()
    );
    map->GetCommandsManager()->Execute(cmd);
    layerdlg->SetViewedLayers();
    UnsetMouseBusy();
  }
}


void GraphicalUserInterface::LayerIns(void)
{
  if (map && create_layer_) {
    create_layer_(map, map->GetActiveLayerIndex());
    layerdlg->SetViewedLayers();
  }
}


void GraphicalUserInterface::LayerDown(void)
{
  /* Note: layer 0 is at the top in the list */
  if (map && map->GetActiveLayerIndex()+1<map->GetNumLayers()) {
    SetMouseBusy();
    Command *cmd = new CommandSwapLayers(
      map, map->GetActiveLayerIndex(), map->GetActiveLayerIndex()+1
    );
    map->GetCommandsManager()->Execute(cmd);
    UnsetMouseBusy();
  }
}


void GraphicalUserInterface::LayerUp(void)
{
  /* Note: layer 0 is at the top in the list */
  if (map && map->GetActiveLayerIndex()-1>=0) {
    SetMouseBusy();
    Command *cmd = new CommandSwapLayers(
      map, map->GetActiveLayerIndex(), map->GetActiveLayerIndex()-1
    );
    map->GetCommandsManager()->Execute(cmd);
    UnsetMouseBusy();
  }
}


void GraphicalUserInterface::Copy(void)
{
  if(GetMap()) {
    SetMouseBusy();
    toolsdlg->Copy();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::CopyBrush(void)
{
  if(GetMap()) {
    SetMouseBusy();
    toolsdlg->CopyBrush();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::Cut(void)
{
  if(GetMap()) {
    SetMouseBusy();
    toolsdlg->Cut();
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::Paste(void)
{
  GUI.toolsdlg->ChangeTool(TOOL_PASTE);
}


void GraphicalUserInterface::PasteBrush(void)
{
  GUI.toolsdlg->ChangeTool(TOOL_PASTE_BRUSH);
}


void GraphicalUserInterface::Click(int tilex, int tiley, int shifts)
{
  toolsdlg->Click(tilex, tiley, shifts);
}


void GraphicalUserInterface::Press(int tilex, int tiley, int shifts)
{
  toolsdlg->Press(tilex, tiley, shifts);
}


void GraphicalUserInterface::Release(int tilex, int tiley, int shifts)
{
  toolsdlg->Release(tilex, tiley, shifts);
}


void GraphicalUserInterface::SelectAll(void)
{
  if(GetMap()) {
    SetMouseBusy();
    PROFILE_BEGIN(select_all)
    SelectedTiles *sel = LayerUtils::SelectAll(map->GetActiveLayer());
    CHECK_POINTER(sel);
    PROFILE_END(select_all)
    PROFILE_BEGIN(new_command)
    Command *cmd = new CommandSelectionReplace(map, sel);
    PROFILE_END(new_command)
    PROFILE_BEGIN(execute)
    GetMap()->GetCommandsManager()->Execute(cmd);
    PROFILE_END(execute)
    PROFILE_BEGIN(delete_sel)
    delete sel;
    PROFILE_END(delete_sel)
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::SelectNone(void)
{
  if(GetMap()) {
    SetMouseBusy();
    SelectedTiles *sel = new SelectedTiles();
    Command *cmd = new CommandSelectionReplace(map, sel);
    GetMap()->GetCommandsManager()->Execute(cmd);
    delete sel;
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::SelectInvert(void)
{
  if(GetMap()) {
    SetMouseBusy();
    SelectedTiles *sel = LayerUtils::InvertSelection(
      map->GetSelectedTiles(), map->GetActiveLayer()
    );
    Command *cmd = new CommandSelectionReplace(map, sel);
    GetMap()->GetCommandsManager()->Execute(cmd);
    delete sel;
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::SelectGrow(void)
{
  if(GetMap()) {
    SetMouseBusy();
    SelectedTiles *sel =
      LayerUtils::GrowSelection(GetMap()->GetSelectedTiles());
    Command *cmd = new CommandSelectionReplace(map, sel);
    map->GetCommandsManager()->Execute(cmd);
    delete sel;
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::SelectShrink(void)
{
  if(GetMap()) {
    SelectedTiles *sel = LayerUtils::ShrinkSelection(
      map->GetSelectedTiles(), map->GetActiveLayer()
    );
    Command *cmd = new CommandSelectionReplace(map, sel);
    map->GetCommandsManager()->Execute(cmd);
    delete sel;
    mapdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::Undo(void)
{
  if(GetMap()) {
    SetMouseBusy();
    GetMap()->GetCommandsManager()->Undo();
    /* When a command is [un]executed, either the map of the layer list has
     * changed */
    mapdlg->Redraw();
    layerdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::Redo(void)
{
  if(GetMap()) {
    SetMouseBusy();
    GetMap()->GetCommandsManager()->Redo();
    /* When a command is [un]executed, either the map of the layer list has
     * changed */
    mapdlg->Redraw();
    layerdlg->Redraw();
    UnsetMouseBusy();
  }
  else {
    NO_MAP_LOADED();
  }
}


void GraphicalUserInterface::UpdateSelection(SelectedTiles *sel, int shifts)
{
  toolsdlg->UpdateSelection(sel, shifts);
}


Brush *GraphicalUserInterface::GetBrush(void) const
{
  return toolsdlg->GetBrush();
}


void GraphicalUserInterface::SetBrush(Brush *brush)
{
  toolsdlg->SetBrush(brush);
}

void GraphicalUserInterface::ClearBrush()
{
  toolsdlg->ClearBrush();
}

void GraphicalUserInterface::ClearClipboard()
{
  toolsdlg->Clear();
}


/* User menu
   ---------------------------------*/
void GraphicalUserInterface::SetUserMenuTitle(const char *title)
{
  menu->SetUserMenuTitle(title);
}


void GraphicalUserInterface::SetUserMenu(
  int menu_index,
  const char *menu_name,
  void (*menu_cb)(Map *),
  int hotkey
)
{
  menu->SetUserMenu(menu_index, menu_name, menu_cb, hotkey);
}


void GraphicalUserInterface::CallUserMenu(int menu_index)
{
  menu->CallUserMenu(menu_index);
}


bool GraphicalUserInterface::MenuHotKey(int key)
{
  return menu->HotKey(key);
}


void GraphicalUserInterface::SetLoadBrush(Brush *(*load_brush)(const char *))
{
  menu->SetLoadBrush(load_brush);
}


void GraphicalUserInterface::SetSaveBrush(bool (*save_brush)(const char *, const Brush *))
{
  menu->SetSaveBrush(save_brush);
}


/* User data
   ---------------------------------*/
void GraphicalUserInterface::SetDrawUserdata(void (*draw)(void *data, BITMAP *bmp, int i, int j, float scale))
{
  draw_userdata_ = draw;
}


void GraphicalUserInterface::DrawUserdata(void *data, BITMAP *bmp, int i, int j, float scale)
{
  if (draw_userdata_) {
    draw_userdata_(data, bmp, i, j, scale);
  }
}



/* Layers
   ---------------------------------*/
void GraphicalUserInterface::IncrActiveLayer(void)
{
  if(map) {
    map->IncrActiveLayer(tiledlg->GetEntry());
    layerdlg->SetActiveLayer(map->GetActiveLayerIndex());
    mapdlg->Redraw();
  }
}


void GraphicalUserInterface::DecrActiveLayer(void)
{
  if(map) {
    map->DecrActiveLayer(tiledlg->GetEntry());
    layerdlg->SetActiveLayer(map->GetActiveLayerIndex());
    mapdlg->Redraw();
  }
}


void GraphicalUserInterface::SetActiveProperty(int prop_index)
{
  if (map) {
    map->SetActiveProperty(prop_index, propdlg->GetEntry());
  }
}


Entry *GraphicalUserInterface::GetTileEntry(void) const // FIXME: try to remove this
{
  return tiledlg->GetEntry();
}


void GraphicalUserInterface::UpdateActiveLayer() const
{
  if (map) {
    map->UpdateActiveLayer(tiledlg->GetEntry());
  }
}

void GraphicalUserInterface::UpdateActiveProperty() const
{
  if (map) {
    map->UpdateActiveProperty(propdlg->GetEntry());
  }
}

void GraphicalUserInterface::UpdateActiveLayerAndProperty() const
{
  if (map) {
    map->UpdateActiveLayer(tiledlg->GetEntry());
    map->UpdateActiveProperty(propdlg->GetEntry());
  }
}

