/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__DATAFILES__
#define EME__DATAFILES__
/*----------------------------------------------------------------------------
  Keep track of the loaded Allegro datafiles.

  You must not free the objects yourself
  You must not call directy 'allegro_exit' but let Allegro exit by an atexit
  You must not use the Datafiles in a static object constructor or destructor
----------------------------------------------------------------------------*/

#include "debug.h"

struct DATAFILE;
struct BITMAP;

class DatafileManager {

public:
  DatafileManager(void);
  ~DatafileManager(void);

  /* Returns the index of object 'objname' in the datafile 'fname' */
  int GetIndex(
    const char *fname,
    const char *objname
  );

  /* Returns the datafile object 'objname' from the datafile 'fname' */
  DATAFILE *Get(
    const char *fname,
    const char *objname
  );

  /* Returns the datafile object at index 'idx' in the datafile 'fname' */
  DATAFILE *Get(
    const char *fname,
    int idx
  );

  /* Returns the name of the object at index 'idx' in the datafile 'fname' */
  const char *GetName(
    const char *fname,
    int idx
  );

  /* Returns the property 'prop_name' of the object at index 'idx' in 'fname' */
  const char *GetProperty(
    const char *fname,
    int idx,
    const char *prop_name
  );

  /* Returns the number of objects in the datafile 'fname' */
  int GetNumObjects(
    const char *fname
  );

  /* Returns the scaled bitmap corresponding to the object at 'idx' in 'fname'
     or NULL if the object is not an image (currently supported format are:
     BITMAP and RLE_SPRITE) */
  BITMAP *GetBitmap(
    const char *fname,
    int idx,
    float scale
  );

protected:

private:
  DatafileManager(const DatafileManager&);
  DatafileManager &operator=(const DatafileManager&);
  struct DatafileInfo;
  typedef DatafileInfo* iterator;

  /* Returns the index of the datafile which name is 'fname', -1 if it is not
     loaded */
  iterator FindDatafile(const char *fname);

  /* Returns the datafile which name is 'fname', loading it if needed */
  DATAFILE *GetDatafile(const char *fname);

  /* Load a datafile and add it to the list of loaded datafiles */
  DATAFILE *AddDatafile(const char *fname);

  /* Given to 'atexit' because the datafiles must be unloaded before Allegro
     exits */
  void UnloadAll(void);

  struct DatafileInfo {
    DATAFILE *datafile;
    char *fname;
    int num_objs;
    BITMAP **bitmaps;
  };

  inline iterator begin() { return datafiles_info; }
  inline iterator end() { return datafiles_info+num_datafiles; }

  int num_datafiles;         /* Number of loaded datafiles */
  DatafileInfo *datafiles_info;
  int atexit_func_is_set;    /* Did 'UnloadAll' was given to 'atexit' */

  friend void unload_all_datafiles(void);
};

extern DatafileManager Datafiles;

#endif /* EME__DATAFILES__ */

