/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "datafile.h"

#include "debug.h"
#include "utils.h"

#include <stdlib.h>
#include <string.h> /* To compare datafile properties */
#include <allegro.h>

#define CHARSTAR_NOCONST(c) ((char*)(c))


DatafileManager Datafiles;

/* To be given to atexit */
void unload_all_datafiles(void)
{
  LOG("unload_all_datafiles\n");
  Datafiles.UnloadAll();
}


DatafileManager::DatafileManager(void):
  num_datafiles(0), datafiles_info(0), atexit_func_is_set(0)
{
}


DatafileManager::~DatafileManager(void)
{
  DBG_ASSERT(num_datafiles==0);
  DBG_ASSERT(!datafiles_info);
}


void DatafileManager::UnloadAll(void)
{
  DBG_ASSERT(num_datafiles>=0);
  for (iterator df=begin(); df!=end(); ++df) {
    unload_datafile(df->datafile);
    free(df->fname);
    for (int n=0; n<df->num_objs; ++n) {
      if (df->bitmaps[n]) destroy_bitmap(df->bitmaps[n]);
    }
    Free(df->bitmaps);
  }
  Free(datafiles_info);
  num_datafiles = 0;
  datafiles_info = 0;
}


int DatafileManager::GetIndex(const char *fname, const char *objname)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *df = GetDatafile(fname);

  if(df) {
    for(int n=0; df[n].type!=DAT_END; n++) {
      /* Allegro does not convert the property, even if in the function
         'find_datafile_object' Allegro uses ustricmp to compare the name */
      if(strcmp(get_datafile_property(&(df[n]), DAT_NAME), objname) == 0) {
        return n;
      }
    }
  }
  return -1;
}


DATAFILE *DatafileManager::Get(const char *fname, const char *objname)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *df = GetDatafile(fname);

  if(!df) return NULL;
  return find_datafile_object(df, CHARSTAR_NOCONST(objname));
}


DATAFILE *DatafileManager::Get(const char *fname, int idx)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *df = GetDatafile(fname);

  if(!df) return NULL;
  if(idx<0 || idx>=GetNumObjects(fname)) {
    return NULL;
  }
  return &(df[idx]);
}


DatafileManager::iterator DatafileManager::FindDatafile(const char *fname)
{
  iterator df;
  for(df=begin(); df!=end(); ++df) {
    if (ustrcmp(fname, df->fname) == 0) {
      break;
    }
  }
  return df;
}


DATAFILE *DatafileManager::GetDatafile(const char *fname)
{
  DBG_ASSERT(num_datafiles>=0);
  iterator df = FindDatafile(fname);
  if (df!=end()) {
    return df->datafile;
  }
  else {
    return AddDatafile(fname);
  }
}


DATAFILE *DatafileManager::AddDatafile(const char *fname)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *datafile = load_datafile(CHARSTAR_NOCONST(fname));

  if(!atexit_func_is_set) {
    atexit(unload_all_datafiles);
    atexit_func_is_set = 1;
  }
  if(datafile) {
    Realloc(datafiles_info, num_datafiles+1);
    DatafileInfo *df = datafiles_info+num_datafiles;

    df->datafile = datafile;
    df->fname = ustrdup(fname);
    CHECK_POINTER(df->fname);
    int nobjs;
    for (nobjs=0; datafile[nobjs].type!=DAT_END; ++nobjs) {
    }
    df->num_objs = nobjs-1; /* Because the last object is GrabberInfo */
    Calloc(df->bitmaps, df->num_objs);
    CHECK_POINTER(df->bitmaps);

    num_datafiles ++;
  }
  return datafile;
}


const char *DatafileManager::GetName(const char *fname, int idx)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *data = Get(fname, idx);
  if(!data) return NULL;
  return get_datafile_property(data, DAT_NAME);
}


const char *DatafileManager::GetProperty(
  const char *fname, int idx, const char *prop_name
)
{
  DBG_ASSERT(num_datafiles>=0);
  DATAFILE *data = Get(fname, idx);
  if(!data) return NULL;
  return get_datafile_property(
    data, DAT_ID(prop_name[0], prop_name[1], prop_name[2], prop_name[3])
  );
}


BITMAP *DatafileManager::GetBitmap(const char *fname, int idx, float scale)
{
  DBG_ASSERT(num_datafiles>=0);
  iterator df = FindDatafile(fname);
  if (df==end()) {
    return NULL;
  }

  if(idx<0 || idx>=df->num_objs) {
    return NULL;
  }

  DATAFILE *data = &(df->datafile[idx]);
  int new_w;
  int new_h;

  switch(data->type) {
    case DAT_RLE_SPRITE:
    {
      RLE_SPRITE *rle = CAST_FROM_VOID_PTR(RLE_SPRITE*, data->dat);
      new_w = (int) (rle->w * scale);
      new_h = (int) (rle->h * scale);
    }
    break;

    case DAT_BITMAP:
    {
      BITMAP *bmp = CAST_FROM_VOID_PTR(BITMAP*, data->dat);
      new_w = (int) (bmp->w * scale);
      new_h = (int) (bmp->h * scale);
    }
    break;

    default:
    return NULL;
  }

  /* Do not allow creating bitmaps with a 0 size */
  if (new_w<1) new_w = 1;
  if (new_h<1) new_h = 1;

  /* The bitmap has already been asked for */
  if (df->bitmaps[idx]) {
    if (df->bitmaps[idx]->w == new_w && df->bitmaps[idx]->h == new_h) {
      /* The bitmap has already the correct size: return it */
      return df->bitmaps[idx];
    }
    else {
      destroy_bitmap(df->bitmaps[idx]);
    }
  }
  /* Create a new bitmap */
  BITMAP *bmp = create_bitmap(new_w, new_h);
  CHECK_POINTER(bmp);
  clear_to_color(bmp, bitmap_mask_color(bmp));
  df->bitmaps[idx] = bmp;

  switch(data->type) {
    case DAT_RLE_SPRITE:
    {
      RLE_SPRITE *rle = CAST_FROM_VOID_PTR(RLE_SPRITE*, data->dat);
      BITMAP *tmp = create_bitmap(rle->w, rle->h);
      clear_to_color(tmp, bitmap_mask_color(bmp));
      CHECK_POINTER(tmp);
      draw_rle_sprite(tmp, rle, 0, 0);
      masked_stretch_blit(tmp, bmp, 0, 0, tmp->w, tmp->h, 0, 0, new_w, new_h);
      destroy_bitmap(tmp);
    }
    break;

    case DAT_BITMAP:
    {
      BITMAP *img = CAST_FROM_VOID_PTR(BITMAP*, data->dat);
      masked_stretch_blit(img, bmp, 0, 0, img->w, img->h, 0, 0, new_w, new_h);
    }
    break;
  }

  return df->bitmaps[idx];
}


int DatafileManager::GetNumObjects(const char *fname)
{
  DBG_ASSERT(num_datafiles>=0);
  iterator df = FindDatafile(fname);
  if (df==end()) {
    AddDatafile(fname);
    df = FindDatafile(fname);
    if (df==end()) {
      return 0;
    }
  }
  return df->num_objs;
}

