/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__COMMANDS__
#define EME__COMMANDS__
/*----------------------------------------------------------------------------
  A command manager, executes, un-executes and keeps track of the commands.

- Constructor/Destructor
- Undo levels
- Commands management
- Private data
----------------------------------------------------------------------------*/

#include "debug.h"

class Command;


class Commands {

public:
  static void ChangeMaxUndoLevels(int new_max);
  static int GetMaxUndoLevels() {
    return global_max_undo_levels_;
  }

  /* Constructor/Destructor
     ---------------------------------*/
  Commands();
  ~Commands(void);

  /* Undo levels: currently, changing the undo levels number will lose
     all the previous commands
     ---------------------------------*/
  void ChangeMaxUndoLevels();

  /* Commands management
     ---------------------------------*/
  /* Executes 'cmd' and add it if needed */
  void Execute(Command *cmd);

  /* Undoes the last command */
  void Undo(void);

  /* Redoes the last command */
  void Redo(void);

  /* Removes all the commands */
  void Clear(void);

  /* Tell the command manager, the map was just saved */
  void Save(void) { save_point_ = last_executed_; }

  /* Returns true if the map has changed since the last save */
  bool HasChanged(void) const { return save_point_!=last_executed_; }

protected:

private:
  static int global_max_undo_levels_;

  Commands(const Commands&);
  Commands &operator=(const Commands&);

  int max_undo_levels_; /* Max number of undo levels */
  Command **cmds_;      /* List of commands that can be [un]executed */
  int last_executed_;   /* Index, in 'cmds' of the last [un]executed command */
  int save_point_;      /* Index of the last command before save was done */
};

#endif /* EME__COMMANDS__ */

