/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__COMMAND_MAP__
#define EME__COMMAND_MAP__
/*----------------------------------------------------------------------------
  Add/Remove row/col commands:

- Inserting a column
- Inserting a row
- Deleting a column
- Deleting a row
- Doing all of this with selection handling
----------------------------------------------------------------------------*/

#include "debug.h"

#include "command.h"

class Tiles;
class SparseTiles;
class ViewedLayers;
class SelectedTiles;
class Map;


/* Inserting a column
   ----------------------*/
class CommandInsertColumn: public Command {

public:
  CommandInsertColumn(Map *map, int l, int i, int j, int count);
  ~CommandInsertColumn(void);

  void UnExecute(void);
  void Execute(void);

  int IsEmpty(void) const;

  /* Returns the correct selection after the command was executed */
  static SelectedTiles *CreateAfter(const SelectedTiles *prev, int i, int j, int count);

protected:

private:
  CommandInsertColumn(const CommandInsertColumn&);
  CommandInsertColumn &operator=(const CommandInsertColumn&);

  Tiles *layer_;
  int i_, j_;              /* Insertion point */
  int count_;              /* Number of added cols */
};


/* Inserting a row
   ----------------------*/
class CommandInsertRow: public Command {

public:
  CommandInsertRow(Map *map, int l, int i, int j, int count);
  ~CommandInsertRow(void);

  void UnExecute(void);
  void Execute(void);

  int IsEmpty(void) const;

  static SelectedTiles *CreateAfter(const SelectedTiles *prev, int i, int j, int count);

protected:

private:
  CommandInsertRow(const CommandInsertRow&);
  CommandInsertRow &operator=(const CommandInsertRow&);

  Tiles *layer_;
  int i_, j_;              /* Insertion point */
  int count_;              /* Number of added rows */
};


/* Deleting a column
   ----------------------*/
class CommandDeleteColumn: public Command {

public:
  CommandDeleteColumn(Map *map, int l, int i, int j, int count);
  ~CommandDeleteColumn(void);

  void UnExecute(void);
  void Execute(void);

  int IsEmpty(void) const;

  static SelectedTiles *CreateAfter(const SelectedTiles *prev, int i, int j, int count);

protected:

private:
  CommandDeleteColumn(const CommandDeleteColumn&);
  CommandDeleteColumn &operator=(const CommandDeleteColumn&);

  void Init();

  Tiles *layer_;
  SparseTiles *saved_tiles_;
  int i_, j_;               /* Deletion point */
  int count_;               /* Number of deleted cols */
};


/* Deleting a row
   ----------------------*/
class CommandDeleteRow: public Command {

public:
  CommandDeleteRow(Map *map, int l, int i, int j, int count);
  ~CommandDeleteRow(void);

  void UnExecute(void);
  void Execute(void);

  int IsEmpty(void) const;

  static SelectedTiles *CreateAfter(const SelectedTiles *prev, int i, int j, int count);

protected:

private:
  CommandDeleteRow(const CommandDeleteRow&);
  CommandDeleteRow &operator=(const CommandDeleteRow&);

  void Init();

  Tiles *layer_;
  SparseTiles *saved_tiles_;
  int i_, j_;               /* Deletion point */
  int count_;               /* Number of deleted rows */
};


class CommandMoveLayer: public Command {
public:
  CommandMoveLayer(Map *map, int l, int di, int dj);
  ~CommandMoveLayer();

  void Execute();
  void UnExecute();

  int IsEmpty() const;

private:
  Tiles *layer_;
  int di_;
  int dj_;
};


class CommandResize: public Command {
public:
  CommandResize(Map *map, int l, int i, int j, int w, int h);
  ~CommandResize();

  void Execute();
  void UnExecute();

  int IsEmpty() const;
private:
  Command *imp_;
  Tiles *layer_before_;
  Tiles *layer_after_;
};


#include "cmdgroup.h"
#include "lviewed.h"
#include "map.h"
#include "lutils.h"
#include "selected.h"
template <class CommandType>
class CommandStack: public Command {
public:
  CommandStack(Map *map, int i, int j, int count):
    cmd_(), map_(map),
    saved_sel_(map_->GetSelectedTiles()->Clone()),
    new_sel_(CommandType::CreateAfter(saved_sel_, i, j, count))
  {
    for (int l=0; l<map_->GetNumLayers(); ++l) {
      cmd_.Add(new CommandType(map_, l, i, j, count));
    }
  }

  CommandStack(Map *map, const ViewedLayers &layers, int i, int j, int count):
    cmd_(), map_(map),
    saved_sel_(map_->GetSelectedTiles()->Clone()),
    new_sel_(CommandType::CreateAfter(saved_sel_, i, j, count))
  {
    for (int l=0; l<map_->GetNumLayers(); ++l) {
      if (layers.at(l)) {
        cmd_.Add(new CommandType(map_, l, i, j, count));
      }
    }
  }

  CommandStack(Map *map, int l, int i, int j, int count):
    cmd_(), map_(map),
    saved_sel_(map_->GetSelectedTiles()->Clone()),
    new_sel_(CommandType::CreateAfter(saved_sel_, i, j, count))
  {
    cmd_.Add(new CommandType(map_, l, i, j, count));
  }

  ~CommandStack() {
    delete saved_sel_;
    delete new_sel_;
  }

  void Execute() {
    map_->GetSelectedTiles()->Replace(new_sel_);
    cmd_.Execute();
    map_->GetLayers()->update();
  }

  void UnExecute() {
    map_->GetSelectedTiles()->Replace(saved_sel_);
    cmd_.UnExecute();
    map_->GetLayers()->update();
  }

  int IsEmpty() const {
    return cmd_.IsEmpty();
  }

protected:
private:
  CommandStack(const CommandStack&);
  CommandStack &operator=(const CommandStack&);

  CommandGroup cmd_;
  Map *map_;
  SelectedTiles *saved_sel_;
  SelectedTiles *new_sel_;
};

#endif /* EME__COMMAND_MAP__ */

