/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "ccolor.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "entry.h"

#include <altheme.h>

#include <allegro.h>



/* Misc
   ---------------------------------*/
void Color::Draw(
  int r, int g, int b, BITMAP *bmp, int x, int y, int w, int h
)
{
  Draw(makecol(r, g, b), bmp, x, y, w, h);
}


void Color::Draw(int color, BITMAP *bmp, int x, int y, int w, int h)
{
  rectfill(bmp, x, y, x+w-1, y+h-1, color);
}



/* Predefined colors
   ---------------------------------*/
const int Color::black = Color::pack(0, 0, 0);
const int Color::grey = Color::pack(128, 128, 128);
const int Color::white = Color::pack(255, 255, 255);
const int Color::majik = Color::pack(255, 0, 255);



/* Property
   ---------------------------------*/
Color::Property::Property(int r, int g, int b):
  BaseProperty(),
  red_(r), green_(g), blue_(b), color_(makecol(red_, green_, blue_))
{
  DBG_SET_PROP_TYPE("Color");
}


Color::Property::Property(int packed_color):
  red_(red(packed_color)), green_(green(packed_color)),
  blue_(blue(packed_color)), color_(makecol(red_, green_, blue_))
{
  DBG_SET_PROP_TYPE("Color");
}


Color::Property::~Property(void)
{
}


void Color::Property::Set(int r, int g, int b)
{
  red_ = r;
  green_ = g;
  blue_ = b;
  color_ = makecol(r, g, b);
}


void Color::Property::CopyFrom(const BaseProperty *p)
{
  const Property *pc = ::Cast<Color>(p);
  Set(pc->red_, pc->green_, pc->blue_);
}


int Color::Property::IsEqualTo(const BaseProperty *p) const 
{
  const Property *pc = ::Cast<Color>(p);
  return pc->color_ == color_;
}


Color::Property *Color::Property::Clone(void) const 
{
  return new Property(Get());
}



/* Creator
   ---------------------------------*/
Color::Creator::Creator(StaticString *name, int r, int g, int b):
  BaseCreator(name, new Property(r, g, b))
{
  DBG_SET_CREATOR_TYPE("Color");
}


Color::Creator::Creator(StaticString *name, int packed_color):
  BaseCreator(name, new Property(packed_color))
{
  DBG_SET_CREATOR_TYPE("Color");
}


Color::Creator *Color::Creator::Clone(void) const
{
  return new Creator(this);
}


Color::Creator::~Creator(void)
{
}


void Color::Creator::Draw(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
  float scale, int l, int i, int j
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pc = ::Cast<Color>(p);
  Color::Draw(pc->GetColor(), bmp, x, y, w, h);
}


int Color::Creator::PrintValue(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int color
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pc = ::Cast<Color>(p);
  textprintf(
    bmp, font, x, y, color, Translation("%s: Color %3d %3d %3d"),
    GetName()->string(), pc->GetRed(), pc->GetGreen(), pc->GetBlue()
  );
  return 1;
}


void Color::Creator::UpdateEntry(Entry *e, const BaseProperty *p) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pc = ::Cast<Color>(p);
  int color = colorpick_extract_color(pc->Get());
  int level = colorpick_extract_level(pc->Get());
  e->SetState(
    GetName()->string(), colorpick_proc, color, level,
    0, CAST_TO_VOID_PTR(this), NULL, 0
  );
}


void Color::Creator::UpdateProperty(BaseProperty *p, const Entry *e) const
{
  DBG_ASSERT(IsSameType(p));
  Property *pc = ::Cast<Color>(p);
  pc->Set(colorpick_pack_color(e->GetD1State(), e->GetD2State()));
}

