/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "brushlst.h"

#include "debug.h"
#include "utils.h"
#include "translat.h"

/* For drawing brushes */
#include "gui.h"
#include "globals.h"
#include "map.h"
#include "mapdlg.h"
#include "stiles.h"

#include "brush.h"

#include <altheme.h>

typedef enum {
  BRUSHLIST_BOX,
  BRUSHLIST_TITLE,
  BRUSHLIST_LIST,
  BRUSHLIST_OK,
  BRUSHLIST_CANCEL,
  NUM_BRUSHLIST_DIALOGS,
} BRUSHLIST_DIALOG;


static int list_cb(BITMAP *dest, int index, int x, int y, int w, DIALOG *d)
{
  const BrushList *bl = (BrushList*) (d->dp3);
  if (index<0) {
    return bl->GetNumBrushes();
  }
  const Brush *b = bl->GetBrush(index);
  int height = MAX(b->MaxY()-b->MinY()+1, text_height(font));
  if (dest) {
    int num_layers = b->GetNumLayers();
    int dx = x-b->MinX();
    int dy = y-b->MinY();
    if (GUI.GetMap()) {
      float scale = GUI.GetMapDialog()->GetScale()/GUI.GetMapDialog()->GetScaledTileDx();
      for (int l=0; l<num_layers; ++l) {
        const SparseTiles *layer = b->GetLayer(l);
        if (layer) {
          const BaseCreator *creator = layer->creator();
          SparseTiles::const_iterator tile;
          for (tile=layer->begin(); tile!=layer->end(); ++tile) {
            const BaseProperty *p = layer->get_tile(tile);
            const int x = layer->get_i(tile) + dx;
            const int y = layer->get_j(tile) + dy;
            const int i = layer->get_i(tile);
            const int j = layer->get_j(tile);
            creator->Draw(p, dest, x, y, 1, 1, scale, l, i, j);
          }
        }
      }
    }
    int rtm = text_mode(-1);
    textout(
      dest, font, bl->GetBrushName(index), dx+b->MaxX(), dy+b->MaxY()/2, d->fg
    );
    text_mode(rtm);
  }
  return height;
}


static DIALOG *dialogs(void)
{
  static const Translation i18nBrushes("Brushes");
  static const Translation i18nOK("OK");
  static const Translation i18nCancel("Cancel");

  static DIALOG dialogs[NUM_BRUSHLIST_DIALOGS+1] = {
  /* proc:
     x,   y,   w,   h, fg, bg, key, flags, d1, d2,              dp, dp2, dp3 */
    {altheme_shadow_box_proc,
     0,   0, 320, 200,  0,  0,   0,     0,  0,  0,               0,      0, 0},
    {altheme_ctext_proc,
   160,   8, 160,  20,  0,  0,   0,     0,  0,  0,
                                            i18nBrushes.void_ptr(),      0, 0},
    {altheme_elist_proc,
     8,  28, 304, 136,  0,  0,   0,D_EXIT,  0,  0,
                                         CAST_TO_VOID_PTR(list_cb),      0, 0},
    {altheme_button_proc,
     8, 172,  80,  20,  0,  0,'\r',D_EXIT,  0,  0,
                                                 i18nOK.void_ptr(),      0, 0},
    {altheme_button_proc,
   232, 172,  80,  20,  0,  0,   0,D_EXIT,  0,  0,
                                             i18nCancel.void_ptr(),      0, 0},
    {}
  };
  return dialogs;
}


BrushList::BrushList(void):
  num_brushes(0), brushes(0), names(0), active_brush(-1)
{
}


BrushList::~BrushList(void)
{
  Clear();
  DBG_ASSERT(num_brushes==0);
  Free(brushes);
  Free(names);
  LOGEXEC();
}


void BrushList::AddBrush(Brush *brush, const char *name)
{
  Realloc(brushes, num_brushes+1);
  brushes[num_brushes] = brush;
  Realloc(names, num_brushes+1);
  names[num_brushes] = ustrdup(name);
  num_brushes++;
  if(num_brushes==1) {
    active_brush=0;
  }
}


void BrushList::DeleteBrush(int n)
{
  if(n==num_brushes-1) {
    active_brush--;
  }
  Brush *brush = brushes[n];
  char *name = names[n];
  for(int b=n; b<num_brushes-1; b++) {
    brushes[b] = brushes[b+1];
    names[b] = names[b+1];
  }
  num_brushes--;
  delete brush;
  free(name);
}


void BrushList::Clear(void)
{
  for(int b=num_brushes-1; b>=0; b--) {
    DeleteBrush(b);
  }
  active_brush = -1;
}


int BrushList::GetNumBrushes(void) const
{
  return num_brushes;
}


Brush *BrushList::GetBrush(int b) const
{
  DBG_ASSERT(b>=0 && b<GetNumBrushes());
  return brushes[b];
}


const char *BrushList::GetBrushName(int b) const
{
  DBG_ASSERT(b>=0 && b<GetNumBrushes());
  return names[b];
}


int BrushList::Popup(void)
{
  dialogs()[BRUSHLIST_LIST].dp3 = this;
  dialogs()[BRUSHLIST_LIST].d1 = GetActive();
  set_dialog_color(dialogs(), gui_fg_color, gui_bg_color);
  centre_dialog(dialogs());
  int ret = popup_dialog(dialogs(), -1);
  if(GetNumBrushes() && (ret==BRUSHLIST_OK || ret==BRUSHLIST_LIST)) {
    SetActive(dialogs()[BRUSHLIST_LIST].d1);
  }
  return ret==BRUSHLIST_OK || ret==BRUSHLIST_LIST;
}

