/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "semestr.h"

#include "semeprop.h"
#include "semefile.h"
#include "semevrsn.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"

#include <altheme.h>

#include <stdlib.h>
#include <stdio.h>
#include <allegro.h>

#include "cstring.h"
typedef String::Creator StringCreator;
typedef String::Property StringProperty;


static const UString &usString()
{
  static const UString str("String");
  return str;
}
SEME_REGISTER(SemeString, usString(), SemeString::Create);


/* Constructor/destructor
   ---------------------------------*/
SemeProperty *SemeString::Create(void)
{
  SemeProperty *ret = new SemeString();
  CHECK_POINTER(ret);
  return ret;
}


SemeString::SemeString(void): SemeProperty(), len(DEFAULT_LEN)
{
}


SemeString::~SemeString(void)
{
}


void SemeString::SubCopyFrom(const SemeProperty *p)
{
  const SemeString *s = cast(p);
  len = s->len;
}


/* Additional data
   ---------------------------------*/
int SemeString::AskAdditionalData(void)
{
  int l=AdditionalData();
  if(l>0) {
    len = l;
    return 1;
  }
  else {
    return 0;
  }
}


int SemeString::AdditionalData(void)
{
  char str[STRING_INT_LENGTH*SIZEOF_CHARACTER+1];
  uszprintf(str, sizeof(str), UString("%d"), len);
  const char *title = CreateTitle(
    EmptyString(), Translation(" : maximum number of characters")
  );
  if(popup_edit(title, str, STRING_INT_LENGTH) == TRUE) {
    return atoi(str);
  }
  else {
    return 0;
  }
}


const char *SemeString::GetDataDescription(void)
{
  if(len>0) {
    return PrintDataDesc(Translation("Max num chars: %d"), len);
  }
  else {
    return PrintDataDesc(Translation("Max num chars: %s"), UNSPECIFIED_USTR);
  }
}


/* Loading/saving format file
   ---------------------------------*/
int SemeString::SaveFormat(FILE *f, int num_indent) const
{
  Field(f, num_indent, "max_chars", "%d", len);
  return 1;
}


int SemeString::LoadFormat(FILE *f, int num_indent)
{
  char *str = ReadField(f, num_indent, "max_chars");
  if(!str) {
    return 0;
  }
  sscanf(str, "%d", &len);
  free(str);
  return 1;
}


/* Creating/loading/saving map
   ---------------------------------*/
BaseCreator *SemeString::CreateCreator(void)
{
  if(len<=0) {
    if(!AskAdditionalData()) {
      return 0;
    }
  }
  BaseCreator *ret = new StringCreator(GetName(), len, SEME_DEFAULT_STRING);
  return ret;
}


BaseCreator *SemeString::LoadProperty(PACKFILE *pf, int version)
{
  int slen = pack_mgetl(pf);
  char *str = (char*) malloc(sizeof(char) * slen);
  if(!str) {
    return 0;
  }
  if(pack_fread(str, slen, pf)!=slen) {
    return 0;
  }
  int l=len;
  if(l<=0) {
    l = AdditionalData();
    if(l<=0) {
      return 0;
    }
  }
  StringCreator *creator=new StringCreator(GetName(), l, str);
  free(str);
  return creator;
}


BaseProperty *SemeString::LoadTile(
  PACKFILE *pf, int version, const BaseCreator *creator
) const
{
  const StringCreator *sc = Cast<String>(creator);
  int slen = pack_mgetl(pf);
  char *str = (char*) malloc(sizeof(char) * slen);
  if(!str) {
    return 0;
  }
  if(pack_fread(str, slen, pf)!=slen) {
    return 0;
  }
  // FIXME: Why not checking that the len is valid ??
  StringProperty *ps = sc->Create(str);
  free(str);
  return ps;
}


int SemeString::Save(PACKFILE *pf, int version, const BaseProperty *p) const
{
  const StringProperty *ps = Cast<String>(p);
  int slen = ustrsizez(ps->Get());
  if (pack_mputl(slen, pf) == EOF) return 0;
  if (pack_fwrite((char*)ps->Get(), slen, pf) == EOF) return 0;
  return 1;
}


/* Saving C code
   ---------------------------------*/
int SemeString::SaveCStructField(FILE *f, int num_indent) const
{
  char *str = CleanForC(GetName()->string());
  if(!str) {
    return 0;
  }
  Print(
    f, num_indent, "char *%s; /* %s */", str,
    SemePropertyManager::GetAsciiName(GetType())
  );
  free(str);
  return 1;
}


int SemeString::SaveCCode(
  FILE *f, int num_indent, const char *leading, const char *cleaning
) const
{
  char *str = CleanForC(GetName()->string());
  if(!str) {
    return 0;
  }
  Print(f, num_indent, "{");
  Print(f, num_indent+1, "int len = pack_mgetl(file);");
  Print(f, num_indent+1, "%s%s = (char*) malloc(sizeof(char)*len);",
        leading, str);
  Print(f, num_indent+1, "if(!%s%s) {", leading, str);
  Print(f, num_indent+1+1, "%s", cleaning);
  Print(f, num_indent+1+1, "return NULL;");
  Print(f, num_indent+1, "}");
  Print(f, num_indent+1, "pack_fread(%s%s, len, file);", leading, str);
  Print(f, num_indent, "}");
  free(str);
  return 1;
}


int SemeString::SaveCCleanCode(
  FILE *f, int num_indent, const char *leading
) const
{
  char *str = CleanForC(GetName()->string());
  if(!str) {
    return 0;
  }
  Print(f, num_indent, "free(%s%s);", leading, str);
  free(str);
  return 1;
}

