/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef SEME__PROPERTY__
#define SEME__PROPERTY__
/*------------------------------------------------------------------------------
  SemeProperty

- Constructor/destructor
- Property name
- Property additional data
- Loading/saving format file
- Creating/loading/saving map
- Saving C code
- Internal functions and data
- Defines for derived classes
------------------------------------------------------------------------------*/

#include "debug.h"

class BaseCreator;
class BaseProperty;

#include "sttcstr.h"
#include "ustring.h"


#include <stdio.h> /* For FILE */
#include <allegro.h> /* For PACKFILE */

#define UNSPECIFIED_STR  "<unspecified>"
#define UNSPECIFIED_USTR UString("<unspecified>").string()
#define DATA_DESC_LEN 256


class SemeProperty {

public:
  virtual int GetType(void) const = 0;

  /* Constructor/destructor
     ---------------------------------*/
  SemeProperty(void);
  virtual ~SemeProperty(void);

  void CopyFrom(const SemeProperty *p);


  /* Property name
     ---------------------------------*/
  /* 'name' must be U_CURRENT */
  void SetName(const char *name);
  StaticString *GetName(void) const;
  /* Must not keep a pointer to the returned string */
  const char *GetAsciiName(void) const;


  /* Property additional data
     ---------------------------------*/
  virtual int AskAdditionalData(void);
  virtual const char *GetDataDescription(void);


  /* Loading/saving format file
     ---------------------------------*/
  virtual int LoadFormat(FILE *f, int num_indent) = 0;
  virtual int SaveFormat(FILE *f, int num_indent) const = 0;


  /* Creating/loading/saving map
     ---------------------------------*/
  /* Create a creator, asking additional data if needed */
  virtual BaseCreator *CreateCreator(void) = 0;

  virtual BaseCreator *LoadProperty(PACKFILE *pf, int version) = 0;

  virtual BaseProperty *LoadTile(
    PACKFILE *pf, int version, const BaseCreator *creator
  ) const = 0;

  virtual int Save(PACKFILE *pf, int version, const BaseProperty *p) const = 0;


  /* Saving C code
     ---------------------------------*/
  virtual int SaveCStructField(FILE *f, int num_indent) const = 0;

  virtual int SaveCCode(
    FILE *f, int num_indent, const char *leading, const char *cleaning
  ) const = 0;

  virtual int SaveCCleanCode(
    FILE *f, int num_indent, const char *leading
  ) const = 0;


protected:
  /* Internal functions and data
     ---------------------------------*/
  /* Unicode aware strings */
  const char *CreateTitle(const char *beginning, const char *ending);

  /* Unicode aware strings */
  const char *PrintDataDesc(const char *format, ...);

  virtual void SubCopyFrom(const SemeProperty *p) = 0;


private:
  SemeProperty(const SemeProperty&);
  SemeProperty &operator=(const SemeProperty&);

  StaticString *name;
  char *title;
  char data_desc[DATA_DESC_LEN];
};


/* Defines for derived classes
   ---------------------------------*/
#include "sememngr.h"
/* Private */
#define SEME_TYPE static int id;
/* Public */
#define SEME_GET_TYPE int GetType(void) const {return id;}
/* In the .cc file */
#define SEME_REGISTER(type, name, creator) \
  int type::id=SemePropertyManager::Register(name, creator)

#endif /* SEME__PROPERTY__ */

