/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "sememap.h"

#include "semeprop.h"
#include "sememngr.h"
#include "semefile.h"
#include "semevrsn.h"

#include "globals.h"
#include "gui.h"
#include "brush.h"
#include "stiles.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "alclean.h"
#include "emepopup.h"

#include <altheme.h>

#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <allegro.h>

#include "plugin.h"
#include "wrapper.h"


/* Constructor/destructor
   ---------------------------------*/
SemeMap::SemeMap(void):
  map_shape(MAP_SHAPE_RECT), tile_shape(TILE_SHAPE_RECT),
  tile_w(32), tile_h(32), tile_odd(ODD_NONE), tile_dx(32), tile_dy(32),
  props_count(0), props(0), flayers_count(0), flayers(0),
  format_fname(0), use_const(1), compress_map(1),
  dirty(0)
{
  SetFormatFname(EmptyString());
}


SemeMap::SemeMap(const SemeMap *map):
  format_fname(0), use_const(1),
  dirty(0)
{
  Compress(map->IsCompressed());
  SetMapShape(map->GetMapShape());
  SetTileShape(map->GetTileShape(), map->GetTileW(), map->GetTileH());
  SetTileOffset(map->GetTileDX(), map->GetTileDY(), map->GetTileOdd());
  props_count = map->props_count;
  props = (SemeProperty**) malloc(sizeof(SemeProperty*) * props_count);
  CHECK_POINTER(props);
  for(int n=0; n<props_count; n++) {
    SemeProperty *p = SemePropertyManager::Create(map->props[n]->GetType());
    CHECK_POINTER(p);
    p->CopyFrom(map->props[n]);
    props[n] = p;
  }
  flayers_count = map->flayers_count;
  flayers = (SemeProperty**) malloc(sizeof(SemeProperty*) * flayers_count);
  CHECK_POINTER(flayers);
  for(int n=0; n<flayers_count; n++) {
    SemeProperty *p = SemePropertyManager::Create(map->flayers[n]->GetType());
    CHECK_POINTER(p);
    p->CopyFrom(map->flayers[n]);
    flayers[n] = p;
  }
  SetFormatFname(map->GetFormatFname());
}


SemeMap::~SemeMap(void)
{
  for(int n=0; n<props_count; n++) {
    delete props[n];
  }
  free(props);
  for(int n=0; n<flayers_count; n++) {
    delete flayers[n];
  }
  free(flayers);
  if(format_fname) free(format_fname);
}


/* Loading/saving format file
   ---------------------------------*/
int SemeMap::LoadFormat(const char *fname)
{
#define TEST(f, test) if(!(test)) {fclose(f); return LS_ERROR_READ;}
  char *line;
  int indent = 0;

  SetFormatFname(fname);

  FILE *f = fopen(GetAsciiFormatFname(), "rt");
  if(!f) {
    return LS_ERROR_FILE;
  }

  /* Read compression */
  line = ReadField(f, indent, SEME_COMPRESS_STR);
  TEST(f, line!=0);
  if(SEME_STRCMP(line, SEME_COMPRESS_YES_STR)) {
    compress_map = 1;
  }
  else if(SEME_STRCMP(line, SEME_COMPRESS_NO_STR)) {
    compress_map = 0;
  }
  else {
    fclose(f);
    free(line);
    return LS_ERROR_READ;
  }
  free(line);

  /* Read Map shape */
  line = ReadField(f, indent, SEME_MAP_STR);
  TEST(f, line!=0)
  if(SEME_STRCMP(line, SEME_MAP_SHAPE_RECT_STR)) {
    map_shape=MAP_SHAPE_RECT;
  }
  else if(SEME_STRCMP(line, SEME_MAP_SHAPE_DIAMOND_STR)) {
    map_shape=MAP_SHAPE_DIAMOND;
  }
  else {
    fclose(f);
    free(line);
    return LS_ERROR_READ;
  }
  free(line);

  /* Read tile shape */
  line = ReadField(f, indent, SEME_TILE_STR);
  TEST(f, line!=0)
  if(SEME_STRCMP(line, SEME_TILE_SHAPE_RECT_STR)) {
    tile_shape=TILE_SHAPE_RECT;
  }
  else if(SEME_STRCMP(line, SEME_TILE_SHAPE_DIAMOND_STR)) {
    tile_shape=TILE_SHAPE_DIAMOND;
  }
  else {
    fclose(f);
    free(line);
    return LS_ERROR_READ;
  }
  free(line);

  /* Read other tile data */
  indent++;
  {
    line = ReadField(f, indent, SEME_TILE_SIZE_STR);
    TEST(f, line!=0)
    sscanf(line, "%d %d", &tile_w, &tile_h);
    free(line);
    TEST(f, tile_w>0 && tile_h>0)

    line = ReadField(f, indent, SEME_TILE_ODDNESS_STR);
    TEST(f, line!=0)
    if(SEME_STRCMP(line, SEME_ODD_NONE_STR)) {
      tile_odd=ODD_NONE;
    }
    else if(SEME_STRCMP(line, SEME_ODD_RIGHT_STR)) {
      tile_odd=ODD_RIGHT;
    }
    else if(SEME_STRCMP(line, SEME_ODD_LEFT_STR)) {
      tile_odd=ODD_LEFT;
    }
    else {
      fclose(f);
      free(line);
      return LS_ERROR_READ;
    }
    free(line);

    line = ReadField(f, indent, SEME_TILE_OFFSET_STR);
    TEST(f, line!=0)
    sscanf(line, "%d %d", &tile_dx, &tile_dy);
    free(line);
    TEST(f, tile_dx>0 && tile_dy>0)
  }
  indent--;

  line = ReadField(f, indent, SEME_MAP_PROPERTIES_STR);
  TEST(f, line!=0)
  sscanf(line, "%d", &props_count);
  free(line);
  TEST(f, props_count>=0)
  props = (SemeProperty**) malloc(sizeof(SemeProperty*) * props_count);
  CHECK_POINTER(props);
  TEST(f, props!=0)
  indent++;
  for(int n=0; n<props_count; n++) {
    char *type = ReadLine(f, indent);
    TEST(f, type!=0);
    char *utype = uconvert_dup(type, U_ASCII, U_CURRENT);
    SemeProperty *p = SemePropertyManager::Create(utype);
    CHECK_POINTER(p);
    free(utype);
    free(type);

    char *name = ReadLine(f, indent+1);
    TEST(f, name!=0)
    p->SetName(UString(name));
    free(name);

    int ret = p->LoadFormat(f, indent+1);
    TEST(f, ret!=0)
    props[n]=p;
  }
  indent--;

  line = ReadField(f, indent, SEME_FULL_LAYERS_STR);
  TEST(f, line!=0)
  sscanf(line, "%d", &flayers_count);
  free(line);
  TEST(f, flayers_count>0)
  flayers = (SemeProperty**) malloc(sizeof(SemeProperty*) * flayers_count);
  CHECK_POINTER(flayers);
  TEST(f, flayers!=0)
  indent++;
  for(int n=0; n<flayers_count; n++) {
    char *type = ReadLine(f, indent);
    TEST(f, type!=0)
    char *utype = uconvert_dup(type, U_ASCII, U_CURRENT);
    SemeProperty *p = SemePropertyManager::Create(utype);
    CHECK_POINTER(p);
    free(utype);
    free(type);
    TEST(f, p!=0)

    char *name = ReadLine(f, indent+1);
    TEST(f, name!=0)
    p->SetName(UString(name));
    free(name);

    int ret = p->LoadFormat(f, indent+1);
    TEST(f, ret!=0)
    flayers[n]=p;
  }
  indent--;

  fclose(f);
  return LS_NO_ERROR;
#undef TEST
}


int SemeMap::SaveFormat(const char *fname)
{
  int indent = 0;

  SetFormatFname(fname);

  FILE *f = fopen(GetAsciiFormatFname(), "wt");
  if(!f) {
    return LS_ERROR_FILE;
  }

  Comment(f, "###");
  Comment(f, "### Map file format for Seme");
  Comment(
    f, "### Comments begin with # in the first column and end with the new line"
  );
  Comment(f, "### Indentation are relevant, and must be tabluations");
  Comment(f, "###");
  NewLine(f);

  /* Save compression */
  Comment(f, "");
  Comment(
    f, " Are the maps compressed? %s or %s",
    SEME_COMPRESS_YES_STR, SEME_COMPRESS_NO_STR
  );
  Comment(f, "");
  char *compress_str = SEME_COMPRESS_YES_STR;
  if(!compress_map) {
    compress_str = SEME_COMPRESS_NO_STR;
  }
  Field(f, indent, SEME_COMPRESS_STR, "%s", compress_str);
  NewLine(f);

  /* Save map shape */
  Comment(f, "");
  Comment(
    f, " Map shape: %s or %s",
    SEME_MAP_SHAPE_RECT_STR, SEME_MAP_SHAPE_DIAMOND_STR
  );
  Comment(f, "");
  char *map_shape_str=0;
  switch(map_shape) {
    case MAP_SHAPE_RECT:
      map_shape_str = SEME_MAP_SHAPE_RECT_STR;
    break;
    case MAP_SHAPE_DIAMOND:
      map_shape_str = SEME_MAP_SHAPE_DIAMOND_STR;
    break;
    default:
    DBG_ASSERT(0);
  }
  Field(f, indent, SEME_MAP_STR, "%s", map_shape_str);
  NewLine(f);

  /* Save tile shape */
  Comment(f, "");
  Comment(
    f, " Tile shape (%s or %s), size (width, height in pixels)",
    SEME_TILE_SHAPE_RECT_STR, SEME_TILE_SHAPE_DIAMOND_STR
  );
  Comment(
    f, " oddness (%s, %s or %s), offset (x, y in pixels)",
    SEME_ODD_NONE_STR, SEME_ODD_RIGHT_STR, SEME_ODD_LEFT_STR
  );
  Comment(f, "");
  char *tile_shape_str=0;
  switch(tile_shape) {
    case TILE_SHAPE_RECT:
      tile_shape_str = SEME_TILE_SHAPE_RECT_STR;
    break;
    case TILE_SHAPE_DIAMOND:
      tile_shape_str = SEME_TILE_SHAPE_DIAMOND_STR;
    break;
    default:
    DBG_ASSERT(0);
  }
  Field(f, indent, SEME_TILE_STR, "%s", tile_shape_str);

  /* Save other tile data */
  indent++;
  {
    Field(f, indent, SEME_TILE_SIZE_STR, "%d %d", tile_w, tile_h);
    char *odd_str=0;
    switch(tile_odd) {
      case ODD_NONE:
        odd_str = SEME_ODD_NONE_STR;
      break;
      case ODD_RIGHT:
        odd_str = SEME_ODD_RIGHT_STR;
      break;
      case ODD_LEFT:
        odd_str = SEME_ODD_LEFT_STR;
      break;
      default:
      DBG_ASSERT(0);
    }
    Field(f, indent, SEME_TILE_ODDNESS_STR, odd_str);
    Field(f, indent, SEME_TILE_OFFSET_STR, "%d %d", tile_dx, tile_dy);
  }
  indent--;

  /* Save map properties */
  NewLine(f);
  Comment(f, "");
  Comment(f, " Map properties count and map properties");
  Comment(f, "");
  Field(f, indent, SEME_MAP_PROPERTIES_STR, "%d", props_count);
  indent++;
  for(int n=0; n<props_count; n++) {
    Comment(f, " Map property type, name [and associated data]");
    Field(
      f, indent,
      SemePropertyManager::GetAsciiName(props[n]->GetType()),
      "%s", props[n]->GetAsciiName()
    );
    props[n]->SaveFormat(f, indent+1);
  }
  indent--;

  /* Save full layers */
  NewLine(f);
  Comment(f, "");
  Comment(f, " Full layers count and full layers");
  Comment(f, "");
  Field(f, indent, SEME_FULL_LAYERS_STR, "%d", flayers_count);
  indent++;
  for(int n=0; n<flayers_count; n++) {
    Comment(f, " Full layer type, name [and associated data]");
    Field(
      f, indent,
      SemePropertyManager::GetAsciiName(flayers[n]->GetType()),
      "%s", flayers[n]->GetAsciiName()
    );
    flayers[n]->SaveFormat(f, indent+1);
  }
  indent--;

  NewLine(f);
  Comment(f, " vim:ts=2");

  fclose(f);
  return LS_NO_ERROR;
}


void SemeMap::SetFormatFname(const char *fname)
{
  if(format_fname) free(format_fname);
  format_fname = ustrdup(fname);
  CHECK_POINTER(format_fname);
}


const char *SemeMap::GetFormatFname(void) const
{
  return format_fname;
}


const char *SemeMap::GetAsciiFormatFname(void) const
{
  static char *ascii_format_fname = 0;
  if (ascii_format_fname) free(ascii_format_fname);
  ascii_format_fname = uconvert_dup(format_fname, U_CURRENT, U_ASCII);
  return ascii_format_fname;
}


/* Creating/loading/saving map
   ---------------------------------*/
int SemeMap::NewMap(void)
{
#define TEST(test) if(!(test)) {return LS_ERROR_UNKNOWN;}

  if(flayers_count<1) {
    alert(
      Translation("You must define at least one layer"),
      Translation("in the map format"), 0,
      Translation("OK"), 0, 0, 0
    );
    return 0;
  }

  int width, height;
#ifndef PROFILE_SEME
  width = 0;
  height = 0;
  if (popup_size_and_pos(
    Translation("New map"), 0, 0, &width, &height, 0, 0
  ) == TRUE) {
#else
  width = height = 512;
  if (1) {
#endif

    if (width>0 && height>0) {

      GUI.SetMouseBusy();

      Wrapper wrapper(width, height, GetLayerCount(), GetPropertyCount());
      wrapper.SetMapShape(GetMapShape());
      wrapper.SetTileShape(GetTileShape(), GetTileW(), GetTileH());
      wrapper.SetTileOffset(GetTileDX(), GetTileDY(), GetTileOdd());

      for(int n=0; n<GetPropertyCount(); n++) {
        BaseCreator *creator = props[n]->CreateCreator();
        TEST(creator)
        wrapper.RawSetProperty(n, creator);
      }

      for(int n=0; n<GetLayerCount(); n++) {
        BaseCreator *creator = flayers[n]->CreateCreator();
        TEST(creator)
        wrapper.RawSetLayer(n, FULL_LAYER, creator);
        wrapper.FillLayer(n);
      }

      wrapper.SetMap();

      GUI.UnsetMouseBusy();
    }
    else {
      /* width_val<=0 || height_val<=0 */
      alert (
        Translation("Map width and height"), Translation("must be strictly positive"),
        0, Translation("OK"), 0, 0, 0
      );
    }
  }

  return LS_NO_ERROR;
#undef TEST
}


int SemeMap::LoadMap(const char *fname)
{
#define TEST_EOF(f) if(pack_feof(f)) {pack_fclose(f); return LS_ERROR_READ;}
#define TEST(f, test) if(!(test)) {pack_fclose(f); return LS_ERROR_READ;}

  PACKFILE *pf = pack_fopen(fname, "rp"); /* mode is ascii */
  if(!pf) {
    return LS_ERROR_FILE;
  }

  TEST_EOF(pf)
  int majik = pack_mgetl(pf);
  TEST(pf, majik==SEME_MAJIK)
  TEST_EOF(pf)
  int version = pack_mgetl(pf);

  TEST_EOF(pf)
  int width = pack_mgetl(pf);
  TEST(pf, width>0)
  TEST_EOF(pf)
  int height = pack_mgetl(pf);
  TEST(pf, height>0)

  Wrapper wrapper(width, height, GetLayerCount(), GetPropertyCount());
  wrapper.SetMapShape(GetMapShape());
  wrapper.SetTileShape(GetTileShape(), GetTileW(), GetTileH());
  wrapper.SetTileOffset(GetTileDX(), GetTileDY(), GetTileOdd());

  for(int n=0; n<GetPropertyCount(); n++) {
    TEST_EOF(pf)
    BaseCreator *creator = props[n]->LoadProperty(pf, version);
    TEST(pf, creator!=0)
    wrapper.RawSetProperty(n, creator);
  }

  for(int n=0; n<GetLayerCount(); n++) {
    BaseCreator *creator = flayers[n]->CreateCreator();
    TEST(pf, creator!=0)
    wrapper.RawSetLayer(n, FULL_LAYER, creator);
  }

  if (version < SEME_WITH_FORTRAN_LOOP) {
    /* This loop order is slow when loading */
    for(int i=0; i<wrapper.GetMapWidth(); i++) {
      for(int j=0; j<wrapper.GetMapHeight(); j++) {
        for(int n=0; n<flayers_count; n++) {
          TEST_EOF(pf)
          const BaseCreator *creator = wrapper.RawGetCreator(n);
          TEST(pf, creator!=0)
          BaseProperty *p = flayers[n]->LoadTile(pf, version, creator);
          TEST(pf, p!=0)
          wrapper.RawSetTileProperty(n, i, j, p);
        }
      }
    }
  }
  else {
    /* This loop order is quicker when loading */
    for(int j=0; j<wrapper.GetMapHeight(); j++) {
      for(int i=0; i<wrapper.GetMapWidth(); i++) {
        for(int n=0; n<flayers_count; n++) {
          TEST_EOF(pf)
          const BaseCreator *creator = wrapper.RawGetCreator(n);
          TEST(pf, creator!=0)
          BaseProperty *p = flayers[n]->LoadTile(pf, version, creator);
          TEST(pf, p!=0)
          wrapper.RawSetTileProperty(n, i, j, p);
        }
      }
    }
  }

  /* We're at the end, eof should be true */
  pack_fclose(pf);
  wrapper.SetMap();
  return LS_NO_ERROR;
#undef TEST
#undef TEST_EOF
}


int SemeMap::SaveMap(const char *fname, const Wrapper &wrapper)
{
  int majik = SEME_MAJIK;
  int version = SEME_CURRENT_VERSION;

  PACKFILE *pf;
  if(compress_map) {
    pf = pack_fopen(fname, "wp"); /* mode is ascii */
  }
  else {
    pf = pack_fopen(fname, "w!"); /* mode is ascii */
  }
  if(!pf) {
    return LS_ERROR_FILE;
  }

  pack_mputl(majik, pf);
  pack_mputl(version, pf);

  pack_mputl(wrapper.GetMapWidth(), pf);
  pack_mputl(wrapper.GetMapHeight(), pf);

  for(int n=0; n<GetPropertyCount(); n++) {
    const BaseProperty *p = wrapper.RawGetProperty(n);
    if(!props[n]->Save(pf, version, p)) {
      pack_fclose(pf);
      return LS_ERROR_WRITE;
    }
  }

  /* This loop order is quicker than "for i for j", has changed
   * between version 0.1 and 0.2 */
  for(int j=0; j<wrapper.GetMapHeight(); j++) {
    for(int i=0; i<wrapper.GetMapWidth(); i++) {
      for(int n=0; n<GetLayerCount(); n++) {
        const BaseProperty *p = wrapper.RawGetTileProperty(n, i, j);
        if(!flayers[n]->Save(pf, version, p)) {
          pack_fclose(pf);
          return LS_ERROR_WRITE;
        }
      }
    }
  }

  pack_fclose(pf);
  return LS_NO_ERROR;
}


/* Loading/saving brush
   ---------------------------------*/
Brush *SemeMap::LoadBrush(const char *fname)
{
#define TEST_EOF(f) if(pack_feof(f)) {pack_fclose(f); return 0;}
#define TEST(f, test) if(!(test)) {pack_fclose(f); return 0;}

  Brush *brush = new Brush(GetLayerCount());

  PACKFILE *pf = pack_fopen(fname, "rp"); /* mode is ascii */
  if(!pf) {
    return 0;
  }

  TEST_EOF(pf)
  int majik = pack_mgetl(pf);
  TEST(pf, majik==SEME_BRUSH_MAJIK)
  TEST_EOF(pf)
  int version = pack_mgetl(pf);

  for(int l=0; l<GetLayerCount(); l++) {
    TEST_EOF(pf)
    int num_tiles = pack_mgetl(pf);
    TEST(pf, num_tiles>=0);
    if(num_tiles) {
      BaseCreator *creator = flayers[l]->CreateCreator();
      if(creator) {
        SparseTiles *tiles = new SparseTiles(creator);

        for(int t=0; t<num_tiles; t++) {
          TEST_EOF(pf)
          int x = pack_mgetl(pf);
          TEST_EOF(pf)
          int y = pack_mgetl(pf);
          TEST_EOF(pf)
          BaseProperty *p = flayers[l]->LoadTile(pf, version, creator);
          if(!p) {
            popup_printf(
              Translation("Error"), Translation("Unable to load brush %s"), fname
            );
            delete tiles;
            delete brush;
            pack_fclose(pf);
            return 0;
          }
          tiles->set(x, y, p);
        }

        brush->SetLayer(l, tiles);
      }
      else {
        popup_printf(
          Translation("Error"), Translation("Unable to load brush %s"), fname
        );
        delete brush;
        pack_fclose(pf);
        return 0;
      }
    }
  }

  brush->MoveLayers(-brush->MinX(), -brush->MinY());

  /* We're at the end, eof should be true */
  pack_fclose(pf);
  return brush;
#undef TEST
#undef TEST_EOF
}


int SemeMap::SaveBrush(const char *fname, const Brush *brush)
{
  int majik = SEME_BRUSH_MAJIK;
  int version = SEME_CURRENT_VERSION;

  PACKFILE *pf;
  /* Always compress */
  pf = pack_fopen(fname, "wp"); /* mode is ascii */
  if(!pf) {
    return LS_ERROR_FILE;
  }

  pack_mputl(majik, pf);
  pack_mputl(version, pf);

  DBG_ASSERT(brush->GetNumLayers() == GetLayerCount());
  for(int l=0; l<GetLayerCount(); l++) {
    const SparseTiles *tiles = brush->GetLayer(l);

    if(tiles) {
      pack_mputl(tiles->count(), pf);
      SparseTiles::const_iterator id;
      for(id=tiles->begin(); id!=tiles->end(); ++id) {
        pack_mputl(tiles->get_i(id), pf);
        pack_mputl(tiles->get_j(id), pf);
        const BaseProperty *p = tiles->get_tile(id);
        if(!flayers[l]->Save(pf, version, p)) {
          pack_fclose(pf);
          return LS_ERROR_WRITE;
        }
      }
    }
    else {
      pack_mputl(0, pf);
    }
  }

  pack_fclose(pf);
  return LS_NO_ERROR;
}



/* Saving C code
   ---------------------------------*/
int SemeMap::SaveCHeader(
  const char *hfile, const char *base_upper, const char *base_lower
)
{
  int indent = 0;
  FILE *header_file = fopen(hfile, "wt");
  if(!header_file) {
    return 0;
  }

  /* Beginning of file */
  Print(header_file, indent, "#ifndef %s_LOADER_H", base_upper);
  Print(header_file, indent, "#define %s_LOADER_H", base_upper);
  NewLine(header_file);

  /* C++ handling */
  Print(header_file, indent, "#ifdef __cplusplus");
  Print(header_file, indent, "extern \"C\" {");
  Print(header_file, indent, "#endif");
  NewLine(header_file);
  NewLine(header_file);

  /* Const define */
  if(use_const) {
    Print(header_file, indent, "#define %s_SEME_CONST const", base_upper);
  }
  else {
    Print(header_file, indent, "#define %s_SEME_CONST", base_upper);
  }
  NewLine(header_file);
  Print(header_file, indent, "#define %s_SEME_MAJIK %d",
        base_upper, SEME_MAJIK);
  Print(header_file, indent, "#define %s_SEME_VERSION %d",
        base_upper, SEME_CURRENT_VERSION);
  Print(header_file, indent, "#define %s_SEME_WITH_FORTRAN_LOOP %d",
        base_upper, SEME_WITH_FORTRAN_LOOP);
  NewLine(header_file);
  NewLine(header_file);

  /* Tile structure */
  Print(header_file, indent, "typedef struct %s_TILE {", base_upper);
  indent++;
  {
    for(int n=0; n<flayers_count; n++) {
      flayers[n]->SaveCStructField(header_file, indent);
    }
  }
  indent--;
  Print(header_file, indent, "} %s_TILE;", base_upper);
  NewLine(header_file);

  /* Map structure */
  Print(header_file, indent, "typedef struct %s_MAP {", base_upper);
  indent++;
  {
    Print(header_file, indent, "int width, height;");
    for(int n=0; n<props_count; n++) {
      props[n]->SaveCStructField(header_file, indent);
    }
    Print(header_file, indent, "%s_TILE **tiles;", base_upper);
  }
  indent--;
  Print(header_file, indent, "} %s_MAP;", base_upper);
  NewLine(header_file);
  NewLine(header_file);
  NewLine(header_file);

  /* Functions */
  Print(header_file, indent, "/* %s_load_map:", base_lower);
  Print(header_file, indent, " *  Load the map file 'fname'");
  Print(header_file, indent, " *  Returns NULL if an error occurred");
  Print(header_file, indent, " */");
  Print(header_file, indent, "%s_MAP *%s_load_map(%s_SEME_CONST char *fname);",
        base_upper, base_lower, base_upper);
  NewLine(header_file);

  Print(header_file, indent, "/* %s_destroy_map:", base_lower);
  Print(header_file, indent, " *  Destroys a map created by '%s_load_map'",
        base_lower);
  Print(header_file, indent, " */");
  Print(header_file, indent, "void %s_destroy_map(%s_MAP *map);",
        base_lower, base_upper);
  NewLine(header_file);

  /* C++ handling */
  Print(header_file, indent, "#ifdef __cplusplus");
  Print(header_file, indent, "}");
  Print(header_file, indent, "#endif");
  NewLine(header_file);
  NewLine(header_file);

  /* End of file */
  Print(header_file, indent, "#endif /* %s_LOADER_H */", base_upper);
  fclose(header_file);

  return 1;
}


int SemeMap::SaveCCleanupFunc(
  FILE *code_file, const char *base_upper, const char *base_lower
)
{
  int indent = 0;

  Print(code_file, indent,
        "static void %s_cleanup(PACKFILE *f, %s_MAP *map)",
        base_lower, base_upper);
  Print(code_file, indent, "{");
  indent++;
  {
    Print(code_file, indent, "%s_destroy_map(map);", base_lower);
    Print(code_file, indent, "if(f) {");
    indent++;
    {
      Print(code_file, indent, "pack_fclose(f);");
    }
    indent--;
    Print(code_file, indent, "}");
  }
  indent--;
  Print(code_file, indent, "}");

  return 1;
}


int SemeMap::SaveCTestFunc(
  FILE *code_file, const char *base_upper, const char *base_lower,
  const char *cleanup
)
{
  int indent = 0;

  Print(code_file, indent,
        "static int %s_test_file(PACKFILE *file, %s_MAP *map)",
        base_lower, base_upper);
  Print(code_file, indent, "{");
  indent++;
  {
    Print(code_file, indent, "if(pack_ferror(file) || pack_feof(file)) {");
    indent++;
    {
      Print(code_file, indent, "%s", cleanup);
      Print(code_file, indent, "return 0;");
    }
    indent--;
    Print(code_file, indent, "}");
    Print(code_file, indent, "return 1;");
  }
  indent--;
  Print(code_file, indent, "}");

  return 1;
}


int SemeMap::SaveCLoadFunc(
  FILE *code_file, const char *base_upper, const char *base_lower,
  const char *cleanup
)
{
  int indent = 0;

  Print(code_file, indent,
        "%s_MAP *%s_load_map(%s_SEME_CONST char *fname)",
        base_upper, base_lower, base_upper);
  Print(code_file, indent, "{");
  indent++;
  {
    /* Declarations */
    Print(code_file, indent, "%s_MAP *map = NULL;", base_upper);
    Print(code_file, indent, "PACKFILE *file = NULL;");
    Print(code_file, indent, "int majik, version;");
    Print(code_file, indent, "int i, j;");
    NewLine(code_file);

    /* File opening */
    Print(code_file, indent, "/* Open the file */");
    Print(code_file, indent, "file = pack_fopen(fname, \"rp\");"); /* mode is ascii */
    Print(code_file, indent, "if (!file) {");
    indent++;
    {
      Print(code_file, indent, "%s", cleanup);
      Print(code_file, indent, "return NULL;");
    }
    indent--;
    Print(code_file, indent, "}");
    NewLine(code_file);

    /* Majik and version loading */
    Print(code_file, indent, "/* Load the majik number and seme version */");
    Print(code_file, indent, "majik = pack_mgetl(file);");
    Print(code_file, indent, "if(majik != %s_SEME_MAJIK) {", base_upper);
    indent++;
    {
      Print(code_file, indent, "%s_cleanup(file, map);", base_lower);
      Print(code_file, indent, "return NULL;");
    }
    indent--;
    Print(code_file, indent, "}");
    Print(code_file, indent, "version = pack_mgetl(file);");
    //Print(code_file, indent, "if(version != %s_SEME_VERSION) {", base_upper);
    //indent++;
    //{
    //  Print(code_file, indent, "%s_cleanup(file, map);", base_lower);
    //  Print(code_file, indent, "return NULL;");
    //}
    //indent--;
    //Print(code_file, indent, "}");
    Print(code_file, indent, "if(!%s_test_file(file, map)) return NULL;",
          base_lower);
    NewLine(code_file);

    /* Map structure creation */
    Print(code_file, indent, "/* Create the map structure */");
    Print(code_file, indent, "map = (%s_MAP*) malloc(sizeof(%s_MAP));",
          base_upper, base_upper);
    Print(code_file, indent, "if (!map) {");
    indent++;
    {
      Print(code_file, indent, "return NULL;");
    }
    indent--;
    Print(code_file, indent, "}");
    NewLine(code_file);

    /* Map size loading */
    Print(code_file, indent, "/* Load the map size */");
    Print(code_file, indent, "map->width = pack_mgetl(file);");
    Print(code_file, indent, "map->height = pack_mgetl(file);");
    Print(code_file, indent, "if(!%s_test_file(file, map)) return NULL;",
          base_lower);
    NewLine(code_file);

    /* Tile array creation */
    Print(code_file, indent, "/* Create the tile array */");
    Print(code_file, indent,
          "map->tiles = (%s_TILE**) malloc(sizeof(%s_TILE) * map->width);",
          base_upper, base_upper);
    Print(code_file, indent, "if (!map->tiles) {");
    indent++;
    {
      Print(code_file, indent, "%s", cleanup);
      Print(code_file, indent, "return NULL;");
    }
    indent--;
    Print(code_file, indent, "}");
    Print(code_file, indent, "for (i=0; i<map->width; i++) {");
    indent++;
    {
      Print(code_file, indent,
            "map->tiles[i] = (%s_TILE*) malloc(sizeof(%s_TILE) * map->height);",
            base_upper, base_upper);
      Print(code_file, indent, "if (!map->tiles[i]) {");
      indent++;
      {
        Print(code_file, indent, "%s", cleanup);
        Print(code_file, indent, "return NULL;");
      }
      indent--;
      Print(code_file, indent, "}");
    }
    indent--;
    Print(code_file, indent, "}");
    NewLine(code_file);

    /* Map properties loading */
    Print(code_file, indent, "/* Load the map properties */");
    for(int n=0; n<props_count; n++) {
      props[n]->SaveCCode(code_file, indent, "map->", cleanup);
    }
    Print(code_file, indent, "if(!%s_test_file(file, map)) return NULL;",
          base_lower);
    NewLine(code_file);

    /* Tile loading */
    Print(code_file, indent, "/* Load the tiles */");
    Print(code_file, indent, "if (version<%s_SEME_WITH_FORTRAN_LOOP) {",
          base_upper);
    indent++;
    {
      Print(code_file, indent, "/* Old version */");
      Print(code_file, indent, "for (i=0; i<map->width; i++) {");
      indent++;
      {
        Print(code_file, indent, "for (j=0; j<map->height; j++) {");
        indent++;
        {
          Print(code_file, indent, "/* Can't do the test after, because it will always be false the last time */");
          Print(code_file, indent, "if(!%s_test_file(file, map)) return NULL;",
                base_lower);
          for(int n=0; n<flayers_count; n++) {
            flayers[n]->SaveCCode(
              code_file, indent, "map->tiles[i][j].", cleanup
            );
          }
        }
        indent--;
        Print(code_file, indent, "}");
      }
      indent--;
      Print(code_file, indent, "}");
    }
    indent--;
    Print(code_file, indent, "}");
    Print(code_file, indent, "else {");
    indent++;
    {
      Print(code_file, indent, "/* New version */");
      Print(code_file, indent, "for (j=0; j<map->height; j++) {");
      indent++;
      {
        Print(code_file, indent, "for (i=0; i<map->width; i++) {");
        indent++;
        {
          Print(code_file, indent, "/* Can't do the test after, because it will always be false the last time */");
          Print(code_file, indent, "if(!%s_test_file(file, map)) return NULL;",
                base_lower);
          for(int n=0; n<flayers_count; n++) {
            flayers[n]->SaveCCode(
              code_file, indent, "map->tiles[i][j].", cleanup
            );
          }
        }
        indent--;
        Print(code_file, indent, "}");
      }
      indent--;
      Print(code_file, indent, "}");
    }
    indent--;
    Print(code_file, indent, "}");
    NewLine(code_file);

    /* Quitting */
    Print(code_file, indent, "/* Every thing went OK */");
    Print(code_file, indent, "pack_fclose(file);");
    NewLine(code_file);

    Print(code_file, indent, "return map;");
  }
  indent--;
  Print(code_file, indent, "}");

  return 1;
}


int SemeMap::SaveCDestroyFunc(
  FILE *code_file, const char *base_upper, const char *base_lower
)
{
  int indent = 0;

  Print(code_file, indent,
        "void %s_destroy_map(%s_MAP *map)",
        base_lower, base_upper);
  Print(code_file, indent, "{");
  indent++;
  {
    Print(code_file, indent, "int i, j;");
    NewLine(code_file);

    Print(code_file, indent, "if(map) {");
    indent++;
    {
      Print(code_file, indent, "/* Clean the map properties */");
      for(int n=0; n<props_count; n++) {
        props[n]->SaveCCleanCode(code_file, indent, "map->");
      }
      NewLine(code_file);

      Print(code_file, indent, "if(map->tiles) {");
      indent++;
      {
        Print(code_file, indent, "/* Clean the tiles */");
        Print(code_file, indent, "for (i=0; i<map->width; i++) {");
        indent++;
        {
          Print(code_file, indent, "if(map->tiles[i]) {");
          indent++;
          {
            Print(code_file, indent, "for (j=0; j<map->height; j++) {");
            indent++;
            {
              for(int n=0; n<flayers_count; n++) {
                flayers[n]->SaveCCleanCode(
                  code_file, indent, "map->tiles[i][j]."
                );
              }
            }
            indent--;
            Print(code_file, indent, "}");
            Print(code_file, indent, "free(map->tiles[i]);");
          }
          indent--;
          Print(code_file, indent, "}");
        }
        indent--;
        Print(code_file, indent, "}");
        Print(code_file, indent, "free(map->tiles);");
      }
      indent--;
      Print(code_file, indent, "}");
      Print(code_file, indent, "free(map);");
    }
    indent--;
    Print(code_file, indent, "}");
  }
  indent--;
  Print(code_file, indent, "}");

  return 1;
}


int SemeMap::SaveCFile(
  const char *cfile,
  const char *base, const char *base_upper, const char *base_lower
)
{
  int indent = 0;
  FILE *code_file = fopen(cfile, "wt");
  if(!code_file) {
    return 0;
  }

  int len = strlen(base_lower) + strlen("_cleanup(file, map);") + 1;
  char *cleanup = (char*) malloc(len * sizeof(char));
  CHECK_POINTER(cleanup);
  sprintf(cleanup, "%s_cleanup(file, map);", base_lower);

  /* Beginning of file */
  Print(code_file, indent, "#include \"%s.h\"", base);
  NewLine(code_file);
  Print(code_file, indent, "#include <stdlib.h>");
  Print(code_file, indent, "#include <allegro.h>");
  NewLine(code_file);
  NewLine(code_file);
  NewLine(code_file);

  /* Static cleanup function */
  SaveCCleanupFunc(code_file, base_upper, base_lower);
  NewLine(code_file);
  NewLine(code_file);
  NewLine(code_file);

  /* Static test function */
  SaveCTestFunc(code_file, base_upper, base_lower, cleanup);
  NewLine(code_file);
  NewLine(code_file);
  NewLine(code_file);

  /* Load function */
  SaveCLoadFunc(code_file, base_upper, base_lower, cleanup);
  NewLine(code_file);
  NewLine(code_file);
  NewLine(code_file);

  /* Destroy function */
  SaveCDestroyFunc(code_file, base_upper, base_lower);
  NewLine(code_file);

  /* End of file */
  NewLine(code_file);
  fclose(code_file);

  free(cleanup);

  return 1;
}


int SemeMap::SaveCCode(const char *fname)
{
#define CLEANUP \
  free(cfile); free(hfile); free(base); free(base_ascii); free(base_upper); free(base_lower);

  int len = ustrsize(fname) + 4*SIZEOF_CHARACTER;

  char *cfile = 0;
  char *hfile = 0;
  char *base = 0;
  char *base_ascii = 0;
  char *base_upper = 0;
  char *base_lower = 0;

  /* Create filenames */
  cfile = (char*) malloc(sizeof(char) * len);
  if(!cfile) {
    ERROR_NOMEM();
    CLEANUP
    return 0;
  }
  hfile = (char*) malloc(sizeof(char) * len);
  if(!hfile) {
    ERROR_NOMEM();
    CLEANUP
    return 0;
  }

  replace_extension(cfile, fname, UString("c"), sizeof(char) * len);
  replace_extension(hfile, fname, UString("h"), sizeof(char) * len);

  /* Test if they already exist */
  if(exists(cfile)) {
    if(!OVERWRITE_FILE(cfile)) {
      CLEANUP
      return 1;
    }
  }
  if(exists(hfile)) {
    if(!OVERWRITE_FILE(hfile)) {
      CLEANUP
      return 1;
    }
  }

  /* Create leading name parts */
  base = ustrdup(get_filename(fname));
  if(!base) {
    ERROR_NOMEM();
    CLEANUP
    return 0;
  }
  usetc(get_extension(base), 0);
  if(ugetc(base) == 0) {
    CLEANUP
    return 0;
  }
  if(ugetat(base, ustrlen(base)-1) == '.') {
    usetat(base, ustrlen(base)-1, 0);
    if(ugetc(base) == 0) {
      CLEANUP
      return 0;
    }
  }

  base_upper = CleanForC(base);
  if(!base_upper) {
    CLEANUP
    return 0;
  }
  base_lower = CleanForC(base);
  if(!base_lower) {
    CLEANUP
    return 0;
  }
  base_ascii = uconvert_dup(base, U_CURRENT, U_ASCII);

  for(char *c=base_upper; *c; c++) {
    *c = toupper(*c);
  }
  for(char *c=base_lower; *c; c++) {
    *c = tolower(*c);
  }

  /* Save header file */
  char *ascii_hfile = uconvert_dup(hfile, U_CURRENT, U_ASCII);
  if(!SaveCHeader(ascii_hfile, base_upper, base_lower)) {
    free(ascii_hfile);
    CLEANUP
    return 0;
  }
  free(ascii_hfile);

  /* Save C code file */
  char *ascii_cfile = uconvert_dup(cfile, U_CURRENT, U_ASCII);
  if(!SaveCFile(ascii_cfile, base_ascii, base_upper, base_lower)) {
    free(ascii_cfile);
    CLEANUP
    return 0;
  }
  free(ascii_cfile);

  CLEANUP

  return 1;
#undef CLEANUP
}


int SemeMap::SaveCppCode(const char *fname)
{
  // TODO
  return 0;
}


/* Editing map properties
   ---------------------------------*/
void SemeMap::AddProperty(SemeProperty *p)
{
  if(!Proceed()) {
    return;
  }
  props = (SemeProperty**)realloc(props, sizeof(SemeProperty*)*(props_count+1));
  CHECK_POINTER(props);
  props[props_count] = p;
  props_count++;
}


SemeProperty *SemeMap::RemoveProperty(int index)
{
  if(!Proceed()) {
    return 0;
  }
  if(index<0 || index>=props_count) {
    return 0;
  }
  SemeProperty *ret = props[index];
  for(int i=index; i<props_count-1; i++) {
    props[i] = props[i+1];
  }
  props_count--;
  return ret;
}


SemeProperty *SemeMap::ReplaceProperty(int index, SemeProperty *p)
{
  if(!Proceed()) {
    return 0;
  }
  // FIXME: should test index validity
  SemeProperty *ret = props[index];
  props[index] = p;
  return ret;
}


SemeProperty *SemeMap::GetProperty(int index)
{
  return props[index];
}


/* Editing layers
   ---------------------------------*/
void SemeMap::AddLayer(SemeProperty *l, int is_full)
{
  // TODO Sparse layers
  if(!Proceed()) {
    return;
  }
  flayers = (SemeProperty**)realloc(
    flayers,sizeof(SemeProperty*)*(flayers_count+1)
  );
  CHECK_POINTER(flayers);
  flayers[flayers_count] = l;
  flayers_count++;
}


SemeProperty *SemeMap::RemoveLayer(int index)
{
  if(!Proceed()) {
    return 0;
  }
  if(index<0 || index>=flayers_count) {
    return 0;
  }
  SemeProperty *ret = flayers[index];
  for(int i=index; i<flayers_count-1; i++) {
    flayers[i] = flayers[i+1];
  }
  flayers_count--;
  return ret;
}


SemeProperty *SemeMap::ReplaceLayer(int index, SemeProperty *l, int is_full)
{
  // TODO Sparse layers
  if(!Proceed()) {
    return 0;
  }
  // FIXME: should test index validity
  SemeProperty *ret = flayers[index];
  flayers[index] = l;
  return ret;
}


SemeProperty *SemeMap::GetLayer(int index)
{
  return flayers[index];
}


int SemeMap::IsLayerFull(int index)
{
  // TODO Sparse layers
  return 1;
}


int SemeMap::Proceed(void)
{
  if(GUI.GetMap()) {
    if(!dirty) {
      if(LOSE_MAP()) {
        dirty = 1;
        return 1;
      }
      else {
        return 0;
      }
    }
  }
  return 1;
}

