/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "semefile.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"

#include "alclean.h"

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>


#include <allegro.h>


/*
 *
 */
#define MY_FGETS_BASE_LEN 256
char *my_fgets(FILE *stream)
{
  DBG_ASSERT(stream);

  int len = MY_FGETS_BASE_LEN;
  char *ret = (char*) malloc(sizeof(char) * len);

  if(!ret) {
    return NULL;
  }

  fgets(ret, len, stream);
  while(!feof(stream) && !strchr(ret, '\n')) {
    len *= 2;
    ret = (char*) realloc(ret, sizeof(char) * len);
    if(!ret) {
      return NULL;
    }
    fgets(ret+len/2, len/2, stream);
  }

  return ret;
}


/*
 * Prints a new line...
 */
void NewLine(FILE *f)
{
  fprintf(f, "\n");
}


/*
 * Indentation fonction for the map format file, add 'num_indents' tabs
 */
void Indent(FILE *f, int num_indent)
{
  for(int n=0; n<num_indent; n++) {
    fprintf(f, "%c", SEME_INDENT);
  }
}


/*
 * Add a comment to the map format file, put a '#', the comments and \n
 */
void Comment(FILE *f, const char *fmt, ...)
{
  va_list ap;

  fprintf(f, "%c", SEME_COMMENT);
  va_start(ap, fmt);
  vfprintf(f, fmt, ap);
  va_end(ap);
  NewLine(f);
}


/*
 * Puts a field in the map format file
 */
void Field(
  FILE *f, int num_indent, const char *field_name, const char *field_fmt, ...
)
{
  /* Field Name */
  Indent(f, num_indent);
  fprintf(f, "%s", field_name);
  NewLine(f);

  /* Field Value */
  va_list ap;

  Indent(f, num_indent+1);
  va_start(ap, field_fmt);
  vfprintf(f, field_fmt, ap);
  va_end(ap);
  NewLine(f);
}


/*
 * Read a line from a map format file, returns a newly allocated string
 */
char *ReadLine(FILE *f, int num_indent)
{
  char *str = 0;
  char *ret;

  do {
    if(str) free(str);
    str = my_fgets(f);
    if(!str) return 0;
    char *nl = strchr(str, '\n');
    if(nl) *nl = 0;
    char *cmt = strchr(str, SEME_COMMENT); /* can't use # in a line */
    if(cmt) *cmt = 0;
  } while(str[0]==0);
  for(int n=0; n<num_indent; n++) {
    if(str[n]!=SEME_INDENT) {
      free(str);
      return 0;
    }
  }
  ret = strdup(str+num_indent);
  free(str);
  return ret;
}


/*
 * Reads a field from a map format file, returning the field value, or NULL
 * if case of failure, returns a newly allocated string
 */
char *ReadField(FILE *f, int num_indent, const char *field_name)
{
  /* Field Name */
  // FIXME: should eat the blanks at eol ?
  char *f_name = ReadLine(f, num_indent);
  if(!f_name) {
    return 0;
  }
  if(strcmp(f_name, field_name) != 0) {
    free(f_name);
    return 0;
  }
  free(f_name);

  /* Field Value */
  return ReadLine(f, num_indent+1);
}


char *CleanForSeme(const char *name)
{
  char *ret = ustrdup(name);
  if (!ret) ERROR_NOMEM();
  for (int i=0; ; i++) {
    int c = ugetat(ret, i);
    if (c==SEME_INDENT || c==SEME_COMMENT) {
      usetat(ret, i, '_');
    }
    if (!c) {
      break;
    }
  }
  return ret;
}


/*
 * Put 'num' indentations (ie num*3 spaces) in a C file
 */
static int CIndent(FILE *f, int num)
{
  for(int n=0; n<num; n++) {
    fprintf(f, "   ");
  }
  return 1;
}


/*
 * Prints a line in a C file
 */
int Print(FILE *f, int num_indent, const char *fmt, ...)
{
  va_list ap;

  CIndent(f, num_indent);
  va_start(ap, fmt);
  vfprintf(f, fmt, ap);
  va_end(ap);
  NewLine(f);
  return 1;
}


/*
 * Transform 'name' to a valid C identifier, returns a newly allocated string
 */
char *CleanForC(const char *name)
{
  char *tmp = ustrdup(name);
  if(!tmp) ERROR_NOMEM();

  /* C identifiers must begin with an ascii alphabetical character or _ */
  if(!isalpha(ugetat(tmp, 0)) || !isascii(ugetat(tmp, 0))) {
    usetat(tmp, 0, '_');
  }
  /* They must contain only ascii alphabetical character, digits or _ */
  for(int i=0, c=ugetat(tmp, i); c; i++, c=ugetat(tmp, i)) {
    if(!isalnum(c) || !isascii(c)) {
      usetat(tmp, i, '_');
    }
  }

  int size = uconvert_size(tmp, U_CURRENT, U_ASCII);
  char *ret = (char*)malloc(sizeof(char)*size);
  if (!ret) ERROR_NOMEM();
  do_uconvert(tmp, U_CURRENT, ret, U_ASCII, size);

  return ret;
}



char *relativize(const char *path)
{
  char *u_path = uconvert_dup(path, U_ASCII, U_CURRENT);
  if(!u_path) {
    return 0;
  }
  char *u_ret = urelativize(u_path);
  free(u_path);
  if(!u_ret) {
    return 0;
  }
  char *ret = uconvert_dup(u_ret, U_CURRENT, U_ASCII);
  free(u_ret);
  return ret;
}


char *derelativize(const char *path)
{
  char *u_path = uconvert_dup(path, U_ASCII, U_CURRENT);
  if(!u_path) {
    return 0;
  }
  char *u_ret = uderelativize(u_path);
  free(u_path);
  if(!u_ret) {
    return 0;
  }
  char *ret = uconvert_dup(u_ret, U_CURRENT, U_ASCII);
  free(u_ret);
  return ret;
}


char *urelativize(const char *path)
{
  char *exe_path = al_clean_get_full_executable_name();
  if(!exe_path) {
    return NULL;
  }

  char *abs_path = al_clean_fix_filename_path(path);
  if(!abs_path) {
    return NULL;
  }

  int len = ustrsizez(abs_path)+ustrsizez(exe_path);
  char *ret = (char*)malloc(len);
  if(!ret) ERROR_NOMEM();
  ustrcpy(ret, EMPTY_STRING);

  int u0 = ugetc(empty_string);
  int first=0;
  for(first=0; ; ) {
    int abs_path_c = ugetc(abs_path+first);
    int exe_path_c = ugetc(exe_path+first);
    if(abs_path_c == u0 || exe_path_c == u0) {
      break;
    }
    if(abs_path_c != exe_path_c) {
      break;
    }
    first += ucwidth(abs_path_c);
  }

  int separator = ugetc(UString("/"));
  char tmp[22];
  usetc(tmp, OTHER_PATH_SEPARATOR);
  int other_separator = ugetc(tmp);
  for(int n=0; ; ) {
    int c = ugetc(exe_path+first+n);
    if(c == u0) {
      break;
    }
    if(c == separator || c == other_separator) {
      ustrcat(ret, UString(".."));
      put_backslash(ret);
    }
    n += ucwidth(c);
  }

  ustrcat(ret, abs_path+first);

  free(exe_path);
  free(abs_path);

  return ret;
}


char *uderelativize(const char *path)
{
  char *exe_path = al_clean_get_full_executable_name();
  if(!exe_path) {
    return NULL;
  }

  int path0 = ugetc(path);
  int path1 = ugetc(path+ucwidth(path0));
  int dot = ugetc(".");
  if(path0 == dot && path1 == dot) {
    int len = ustrsizez(exe_path)+ustrsizez(path);
    char *tmp = (char*) malloc(len*sizeof(char));
    char *ret = (char*) malloc(len*sizeof(char));
    if(tmp && ret) {
      replace_filename(tmp, exe_path, path, len);
      fix_filename_path(ret, tmp, len);
      free(exe_path);
      free(tmp);
      return ret;
    }
    else {
      ERROR_NOMEM();
      if(ret) free(ret);
      if(tmp) free(tmp);
      free(exe_path);
      return ustrdup(path);
    }
  }
  else {
    char *ret;
    free(exe_path);
    ret = ustrdup(path);
    if(!ret) ERROR_NOMEM();
    return ret;
  }
}

