/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "wrapper.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"
#include "emepopup.h"

#include "globals.h"
#include "gui.h"
#include "map.h"
#include "tiles.h"
#include "ftiles.h"
#include "stiles.h"

#include "ccomposi.h"

#include <allegro.h> /* For ustrcpy */


Wrapper::Wrapper(int width, int height, int num_layers, int num_map_props):
  w_(width), h_(height), owns_map_(true)
{
  ustrcpy(hackstr, EMPTY_STRING);
  map_ = new Map(width, height, num_layers);
  const_map_ = map_;
  SetNumProperties(num_map_props);
  SetNumLayers(num_layers);
}


Wrapper::Wrapper(int width, int height):
  w_(width), h_(height), owns_map_(true)
{
  ustrcpy(hackstr, EMPTY_STRING);
  map_ = new Map(width, height);
  const_map_ = map_;
}


Wrapper::Wrapper(const Map *map):
  map_(0), const_map_(map), w_(map->GetWidth()), h_(map->GetHeight()),
  layers_fullness_(0), owns_map_(false)
{
  ustrcpy(hackstr, EMPTY_STRING);
}


Wrapper::Wrapper(Map *map):
  map_(map), const_map_(map), w_(GetMap()->GetWidth()), h_(GetMap()->GetHeight()),
  layers_fullness_(0), owns_map_(false)
{
  ustrcpy(hackstr, EMPTY_STRING);
}


Wrapper::~Wrapper(void)
{
  if (owns_map_ && GetMap()) {
    /* If we still have ownership, delete it */
    delete GetMap();
  }
}


void Wrapper::SetNumLayers(int layer_count)
{
  for (int l=0; l<layer_count; ++l) {
    if (l>=GetMap()->GetLayers()->depth()) {
      GetMap()->GetLayers()->set(l, 0);
    }
  }
}


void Wrapper::SetNumProperties(int property_count)
{
  Composite::Creator *proplist;
  proplist = new Composite::Creator(
    new StaticString(Translation("Map Properties")), property_count
  );
  GetMap()->SetProperties(proplist);
}


void Wrapper::SetLayerFullness(LAYER_TYPE type, int where)
{
  if ((unsigned int)where < layers_fullness_.size()) {
    std::vector<LAYER_TYPE>::iterator it = layers_fullness_.begin();
    advance(it, where);
    layers_fullness_.insert(it, type);
  }
  else {
    int count = where-layers_fullness_.size();
    for (int n=0; n<count; ++n) {
      layers_fullness_.push_back(LAYER_TYPE_COUNT);
    }
    layers_fullness_.push_back(type);
  }
}


/* Helpers to create error messages */
const char *Wrapper::HackStrLayer(int index) const
{
  uszprintf(hackstr, sizeof(hackstr), Translation("layer %d"), index);
  return hackstr;
}

const char *Wrapper::HackStrLayerAt(int index, int x, int y) const
{
  uszprintf(
    hackstr, sizeof(hackstr), Translation("layer %d at %d %d"), index, x, y
  );
  return hackstr;
}

const char *Wrapper::HackStrProperty(int index) const
{
  uszprintf(hackstr, sizeof(hackstr), Translation("property %d"), index);
  return hackstr;
}



void Wrapper::SetMapShape(MAP_SHAPE shape)
{
  GUI.GetMapDialog()->SetMapShape(shape);
}


void Wrapper::SetTileShape(TILE_SHAPE shape, int w, int h)
{
  GUI.GetMapDialog()->SetTileShape(shape, w, h);
}


void Wrapper::SetTileOffset(int dx, int dy, ODD odd)
{
  GUI.GetMapDialog()->SetTileOffset(dx, dy, odd);
}


int Wrapper::GetMapWidth(void) const
{
  return w_;
}


int Wrapper::GetMapHeight(void) const
{
  return h_;
}


bool Wrapper::IsIn(int i, int j) const
{
  return GetMap()->IsIn(i, j);
}


void Wrapper::SetUserdata(void *userdata, void (*destroyer)(void *userdata))
{
  GetMap()->SetUserdata(userdata, destroyer);
}


void *Wrapper::GetUserdata() const
{
  return GetMap()->GetUserdata();
}



void Wrapper::SetMap(void)
{
  USER_CHECK(
    if (!GetMap()->GetProperties()) {
      popup_quit(
        UString("SetMap"), Translation("Map properties aren't set"), 0
      );
    }
    for(int p=0; p<GetMap()->GetNumProperties(); p++) {
      TEST_EXISTS(
        GetMap()->GetProperties()->Get(p), UString("SetMap"), HackStrProperty(p)
      );
    }
  );
  USER_CHECK(
    if(GetMap()->GetNumLayers()<=0) {
      popup_quit(
        UString("SetMap"), Translation("Must have at least one layer"), 0
      );
    }
  );
  USER_CHECK(
  for(int l=0; l<GetMap()->GetNumLayers(); l++) {
    if(!GetMap()->GetLayer(l)) {
      popup_quit(
        UString("SetMap"), Translation("Layer is not set"), HackStrLayer(l)
      );
    }
    USER_ASSERT(owns_map_);
    USER_CHECK(
      if(layers_fullness_[l]==FULL_LAYER) {
        int width = GetMapWidth();
        int height = GetMapHeight();
        for(int i=0; i<width; i++) for(int j=0; j<height; j++) {
          if(!RawGetTileProperty(l, i, j)) {
            popup_quit(
              UString("SetMap"), Translation("Tile is not set"),
              HackStrLayerAt(l, i, j)
            );
          }
        }
      }
    );
  }
  );
  /* Give ownership to GUI */
  GUI.SetMap(GetMap());
  map_ = 0;
  const_map_ = 0;
}


CompositeHandle Wrapper::CreateComposite(const char *name, int num_properties)
{
  Composite::Creator *creator =
    new Composite::Creator(new StaticString(name), num_properties);
  return CompositeHandle(creator);
}


void Wrapper::SetCompositeProperty(
  int index, CompositeHandle handle
)
{
  USER_CHECK(
    for(int i=0; i<handle.cc->Count(); i++) {
      const Composite::Creator *c = handle.cc->GetReference();
      const Composite::Property *p = c->Get(i);
      TEST_EXISTS(p, UString("SetCompositeProperty"), HackStrProperty(index));
    }
  );
  RawSetProperty(index, handle.cc);
}


void Wrapper::SetCompositeLayer(
  int index, LAYER_TYPE full_or_sparse, CompositeHandle handle,
  int x, int y, int w, int h
)
{
  USER_CHECK(
    for(int i=0; i<handle.cc->Count(); i++) {
      const Composite::Creator *c = handle.cc->GetReference();
      const Composite::Property *p = c->Get(i);
      TEST_EXISTS(p, UString("SetCompositeLayer"), HackStrLayer(index));
    }
  );
  RawSetLayer(index, full_or_sparse, handle.cc, x, y, w, h);
}


Composite::Property *Wrapper::SetCompositeTile(int index, int x, int y)
{
  Tiles *layer = GetMap()->GetLayer(index);
  USER_CHECK(
    if(!layer) {
      popup_quit(
        UString("SetCompositeTile"), Translation("Layer is not set"),
        HackStrLayer(index)
      );
    }
  );
  const BaseCreator *creator = layer->creator();
  TEST_TYPE(
    Composite::Creator, *creator, UString("SetCompositeTile"),
    HackStrLayerAt(index, x, y)
  );
  Composite::Property *prop = Cast<Composite>(creator->Create());
  CHECK_POINTER(prop);
  RawSetTileProperty(index, x, y, prop);
  return prop;
}


const Composite::Property *Wrapper::GetCompositeProperty(int index) const
{
  const BaseProperty *prop = RawGetProperty(index);
  TEST_TYPE(
    Composite::Property, *prop, UString("GetCompositeProperty"),
    HackStrProperty(index)
  );
  return (const Composite::Property*)prop;
}


Composite::Property *Wrapper::GetCompositeProperty(int index)
{
  const BaseProperty *prop = RawGetProperty(index);
  TEST_TYPE(
    Composite::Property, *prop, UString("GetCompositeProperty"),
    HackStrProperty(index)
  );
  return (Composite::Property*)prop;
}


const Composite::Property *Wrapper::GetCompositeTile(
  int index, int x, int y
) const
{
  const BaseProperty *prop = RawGetTileProperty(index, x, y);
  TEST_TYPE(
    Composite::Property, *prop, UString("GetCompositeTile"),
    HackStrLayerAt(index, x, y)
  );
  return (const Composite::Property*)prop;
}


Composite::Property *Wrapper::GetCompositeTile(
  int index, int x, int y
)
{
  const BaseProperty *prop = RawGetTileProperty(index, x, y);
  TEST_TYPE(
    Composite::Property, *prop, UString("GetCompositeTile"),
    HackStrLayerAt(index, x, y)
  );
  return (Composite::Property*)prop;
}


int Wrapper::Count(int layer) const
{
  return GetMap()->GetLayer(layer)->count();
}


typedef Wrapper::iterator_ iterator_;
static iterator_ pack_(int i, int j) { return iterator_((j<<16)+i); }
static int unpacki_(iterator_ packed) { return packed.it&0xffff; }
static int unpackj_(iterator_ packed) { return (packed.it>>16) & 0xffff; }
static iterator_ end_(const Tiles *t) { return pack_(t->endi(), t->endj()); }
static iterator_ begin_(const Tiles *t) {
  //DBG_ASSERT(t);
  int i = 0;
  int j = 0;
  while (!t->get(i, j)) {
    j++;
    if (j==t->endj()) {
      j = t->beginj();
      i++;
      if (i==t->endi()) return end_(t);
    }
  }
  return pack_(i, j);
}
static iterator_ next_(const Tiles *t, iterator_ packed) {
  //DBG_ASSERT(t);
  //DBG_ASSERT(packed != end_(t));
  int i = unpacki_(packed);
  int j = unpackj_(packed);
  do {
    j++;
    if (j==t->endj()) {
      j = t->beginj();
      i++;
      if (i==t->endi()) return end_(t);
    }
  } while (!t->get(i, j));
  return pack_(i, j);
}


Wrapper::const_iterator Wrapper::Begin(int layer) const
{
  const Tiles *t = GetMap()->GetLayer(layer);
  const_iterator id = begin_(t);
  return id;
}


Wrapper::const_iterator Wrapper::End(int layer) const
{
  const Tiles *t = GetMap()->GetLayer(layer);
  return end_(t);
}


Wrapper::const_iterator Wrapper::Next(int layer, const_iterator id) const
{
  const Tiles *t = GetMap()->GetLayer(layer);
  return next_(t, id);
}



Wrapper::iterator Wrapper::Begin(int layer)
{
  Tiles *t = GetMap()->GetLayer(layer);
  const_iterator id = begin_(t);
  return id;
}


Wrapper::iterator Wrapper::End(int layer)
{
  Tiles *t = GetMap()->GetLayer(layer);
  return end_(t);
}


Wrapper::iterator Wrapper::Next(int layer, iterator id)
{
  Tiles *t = GetMap()->GetLayer(layer);
  return next_(t, id);
}


int Wrapper::GetTileI(int layer, const_iterator id) const
{
  return unpacki_(id);
}


int Wrapper::GetTileJ(int layer, const_iterator id) const
{
  return unpackj_(id);
}


const BaseProperty *Wrapper::RawGetTileProperty(int layer, const_iterator id) const
{
  // tests...
  return GetLayer(layer)->get(unpacki_(id), unpackj_(id));
}


BaseProperty *Wrapper::RawGetTileProperty(int layer, iterator id)
{
  // tests...
  return GetLayer(layer)->get(unpacki_(id), unpackj_(id));
}


//const Map *Wrapper::GetMap(void) const
//{
  //return map_;
//}
//
//
//Map *Wrapper::GetMap(void)
//{
  //return map_;
//}


int Wrapper::FindLayerIndex(const char *name) const
{
  return GetMap()->FindLayerIndex(name);
}


int Wrapper::GetNumProperties(void) const
{
  return GetMap()->GetNumProperties();
}


int Wrapper::GetNumLayers(void) const
{
  return GetMap()->GetNumLayers();
}


const Tiles *Wrapper::GetLayer(int layer) const
{
  return GetMap()->GetLayer(layer);
}


Tiles *Wrapper::GetLayer(int layer)
{
  return GetMap()->GetLayer(layer);
}


const BaseCreator *Wrapper::RawGetCreator(int index) const
{
  return GetLayer(index)->creator();
}


const char *Wrapper::GetLayerName(int l) const
{
  return GetMap()->GetLayerName(l);
}
const BaseProperty *Wrapper::GetLayerReference(int l) const
{
  return GetMap()->GetLayerReference(l);
}
int Wrapper::GetActiveLayerIndex() const
{
  return GetMap()->GetActiveLayerIndex();
}


void Wrapper::RawSetProperty(int index, BaseCreator *creator)
{
  TEST_MAP_PROP_INDEX(index, UString("RawSetProperty"), HackStrProperty(index));
  USER_CHECK(
    if(GetMap()->GetProperties()->Get(index)) {
      popup_quit(
        UString("RawSetProperties"), Translation("Property is already set"),
        HackStrProperty(index)
      );
    }
  );
  GetMap()->GetProperties()->Add(index, creator);
}


void Wrapper::RawSetLayer(
  int index, LAYER_TYPE full_or_sparse, BaseCreator *creator, int x, int y, int w, int h
)
{
  TEST_LAYER_INDEX_POSITIVE(index, UString("RawSetLayer"), HackStrLayer(index));
  USER_CHECK(
    if(GetMap()->GetLayer(index)) {
      popup_quit(
        UString("RawSetLayer"), Translation("Layer is already set"),
        UString(GetMap()->GetLayerName(index))
      );
    }
  );
  Tiles *layer = 0;
  switch(full_or_sparse) {
    case FULL_LAYER:
      layer = new FullTiles(creator, x, y, w, h);
    break;
    case SPARSE_LAYER:
      layer = new SparseTiles(creator, x, y, w, h);
    break;
    default:
      popup_quit(
        UString("RawSetLayer"), Translation("Unknown layer type"),
        UString(GetMap()->GetLayerName(index))
      );
    break;
  }
  if(layer) GetMap()->SetLayer(index, layer);
  SetLayerFullness(full_or_sparse, index);
}


void Wrapper::FillLayer(int index)
{
  if (layers_fullness_[index] == FULL_LAYER) {
    TEST_LAYER_INDEX(index, UString("FillLayer"), HackStrLayer(index));
    Tiles *layer = GetMap()->GetLayer(index);
    int width = GetMapWidth();
    int height = GetMapHeight();
    const BaseCreator *creator = layer->creator();
    for(int j=0; j<height; j++) for(int i=0; i<width; i++) {
      USER_ASSERT(!RawGetTileProperty(index, i, j));
      BaseProperty *p = creator->Create();
      CHECK_POINTER(p);
      DBG_ASSERT(!layer->get(i, j));
      layer->set(i, j, p);
    }
  }
}


void Wrapper::RawSetTileProperty(int index, int x, int y, BaseProperty *prop)
{
  TEST_LAYER_INDEX(index, UString("RawSetTileProperty"), HackStrLayer(index));
  Tiles *layer = GetMap()->GetLayer(index);
  USER_CHECK(
    if(!layer) {
      popup_quit(
        UString("RawSetTileProperty"),
        Translation("Layer is not set"), HackStrLayer(index)
      );
    }
  );
  USER_CHECK(
    if(layer->get(x, y)) {
      popup_quit(
        UString("RawSetTileProperty"), Translation("Tile is already set"),
        HackStrLayerAt(index, x, y)
      );
    }
  );
  //DBG_ASSERT(!layer->get(x, y));
  layer->set(x, y, prop);
}


const BaseProperty *Wrapper::RawGetProperty(int index) const
{
  TEST_MAP_PROP_INDEX(index, UString("RawGetProperty"), HackStrProperty(index));
  const Composite::Creator *props = GetMap()->GetProperties();
  const BaseCreator *creator = props->Get(index);
  DBG_ASSERT(creator);
  return creator->GetReference();
}


const BaseProperty *Wrapper::RawGetTileProperty(int index, int x, int y) const
{
  TEST_LAYER_INDEX(
    index, UString("RawGetTileProperty"), HackStrLayerAt(index, x, y)
  );
  const Tiles *layer = GetMap()->GetLayer(index);
  DBG_ASSERT(layer);
  return layer->get(x, y);
}


BaseProperty *Wrapper::RawGetTileProperty(int index, int x, int y)
{
  TEST_LAYER_INDEX(
    index, UString("RawGetTileProperty"), HackStrLayerAt(index, x, y)
  );
  Tiles *layer = GetMap()->GetLayer(index);
  DBG_ASSERT(layer);
  return layer->get(x, y);
}


const class ViewedLayers &Wrapper::GetViewedLayers() const
{
  return GetMap()->GetViewedLayers();
}

const class SelectedTiles *Wrapper::GetSelectedTiles() const
{
  return GetMap()->GetSelectedTiles();
}

void Wrapper::Execute(Command *cmd)
{
  GetMap()->GetCommandsManager()->Execute(cmd);
}

void Wrapper::ClearUndoStack()
{
  GetMap()->GetCommandsManager()->Clear();
}

const class TilesStack *Wrapper::GetLayers() const
{
  return GetMap()->GetLayers();
}
class TilesStack *Wrapper::GetLayers()
{
  return GetMap()->GetLayers();
}

