/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "propdlg.h"

#include "debug.h"
#include "utils.h"
#include "help.h"
#include "map.h"
#include "globals.h"
#include "gui.h"
#include "entry.h"

#include <altheme.h>

#include <allegro.h>


static int prop_box_proc(int msg, DIALOG *d, int c)
{
  msg = do_help(msg, "Props");
  return altheme_box_proc(msg, d, c);
}


int PropertyDialog::PropListProc(int msg, DIALOG *d, int c)
{
  PropertyDialog *propdlg = CAST_FROM_VOID_PTR(PropertyDialog*, d->dp3);
  int old_d1 = d->d1;

  if(msg==MSG_IDLE) {
    GUI.UpdateActiveProperty();
    return D_O_K;
  }

  msg = do_help(msg, "Props");

  int ret = altheme_list_proc(msg, d, c);
  if (propdlg && propdlg->GetMap()) {
    if (old_d1 != d->d1) {
      GUI.SetActiveProperty(d->d1);
    }
  }
  return ret;
}


const char *PropertyDialog::PropListCb(int index, int *list_size, DIALOG *d)
{
  PropertyDialog *propdlg = CAST_FROM_VOID_PTR(PropertyDialog*, d->dp3);
  if (index>=0) {
    if (propdlg && propdlg->GetMap()) {
      return propdlg->GetMap()->GetPropertyName(index);
    }
    else {
      return EmptyString();
    }
  }
  else {
    if (propdlg && propdlg->GetMap()) {
      *list_size = propdlg->GetMap()->GetNumProperties();
    }
    else {
      *list_size = 0;
    }
    return NULL;
  }
}


PropertyDialog::PropertyDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d)
{
  Init(d);
}


PropertyDialog::PropertyDialog(DIALOG *d): Dialog(d)
{
  Init(d);
}


PropertyDialog::~PropertyDialog(void)
{
  delete map_entry;
}


void PropertyDialog::Init(DIALOG *d)
{
  box = &(d[PROPERTY_DIALOG_BOX]);
  box->proc = prop_box_proc;

  prop = &(d[PROPERTY_DIALOG_PROP]);
  prop->proc = PropListProc;
  prop->d1 = 0;
  prop->d2 = 0;
  prop->dp = CAST_TO_VOID_PTR(PropListCb);
  prop->dp3 = this;

  /* Entry should be the last one, because we don't want to know how many
     DIALOGs it uses */
  entry = &(d[PROPERTY_DIALOG_ENTRY]);
  map_entry = new Entry(entry, "Props");
  map_entry->ShowName(false);
  CHECK_POINTER(map_entry);
}


void PropertyDialog::InitMap(void)
{
  if(GetMap()) {
    prop->d1 = 0;
    prop->d2 = 0;
    GUI.SetActiveProperty(prop->d1);
  }
  else {
    map_entry->SetDefaultState();
    prop->d1 = 0;
    prop->d2 = 0;
  }
}


void PropertyDialog::LazyInit(void)
{
  prop->x = box->x + DIALOG_PADDING;
  prop->y = box->y + DIALOG_PADDING;
  prop->w = box->w - 2*DIALOG_PADDING;
  prop->h = SINGLE_HEIGHT * 2;

  map_entry->SetSize(
    box->x + DIALOG_PADDING,
    prop->y + prop->h + DIALOG_PADDING,
    box->w - 2*DIALOG_PADDING,
    box->y+box->h - (prop->y+prop->h) - 2*DIALOG_PADDING
  );
}


void PropertyDialog::SetActiveProperty(int new_prop)
{
  if (new_prop!=prop->d1) {
    prop->d1 = new_prop;
    prop->flags = D_DIRTY;
  }
}


Entry *PropertyDialog::GetEntry() const
{
  return map_entry;
}


const char *PropertyDialog::HelpText() const
{
  static const Translation i18nCurrentProp("Current property");
  static const Translation i18nPropList("Properties list");

  int mx = mouse_x;
  int my = mouse_y;
  if (mx>=prop->x && mx<prop->x+prop->w && my>=prop->y && my<prop->y+prop->h) {
    return i18nPropList;
  }
  return i18nCurrentProp;
}

