/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "menudlg.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "alclean.h"

#include "help.h"
#include "map.h"
#include "globals.h"
#include "selected.h"
#include "gui.h"
#include "brushlst.h"
#include "brush.h"
#include "commands.h"
#include "cmdselec.h"
#include "cmdmap.h"
#include "cmdgroup.h"
#include "cmdlayer.h"
#include "editmap.h"

#include "settheme.h"
#include "font.h"

#include <altheme.h>

#include <allegro.h>


typedef enum {
  FILE_MENU_IDX = 0,
  MAP_MENU_IDX,
  EDIT_MENU_IDX,
  SELECTION_MENU_IDX,
  BRUSHES_MENU_IDX,
  USER_MENU_IDX,
  OPTION_MENU_IDX,
  HELP_MENU_IDX,
  NUM_MENUS
} MENU_INDEX;


#include "plugin.h"

extern void save_config(void);


static void allow(bool yesno, MENU &m)
{
  if (yesno) m.flags &= ~D_DISABLED;
  else m.flags |= D_DISABLED;
}


/* File menu
   ---------------------------------*/
/* Returns true if has saved */
static bool save()
{
  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;
  ustrzcpy(path, sizeof(path), GUI.GetPath());

  bool saved = false;

  if(file_select(Translation("Save map file"), path, NULL) == TRUE) {
    if (!exists(path) || OVERWRITE_FILE(path)) {
      GUI.SetPath(path);
      int err = plugin_save(path, GUI.GetMap());
      if(err) {
        const char *err_str = get_error_string(err);
        ERROR_WRITING(path, err_str);
      }
      else {
        GUI.GetMap()->GetCommandsManager()->Save();
        GUI.RedrawMap();
        saved = true;
      }
    }
  }
  return saved;
}


/* Returns true if should proceed */
static bool save_map_changes()
{
  static const Translation str1("Warning !");
  static const Translation str2("Current map changes will be lost");
  static const Translation proceed_str("Proceed anyway");
  static const Translation cancel_str("Cancel");

  bool proceed = true;
  if (GUI.GetMap() && GUI.GetMap()->GetCommandsManager()->HasChanged()) {
    int c = alert(str1, 0, str2, proceed_str, cancel_str, 0, 0);
    if (c==2) proceed = false;
  }
  return proceed;
}


static int menu_exit(void)
{
  if (save_map_changes() && REALLY_QUIT()) {
    return D_CLOSE;
  }
  return D_O_K;
}


static int menu_new(void)
{
  if (save_map_changes()) {
    int err = plugin_new();
    if(err) {
      const char *err_str = get_error_string(err);
      ERROR_CREATING(err_str);
    }
    GUI.SetWindowTitle();
  }
  return D_REDRAW;
}


static int menu_load(void)
{
  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;
  if (save_map_changes()) {
    ustrzcpy(path, sizeof(path), GUI.GetPath());
    if(file_select(Translation("Load map file"), path, NULL) == TRUE) {
      GUI.SetPath(path);
      int err = plugin_load(path);
      if(err) {
        const char *err_str = get_error_string(err);
        ERROR_READING(path, err_str);
      }
    }
    GUI.SetWindowTitle();
  }
  return D_REDRAW;
}


static int menu_save(void)
{
  if (GUI.GetMap()) {
    save();
  }
  GUI.SetWindowTitle();
  return D_O_K;
}



/* Map menu
   ---------------------------------*/
static void get_selected_span(Map *map, int *x, int *y, int *w, int *h)
{
  int x1 = map->GetWidth()/2;
  int y1 = map->GetHeight()/2;
  int x2 = map->GetWidth()/2-1;
  int y2 = map->GetHeight()/2-1;
  SelectedTiles *s = map->GetSelectedTiles();
  if (s && !s->Empty()) {
    x1 = x2 = s->GetX(s->Begin());
    y1 = y2 = s->GetY(s->Begin());
    for (SelectedTiles::const_iterator id=s->Begin(); id!=s->End(); ++id) {
      int xx = s->GetX(id);
      int yy = s->GetY(id);
      if (xx<x1) x1=xx;
      if (yy<y1) y1=yy;
      if (xx>x2) x2=xx;
      if (yy>y2) y2=yy;
    }
  }
  *x = x1;
  *y = y1;
  *w = x2-x1+1;
  *h = y2-y1+1;
}


static int menu_resize_map(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    resize_all_layers(GUI.GetMap());
    ret = D_REDRAW;
  }
  return ret;
}


static int menu_resize_layer(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    resize_active_layer(GUI.GetMap());
    ret = D_REDRAW;
  }
  return ret;
}


static int menu_insert_in_map(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    int x, y, w, h;
    get_selected_span(GUI.GetMap(), &x, &y, &w, &h);
    insert_in_all_layers(GUI.GetMap(), x, y, 0, 0);
    ret = D_REDRAW;
  }
  return ret;
}


static int menu_delete_in_map(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    int x, y, w, h;
    get_selected_span(GUI.GetMap(), &x, &y, &w, &h);
    delete_in_all_layers(GUI.GetMap(), x, y, w, h);
    ret = D_REDRAW;
  }
  return ret;
}


static int menu_insert_in_layer(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    int x, y, w, h;
    get_selected_span(GUI.GetMap(), &x, &y, &w, &h);
    insert_in_active_layer(GUI.GetMap(), x, y, 0, 0);
    ret = D_REDRAW;
  }
  return ret;
}


static int menu_delete_in_layer(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    int x, y, w, h;
    get_selected_span(GUI.GetMap(), &x, &y, &w, &h);
    delete_in_active_layer(GUI.GetMap(), x, y, w, h);
    ret = D_REDRAW;
  }
  return ret;
}


#if 0
static int menu_merge_visible(void)
{
  int ret = D_O_K;
  if (GUI.GetMap()) {
    GUI.SetMouseBusy();
    Command *cmd = new CommandMergeLayers(GUI.GetMap(), GUI.GetMap()->GetViewedLayers());
    GUI.GetMap()->GetCommandsManager()->Execute(cmd);
    GUI.UnsetMouseBusy();
    ret = D_REDRAW;
  }
  return ret;
}
#endif



/* Edit menu
   ---------------------------------*/
static int menu_undo(void)
{
  GUI.Undo();
  return D_O_K;
}


static int menu_redo(void)
{
  GUI.Redo();
  return D_O_K;
}


static int menu_copy(void)
{
  GUI.Copy();
  return D_O_K;
}


static int menu_copy_b(void)
{
  GUI.CopyBrush();
  return D_O_K;
}


static int menu_cut(void)
{
  GUI.Cut();
  return D_O_K;
}



/* Select menu
   ---------------------------------*/
static int menu_selall(void)
{
  GUI.SelectAll();
  return D_O_K;
}


static int menu_selnone(void)
{
  GUI.SelectNone();
  return D_O_K;
}


static int menu_selinv(void)
{
  GUI.SelectInvert();
  return D_O_K;
}


static int menu_selgrow(void)
{
  GUI.SelectGrow();
  return D_O_K;
}


static int menu_selshrink(void)
{
  GUI.SelectShrink();
  return D_O_K;
}



/* Brushes menu
   ---------------------------------*/
static int menu_brushes(void)
{
  if(brush_list.Popup() && brush_list.GetActive()!=-1) {
    GUI.SetBrush(brush_list.GetBrush(brush_list.GetActive()));
  }
  return D_O_K;
}


static Brush *(*_menu_load_brush)(const char *) = 0; // FIXME
static int menu_load_brush(void)
{
  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;
  if (_menu_load_brush) {
    if(file_select(Translation("Load brush file"), path, NULL) == TRUE) {
      Brush *b = _menu_load_brush(path);
      if (b) {
        brush_list.AddBrush(b, path);
      }
    }
  }
  return D_O_K;
}


static bool (*_menu_save_brush)(const char *, const Brush *) = 0; // FIXME
static int menu_save_brush(void)
{
  static char path[STRING_PATH_BASE_LENGTH] = EMPTY_STRING;
  const Brush *b = GUI.GetBrush();
  if (b && _menu_save_brush) {
    if(file_select(Translation("Save brush file"), path, NULL) == TRUE) {
      if (!exists(path) || OVERWRITE_FILE(path)) {
        if (_menu_save_brush(path, b)) {
          brush_list.AddBrush(b->Clone(), path);
        }
      }
    }
  }
  return D_O_K;
}



/* User menu
   ---------------------------------*/
static int menu_user(void)
{
  int m = CAST_FROM_VOID_PTR(unsigned int, active_menu->dp);
  GUI.CallUserMenu(m);
  return D_REDRAW;
}



/* Options menu
   ---------------------------------*/
static int menu_grid(void)
{
  if(active_menu->flags&D_SELECTED) {
    active_menu->flags &= ~D_SELECTED;
    GUI.HideGrid();
  }
  else {
    active_menu->flags |= D_SELECTED;
    GUI.ShowGrid();
  }
  return D_O_K;
}


static int menu_undo_levels(void)
{
  char undo_levels[STRING_INT_LENGTH*SIZEOF_CHARACTER];
  uszprintf(
    undo_levels, sizeof(undo_levels),
    UString("%d"), Commands::GetMaxUndoLevels()
  );
  if(
    popup_edit(Translation("Max Undo Levels"), undo_levels, STRING_INT_LENGTH)
    == TRUE
  ) {
    int max = uatoi(undo_levels);
    if(max>0) {
      if(alert(
        Translation("Warning !"), EmptyString(),
        Translation("Current undo stack will be lost"),
        Translation("Proceed anyway"), Translation("Cancel"), 0, 0
      ) == 1) {
        Commands::ChangeMaxUndoLevels(max);
        GUI.GetMap()->GetCommandsManager()->ChangeMaxUndoLevels();
      }
    }
    else {
      ERROR_BAD_UNDO_LEVEL();
    }
  }
  return D_O_K;
}


static int menu_font(void)
{
  if(active_menu->flags&D_SELECTED) {
    active_menu->flags &= ~D_SELECTED;
    font = get_font(FONT_SIZE_SMALL);
  }
  else {
    active_menu->flags |= D_SELECTED;
    font = get_font(FONT_SIZE_BIG);
  }
  /* FIXME; This doen't clear the area under the menu because
   * Allegro redraw it */
  GUI.ClearMenu();
  return D_REDRAW;
}


static int menu_fgcolor(void)
{
  int r = getr(gui_fg_color);
  int g = getg(gui_fg_color);
  int b = getb(gui_fg_color);
  if(color_select(Translation("Foreground color"), &r, &g, &b) == TRUE) {
    GUI.SetTheme(0, makecol(r, g, b), gui_bg_color);
  }
  return D_REDRAW;
}


static int menu_bgcolor(void)
{
  int r = getr(gui_bg_color);
  int g = getg(gui_bg_color);
  int b = getb(gui_bg_color);
  if(color_select(Translation("Background color"), &r, &g, &b) == TRUE) {
    GUI.SetTheme(0, gui_fg_color, makecol(r, g, b));
  }
  return D_REDRAW;
}


static int menu_screensize(void)
{
  static const Translation i18nScreenSize("Screen size");
  static const Translation i18n640x480("640 x 480");
  static const Translation i18n800x600("800 x 600");
  static const Translation i18n1024x768("1024 x 768");
  static const Translation i18n1280x1024("1280 x 1024");
  static const Translation i18n1600x1200("1600 x 1200");
  static const struct {
    const char *name;
    int w; int h;
  } sizes[] = {
    { i18n640x480.string(), 640, 480 },
    { i18n800x600.string(), 800, 600 },
    { i18n1024x768.string(), 1024, 768 },
    { i18n1280x1024.string(), 1280, 1024 },
    { i18n1600x1200.string(), 1600, 1200 },
    { }
  };
  static int nsizes = 5;

  int old_w = SCREEN_W;
  int old_h = SCREEN_H;

  int sel = 0;
  for (int n=0; n<nsizes; ++n) {
    if (sizes[n].w == old_w && sizes[n].h == old_h) {
      sel = n;
    }
  }

  int ret = popup_radio(
    i18nScreenSize.string(), sel, nsizes,
    sizes[0].name, sizes[1].name,
    sizes[2].name, sizes[3].name,
    sizes[4].name
  );

  if (ret>=0) {
    /* Allegro seems to loose the gui_*_colors when changing gfx mode */
    int fg=gui_fg_color;
    int bg=gui_bg_color;
    show_mouse(NULL);
    int err = set_gfx_mode(GFX_AUTODETECT, sizes[ret].w, sizes[ret].h, 0, 0);
    if (err<0) {
      set_gfx_mode(GFX_AUTODETECT, old_w, old_h, 0, 0);
    }
    clear_to_color(screen, gui_mg_color);
    GUI.SetSize(SCREEN_W, SCREEN_H);
    GUI.SetTheme(0, fg, bg);
    show_mouse(screen);
  }

  return D_REDRAW;
}


static int menu_saveconf(void)
{
  save_config();
  return D_O_K;
}



/* Help menu
   ---------------------------------*/
int popup_help(void)
{
  popup_help("Main");
  return D_O_K;
}


static int menu_about(void)
{
  popup_printf(
    UString("eme").string(),
    //UString("  eme, version %d.%d\n  Copyright %s\n\n%s\n\n%s").string(),
    //EME_VERSION_MAJOR, EME_VERSION_MINOR,
    UString("  eme, version %s\n  Copyright %s\n\n%s\n\n%s").string(),
    UString(EME_VERSION_STR).string(),
    UString(EME_COPYRIGHT).string(), Translation(EME_INFO).string(), plugin_about()
  );
  return D_O_K;
}



/* Menus array
   ---------------------------------*/
static MENU *file_menu(void)
{
  static const Translation i18nLoad("&Load");
  static const Translation i18nSave("&Save");
  static const Translation i18nNew("&New");
  static const Translation i18nQuit("&Quit");

  static MENU file_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nLoad.char_ptr(), menu_load },
    { i18nSave.char_ptr(), menu_save },
    { i18nNew.char_ptr(),  menu_new },
    { EMPTY_STRING },
    { i18nQuit.char_ptr(), menu_exit },
    {}
  };
  return file_menu;
}


static MENU *map_menu(bool yesno=0, unsigned int features=0)
{
  using namespace Features;

  static const Translation i18nResizeMap("Resize all layers");
  static const Translation i18nInsertInMap("Insert in all layers");
  static const Translation i18nDeleteInMap("Delete in all layers");
  static const Translation i18nResizeLayer("Resize active layer");
  static const Translation i18nInsertInLayer("Insert in active layer");
  static const Translation i18nDeleteInLayer("Delete in active layer");
#if 0
  static const Translation i18nMerge("Merge visible layers");
#endif

  static MENU map_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nResizeMap.char_ptr(),     menu_resize_map },
    { i18nInsertInMap.char_ptr(),   menu_insert_in_map },
    { i18nDeleteInMap.char_ptr(),   menu_delete_in_map },
    { EMPTY_STRING },
    { i18nResizeLayer.char_ptr(),   menu_resize_layer },
    { i18nInsertInLayer.char_ptr(), menu_insert_in_layer },
    { i18nDeleteInLayer.char_ptr(), menu_delete_in_layer },
#if 0
    { EMPTY_STRING },
    { i18nMerge.char_ptr(),         menu_merge_visible },
#endif
    {}
  };

  if (features) {
    if (features&MapMenu::ResizeAll) allow(yesno, map_menu[0]);
    if (features&MapMenu::InsertInAll) allow(yesno, map_menu[1]);
    if (features&MapMenu::DeleteInAll) allow(yesno, map_menu[2]);
    if (features&MapMenu::ResizeActive) allow(yesno, map_menu[4]);
    if (features&MapMenu::InsertInActive) allow(yesno, map_menu[5]);
    if (features&MapMenu::DeleteInActive) allow(yesno, map_menu[6]);
#if 0
    if (features&MapMenu::MergeVisible) allow(yesno, map_menu[8]);
#endif
  }

  return map_menu;
}


static MENU *edit_menu(bool yesno=0, unsigned int features=0)
{
  using namespace Features;

  static const Translation i18nUndo("Undo (Ctrl-Z)");
  static const Translation i18nRedo("Redo (Ctrl-R)");
  static const Translation i18nCopy("Copy (Ctrl-C)");
  static const Translation i18nCopy_Brush("Copy Brush (Ctrl-B)");
  static const Translation i18nCut("Cut (Ctrl-X)");

  static MENU edit_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nUndo.char_ptr(),       menu_undo },
    { i18nRedo.char_ptr(),       menu_redo },
    { EMPTY_STRING },
    { i18nCopy.char_ptr(),       menu_copy },
    { i18nCopy_Brush.char_ptr(), menu_copy_b },
    { i18nCut.char_ptr(),        menu_cut },
    {}
  };

  if (features) {
    if (features&EditMenu::Undo) allow(yesno, edit_menu[0]);
    if (features&EditMenu::Redo) allow(yesno, edit_menu[1]);
    if (features&EditMenu::Copy) allow(yesno, edit_menu[3]);
    if (features&EditMenu::CopyBrush) allow(yesno, edit_menu[4]);
    if (features&EditMenu::Cut) allow(yesno, edit_menu[5]);
  }

  return edit_menu;
}


static MENU *selection_menu(bool yesno=0, unsigned int features=0)
{
  using namespace Features;

  static const Translation i18nAll("All (Ctrl-A)");
  static const Translation i18nNone("None (Ctrl-N)");
  static const Translation i18nInvert("Invert (Ctrl-I)");
  static const Translation i18nGrow("Grow");
  static const Translation i18nShrink("Shrink");

  static MENU selection_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nAll.char_ptr(),    menu_selall },
    { i18nNone.char_ptr(),   menu_selnone },
    { i18nInvert.char_ptr(), menu_selinv },
    { EMPTY_STRING },
    { i18nGrow.char_ptr(),   menu_selgrow },
    { i18nShrink.char_ptr(), menu_selshrink },
    {}
  };

  if (features) {
    if (features&SelectMenu::All) allow(yesno, selection_menu[0]);
    if (features&SelectMenu::None) allow(yesno, selection_menu[1]);
    if (features&SelectMenu::Invert) allow(yesno, selection_menu[2]);
    if (features&SelectMenu::Grow) allow(yesno, selection_menu[4]);
    if (features&SelectMenu::Shrink) allow(yesno, selection_menu[5]);
  }

  return selection_menu;
}


static MENU *brushes_menu(bool yesno=0, unsigned int features=0)
{
  using namespace Features;

  static const Translation i18nLoad("&Load");
  static const Translation i18nSave("&Save");
  static const Translation i18nChoose("&Choose...");

  static MENU brushes_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nLoad.char_ptr(),   menu_load_brush },
    { i18nSave.char_ptr(),   menu_save_brush },
    { EMPTY_STRING} ,
    { i18nChoose.char_ptr(), menu_brushes },
    {}
  };

  if (features) {
    if (features&BrushMenu::Load) allow(yesno, brushes_menu[0]);
    if (features&BrushMenu::Save) allow(yesno, brushes_menu[1]);
    if (features&BrushMenu::Choose) allow(yesno, brushes_menu[2]);
  }

  return brushes_menu;
}


static MENU *option_menu(void)
{
  static const Translation i18nGrid("&Grid");
  static const Translation i18nUndo_Levels("&Undo Levels");
  static const Translation i18nThemes("&Themes");
  static const Translation i18nFont("&Big Font");
  static const Translation i18nForeground("&Foreground");
  static const Translation i18nBackground("&Background");
  static const Translation i18nScreenSize("&Screen Size");
  static const Translation i18nSave_Config("Save Config");

  static MENU option_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nGrid.char_ptr(),        menu_grid },
    { i18nUndo_Levels.char_ptr(), menu_undo_levels },
    { i18nThemes.char_ptr(),      0, theme_menu() },
    { i18nFont.char_ptr(),        menu_font },
    { i18nForeground.char_ptr(),  menu_fgcolor },
    { i18nBackground.char_ptr(),  menu_bgcolor },
    { i18nScreenSize.char_ptr(),  menu_screensize },
    { EMPTY_STRING },
    { i18nSave_Config.char_ptr(), menu_saveconf },
    {}
  };

  if (font_is_small(font)) {
    option_menu[3].flags &= ~D_SELECTED;
  }
  else {
    option_menu[3].flags |= D_SELECTED;
  }

  return option_menu;
}
#define OPTION_MENU_GRID 0


static MENU *help_menu(void)
{
  static const Translation i18nHelp("Help (F1)");
  static const Translation i18nAbout("About");

  static MENU help_menu[] = {
  /* char *text, int (*proc)(), child, flags, dp */
    { i18nHelp.char_ptr() ,    popup_help },
    { i18nAbout.char_ptr(),    menu_about },
    {}
  };

  return help_menu;
}


static MENU user_menu[NUM_USER_MENUS+1] = {
/* char *text, proc(), child, flags, dp */
  {}
};



/* Menu class
   ---------------------------------*/
MenuDialog::MenuDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d)
{
  Init(d);
}


MenuDialog::MenuDialog(DIALOG *d): Dialog(d)
{
  Init(d);
}


MenuDialog::~MenuDialog(void)
{
  free(menus[USER_MENU_IDX].text);
  for(int i=0; i<NUM_USER_MENUS; i++) {
    SetUserMenu(i, 0, 0);
  }
  Free(menus);
}


void MenuDialog::Init(DIALOG *d)
{
  dialog = d;

  Calloc(menus, NUM_MENUS+1);

  dialog->proc = altheme_menu_proc;
  dialog->dp = menus;

  for(unsigned int i=0; i<NUM_USER_MENUS; i++) {
    user_menu[i].text=0;
    user_menu[i].proc=menu_user;
    user_menu[i].child=0;
    user_menu[i].flags=D_DISABLED;
    user_menu[i].dp=CAST_TO_VOID_PTR(i);
    user_cb[i]=0;
    user_hotkeys[i] = 0;
  }
}


void MenuDialog::LazyInit(void)
{
  static const Translation i18nMenuFile("&File");
  static const Translation i18nMenuMap("&Map");
  static const Translation i18nMenuEdit("&Edit");
  static const Translation i18nMenuSelect("&Select");
  static const Translation i18nMenuBrushes("&Brushes");
  static const Translation i18nMenuOptions("&Options");
  static const Translation i18nMenuUser("&User");
  static const Translation i18nMenuHelp("&Help");

  menus[FILE_MENU_IDX].text = i18nMenuFile.char_ptr();
  menus[FILE_MENU_IDX].child = file_menu();

  menus[MAP_MENU_IDX].text = i18nMenuMap.char_ptr();
  menus[MAP_MENU_IDX].child = map_menu();

  menus[EDIT_MENU_IDX].text = i18nMenuEdit.char_ptr();
  menus[EDIT_MENU_IDX].child = edit_menu();

  menus[SELECTION_MENU_IDX].text = i18nMenuSelect.char_ptr();
  menus[SELECTION_MENU_IDX].child = selection_menu();

  menus[BRUSHES_MENU_IDX].text = i18nMenuBrushes.char_ptr();
  menus[BRUSHES_MENU_IDX].child = brushes_menu();

  menus[OPTION_MENU_IDX].text = i18nMenuOptions.char_ptr();
  menus[OPTION_MENU_IDX].child = option_menu();

  if (!menus[USER_MENU_IDX].text) {
    menus[USER_MENU_IDX].text = ustrdup(i18nMenuUser);
  }
  menus[USER_MENU_IDX].child = user_menu;

  menus[HELP_MENU_IDX].text = i18nMenuHelp.char_ptr();
  menus[HELP_MENU_IDX].child = help_menu();
}


void MenuDialog::DrawGrid(int yesno)
{
  if(yesno) {
    option_menu()[OPTION_MENU_GRID].flags |= D_SELECTED;
  }
  else {
    option_menu()[OPTION_MENU_GRID].flags &= ~D_SELECTED;
  }
}


void MenuDialog::SetUserMenuTitle(const char *title)
{
  free(menus[USER_MENU_IDX].text);
  menus[USER_MENU_IDX].text = ustrdup(title);
}


void MenuDialog::SetUserMenu(
  int menu_index, const char *name, void (*cb)(Map*), int hotkey
)
{
  DBG_ASSERT(menu_index>=0 && menu_index<NUM_USER_MENUS);
  if(user_menu[menu_index].text) {
    free(user_menu[menu_index].text);
    user_menu[menu_index].text = 0;
  }
  if(name) {
    user_menu[menu_index].text = ustrdup(name);
    DBG_ASSERT(user_menu[menu_index].text);
  }
  allow(!!name, user_menu[menu_index]);
  user_cb[menu_index] = cb;
  user_hotkeys[menu_index] = hotkey;
}


void MenuDialog::CallUserMenu(int menu_index)
{
  DBG_ASSERT(menu_index>=0 && menu_index<NUM_USER_MENUS);
  if(user_cb[menu_index]) {
    GUI.UpdateActiveLayerAndProperty();
    user_cb[menu_index](GetMap());
  }
}


bool MenuDialog::HotKey(int key)
{
  for (int n=0; n<NUM_USER_MENUS; ++n) {
    if (key && key==user_hotkeys[n]) {
      CallUserMenu(n);
      return true;
    }
  }
  return false;
}


void MenuDialog::SetLoadBrush(Brush *(*cb)(const char *))
{
  _menu_load_brush = cb;
}


void MenuDialog::SetSaveBrush(bool (*cb)(const char *, const Brush *b))
{
  _menu_save_brush = cb;
}


void MenuDialog::AllowMap(bool yesno, unsigned int features)
{
  if (features==Features::MapMenu::Everything) {
    allow(yesno, menus[MAP_MENU_IDX]);
  }
  else {
    map_menu(yesno, features);
  }
}


void MenuDialog::AllowEdit(bool yesno, unsigned int features)
{
  if (features==Features::EditMenu::Everything) {
    allow(yesno, menus[EDIT_MENU_IDX]);
  }
  else {
    edit_menu(yesno, features);
  }
}


void MenuDialog::AllowSelect(bool yesno, unsigned int features)
{
  if (features==Features::SelectMenu::Everything) {
    allow(yesno, menus[SELECTION_MENU_IDX]);
  }
  else {
    selection_menu(yesno, features);
  }
}


void MenuDialog::AllowBrush(bool yesno, unsigned int features)
{
  if (features==Features::BrushMenu::Everything) {
    allow(yesno, menus[BRUSHES_MENU_IDX]);
  }
  else {
    brushes_menu(yesno, features);
  }
}


void MenuDialog::AllowUser(bool yesno, unsigned int features)
{
  if (features==Features::MapMenu::Everything) {
    allow(yesno, menus[USER_MENU_IDX]);
  }
}

