/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__LAYERS__
#define EME__LAYERS__
/*----------------------------------------------------------------------------
  Layer utilities

- Drawing
- Selections
- Copy/cut/paste
- Insert/delete row/column
 ----------------------------------------------------------------------------*/

struct BITMAP;
class BaseProperty;
class Entry;
class ViewedLayers;
class SelectedTiles;
class TilesStack;
class Tiles;
class SparseTiles;

namespace LayerUtils {

/* Drawing
   ---------------------------------*/
/*
 * Displays the layer on a bitmap, as an overall rectangle
 */
void DrawRect(
  const Tiles *tiles,       /* Tiles to display */
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale,
  int l                     /* Layer index */
);

/*
 * Displays the layer on a bitmap, as an overall diamond
 */
void DrawDiamond(
  const Tiles *tiles,       /* Tiles to display */
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale,
  int l                     /* Layer index */
);


/*
 * Displays the viewed layers of the layer stack on a bitmap,
 * as an overall rectangle
 */
void DrawRect(
  const TilesStack *layers,
  const ViewedLayers *viewed_layers,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale
);

/*
 * Displays the viewed layers of the layer stack on a bitmap,
 * as an overall diamond
 */
void DrawDiamond(
  const TilesStack *layers,
  const ViewedLayers *viewed_layers,
  BITMAP *bmp,              /* Bitmap where to draw */
  int first_i, int first_j, /* First tile to display indices */
  int num_i, int num_j,     /* Number of tiles to display */
  int tile_w, int tile_h,   /* Tile size in pixels */
  float tile_odd,           /* Odd lines indentation (tile width factor) */
  float scale
);



/* Selections
 *
 * Note that all selection function returns a newly created SelectedTiles object.
 * With the exeception of 'SelectEmpty' they select only *existing* tiles.
   ---------------------------------*/
/*
 * Select all the existing tiles, returning a new SelectedTiles object
 */
SelectedTiles *SelectAll(const Tiles *tiles);

/*
 * Returns a SelectedTiles object containing all of the tiles existing in 'sel'
 * but not in 'tiles'
 */
SelectedTiles *SelectEmpty(
  const Tiles *tiles,
  const SelectedTiles *sel
);

/*
 * Returns a SelectedTiles object containing all the tiles existing in 'tiles'
 * and that are in the rectangle defined by (x1, y1) (x2, y2).
 */
SelectedTiles *SelectRect(
  const Tiles *tiles,
  int i1, int j1,           /* Top left corner indices (inclusive) */
  int i2, int j2            /* Bottom right corner (inclusive) */
);

/*
 * Returns a SelectedTiles object containing all the tiles existing in 'tiles'
 * and that are in the circle with center (x, y) and radius square_root(r2)
 */
SelectedTiles *SelectCircle(
  const Tiles *tiles,
  int i, int j,             /* Circle center indices */
  int r                     /* Circle radius (in tiles, inclusive) */
);

/*
 * Returns a SelectedTiles object containing all the tiles in 'tiles', which
 * value is equal to the value at (x, y)
 */
SelectedTiles *SelectByProperty(
  const Tiles *tiles,
  int i, int j              /* Reference tile indices */
);

/*
 * Returns a SelectedTiles object containing all the tiles in 'tiles', which
 * value is equal to the value at (x, y) and which are reachable from (x, y)
 * with a flood fill
 */
SelectedTiles *SelectByWand(
  const Tiles *tiles,
  int i, int j    /* Reference tile indices */
);

/*
 * Returns a SelectedTiles object containing the intersection of s1 and s2
 */
SelectedTiles *Intersection(
  const SelectedTiles *s1, const SelectedTiles *s2
);

/*
 * Returns a SelectedTiles object containing all tiles in sel plus a border
 */
SelectedTiles *GrowSelection(const SelectedTiles *sel);

/*
 * Returns a SelectedTiles object containing all tiles in sel minus the border
 */
SelectedTiles *ShrinkSelection(const SelectedTiles *sel, const Tiles *tiles);

/*
 * Returns a new SelectedTiles object containing the opposite of sel
 */
SelectedTiles *InvertSelection(const SelectedTiles *sel, const Tiles *tiles);



/* Copy/cut/paste
   ---------------------------------*/
/*
 * Returns a newly created SparseTiles containing all the tiles that are both
 * in 'sel' and 'tiles'
 */
SparseTiles *Copy(
  const Tiles *tiles,
  const SelectedTiles *sel      /* Which tiles to copy */
);

/*
 * Returns a newly created SparseTiles containing all the tiles that are both
 * in 'sel' and 'layers', if 'viewed' is true for this layer.
 */
TilesStack *Copy(
  const TilesStack *layers,
  const ViewedLayers *viewed,
  const SelectedTiles *sel
);

/*
 * Fills all the tiles in 'tiles' specified by 'sel', with the property 'value'
 */
void Fill(
  Tiles *tiles,
  const SelectedTiles *sel,     /* Which tiles to fill */
  const BaseProperty *value     /* Which value to paste */
);

/*
 * Sets the tile at (x, y) in 'tiles' to the property 'value'
 */
void Fill(
  Tiles *tiles,
  int i, int j,              /* Which tile to fill */
  const BaseProperty *value  /* Which value to paste */
);

/*
 * Remove from 'tiles' the tiles specified in 'sel'. These tiles are either
 * replaced by a default value or simply destroyed, depending on 'tiles' type
 * (i.e. FullTiles or SparseTiles)
 */
void Cut(
  Tiles *tiles,
  const SelectedTiles *sel      /* Which tiles to cut */
);

void Paste(
  Tiles *tiles,
  int i, int j,             /* Where to paste (top left corner, inclusive) */
  const SparseTiles *to_paste    /* Which tiles to paste */
);

void Paste(
  TilesStack *layers,
  int i, int j,
  const TilesStack *to_paste
);

void Paste(
  Tiles *tiles,
  int i, int j,             /* Where to paste */
  const Entry *e            /* Which value to paste */
);



/* Insert/delete row/column
   ---------------------------------*/
void Move(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int di, int dj
);

void InsertColumns(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
);

void InsertRows(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
);

void DeleteColumns(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
);

void DeleteRows(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
);

#ifdef EME__COMPATIBILITY
#if 1
inline void InsertColumn(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
) { InsertColumns(layers, viewed, i, count); }

inline void InsertRow(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
) { InsertRows(layers, viewed, j, count); }

inline void DeleteColumn(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int i, int count
) { DeleteColumns(layers, viewed, i, count); }

inline void DeleteRow(
  TilesStack *layers,
  const ViewedLayers *viewed,
  int j, int count
) { DeleteRows(layers, viewed, j, count); }
#else
#define InsertColumn InsertColumns
#define InsertRow InsertRows
#define DeleteColumn DeleteColumns
#define DeleteRow DeleteRows
#endif
#endif

} /* namespace LayerUtils */

#endif /* EME__LAYERS__ */

