/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "layerdlg.h"

#include "debug.h"
#include "utils.h"

#include "help.h"
#include "map.h"
#include "globals.h"
#include "mapdlg.h"
#include "gui.h"
#include "lviewed.h"
#include "icon.h"

#include <altheme.h>
#include <altheme/internal/theme.h>

#include <allegro.h>


static void layer_up_cb(void)
{
  GUI.LayerUp();
}

static void layer_down_cb(void)
{
  GUI.LayerDown();
}

static void layer_del_cb()
{
  GUI.LayerDel();
}

static void layer_ins_cb()
{
  GUI.LayerIns();
}


static int layer_button_proc(int msg, DIALOG *d, int c)
{
  msg = do_help(msg, "Layer");

  if (msg==MSG_DRAW) {
    altheme_draw_shadow_box(
      screen, d->x, d->y, d->x+d->w-1, d->y+d->h-1,
      d->flags&D_SELECTED, d->flags&D_GOTFOCUS
    );
    if (d->dp) {
      BITMAP *icon = (BITMAP*)d->dp;
      draw_sprite(screen, icon, d->x+(d->w-icon->w)/2, d->y+(d->h-icon->h)/2);
    }
    return D_O_K;
  }
  else if (msg==MSG_CLICK || msg==MSG_KEY) {
    d->flags |= D_SELECTED;
    scare_mouse_area(d->x, d->y, d->w, d->h);
    altheme_button_proc(MSG_DRAW, d, c);
    unscare_mouse();
    while (mouse_b) {
    }
    d->flags &= ~D_SELECTED;
    if (d->dp3 && !(d->flags&D_DISABLED)) {
      void (*cb)(void) = (void (*)(void))(d->dp3);
      cb();
    }
    return D_REDRAWME;
  }
  else {
    return altheme_button_proc(msg, d, c);
  }
}


int LayerDialog::LayerListProc(int msg, DIALOG *d, int c)
{
  LayerDialog *layerdlg = CAST_FROM_VOID_PTR(LayerDialog*, d->dp3);

  if (msg==MSG_IDLE && layerdlg->GetMap()) {
    GUI.UpdateActiveLayer();
    if (layerdlg->GetMap()->GetActiveLayerIndex() != d->d1) {
      layerdlg->SetActiveLayer(layerdlg->GetMap()->GetActiveLayerIndex());
      layerdlg->SetViewedLayers();
      return D_REDRAWME;
    }
    return D_O_K;
  }

  msg = do_help(msg, "Layer");

  int ret = altheme_list_proc(msg, d, c);
  if (msg==MSG_CLICK || msg==MSG_KEY || msg==MSG_CHAR || msg==MSG_UCHAR) {
    if (layerdlg && layerdlg->GetMap()) {
      char *sel = CAST_FROM_VOID_PTR(char*, d->dp2);
      sel[d->d1] = 1;
      layerdlg->GetMap()->SetViewedLayers(layerdlg->GetViewedLayers());
      layerdlg->GetMap()->SetActiveLayer(d->d1);
      GUI.RedrawMap();
    }
  }
  return ret;
}


const char *LayerDialog::LayerNameCb(int index, int *list_size, DIALOG *d)
{
  const LayerDialog *layerdlg = CAST_FROM_VOID_PTR(const LayerDialog*, d->dp3);

  if (index>=0) {
    if (layerdlg && layerdlg->GetMap()) {
      return layerdlg->GetMap()->GetLayerName(index);
    }
    else {
      return EmptyString();
    }
  }
  else {
    if (layerdlg && layerdlg->GetMap()) {
      *list_size = layerdlg->GetMap()->GetNumLayers();
    }
    else {
      *list_size = 0;
    }
    return NULL;
  }
}


static void allow(bool yesno, DIALOG &d, BITMAP *(*icon_getter)(int))
{
  if (yesno) d.flags &= ~D_DISABLED;
  else d.flags |= D_DISABLED;
  if (icon_getter) d.dp = icon_getter(yesno);
}


LayerDialog::LayerDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d), viewed_layers_(0)
{
  Init(d);
}


LayerDialog::LayerDialog(DIALOG *d): Dialog(d), viewed_layers_(0)
{
  Init(d);
}


LayerDialog::~LayerDialog(void)
{
}


void LayerDialog::Init(DIALOG *d)
{
  box_ = &(d[0]);
  box_->proc = altheme_box_proc;

  list_ = &(d[1]);
  list_->proc = LayerListProc;
  list_->d1 = 0;
  list_->d2 = 0;
  list_->dp = CAST_TO_VOID_PTR(LayerNameCb);
  list_->dp3 = CAST_TO_VOID_PTR(this);
}


void LayerDialog::LazyInit()
{
  static const Translation i18nAdd("Add layer");
  static const Translation i18nDel("Remove layer");
  static const Translation i18nUp("Move layer up");
  static const Translation i18nDown("Move layer down");

  const int icon_height = 16;
  list_->x = box_->x+DIALOG_PADDING;
  list_->y = box_->y+DIALOG_PADDING;
  list_->w = box_->w-2*DIALOG_PADDING;
  list_->h = box_->h-3*DIALOG_PADDING - icon_height;

  int w = (box_->w-2*DIALOG_PADDING)/4;

  for (int n=BUTTON_ADD; n<BUTTON_COUNT; ++n) {
    box_[n].proc = layer_button_proc;
    box_[n].x = box_->x+DIALOG_PADDING + (n-BUTTON_ADD)*w;
    box_[n].y = box_->y+box_->h - DIALOG_PADDING - icon_height;
    box_[n].w = w;
    box_[n].h = icon_height;
  }

  if (!box_[BUTTON_ADD].dp2) {
    box_[BUTTON_ADD].dp2 = i18nAdd.void_ptr();
    box_[BUTTON_ADD].dp3 = CAST_TO_VOID_PTR(layer_ins_cb);

    box_[BUTTON_DEL].dp2 = i18nDel.void_ptr();
    box_[BUTTON_DEL].dp3 = CAST_TO_VOID_PTR(layer_del_cb);

    box_[BUTTON_UP].dp2 = i18nUp.void_ptr();
    box_[BUTTON_UP].dp3 = CAST_TO_VOID_PTR(layer_up_cb);

    box_[BUTTON_DOWN].dp2 = i18nDown.void_ptr();
    box_[BUTTON_DOWN].dp3 = CAST_TO_VOID_PTR(layer_down_cb);

    Allow(false, Features::LayersBox::Everything);
  }
}


void LayerDialog::Redraw(void) const
{
  list_->flags |= D_DIRTY;
}


void LayerDialog::InitMap(void)
{
  list_->dp2 = 0;
  if (viewed_layers_) delete viewed_layers_;
  viewed_layers_ = 0;
  if(GetMap()) {
    SetViewedLayers();
  }
}


void LayerDialog::SetActiveLayer(int n)
{
  if (n!=list_->d1) {
    list_->d1 = n;
    Redraw();
  }
}


void LayerDialog::SetViewedLayers()
{
  const ViewedLayers &v = GetMap()->GetViewedLayers();
  if (!viewed_layers_ || v.count()!=viewed_layers_->count()) {
    if (viewed_layers_) delete viewed_layers_;
    viewed_layers_ = new ViewedLayers(v);
    CHECK_POINTER(viewed_layers_);
    list_->dp2 = viewed_layers_->get();
    Redraw();
  }
}


const char *LayerDialog::HelpText() const
{
  static const Translation i18nHelp("Layers list");
  static const Translation i18nDisabled("Disabled");

  int mx = mouse_x;
  int my = mouse_y;

  for (int n=2; n<NUM_LAYER_DIALOGS; ++n) {
    DIALOG *b = &(box_[n]);
    if (mx>=b->x && mx<b->x+b->w && my>=b->y && my<b->y+b->h) {
      if (b->flags & D_DISABLED) {
        return i18nDisabled.string();
      }
      else {
        return (char *)(b->dp2);
      }
    }
  }

  return i18nHelp.string();
}


void LayerDialog::Allow(bool yesno, unsigned int feature)
{
  if (feature&Features::LayersBox::Add) {
    allow(yesno, box_[BUTTON_ADD], get_add_icon);
  }
  if (feature&Features::LayersBox::Del) {
    allow(yesno, box_[BUTTON_DEL], get_del_icon);
  }
  if (feature&Features::LayersBox::Swap) {
    allow(yesno, box_[BUTTON_UP], get_up_icon);
    allow(yesno, box_[BUTTON_DOWN], get_down_icon);
  }
}

